import React, { useState, useEffect } from 'react';
import { Link } from '@inertiajs/react';
import { DollarSign, Clock, Wallet } from 'lucide-react';
import axios from 'axios';

export default function WriterPaymentStatus() {
    const [paymentSummary, setPaymentSummary] = useState({
        pending_amount: 0,
        available_amount: 0,
        pending_orders_count: 0,
        available_orders_count: 0
    });
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);

    useEffect(() => {
        fetchPaymentSummary();
    }, []);

    const fetchPaymentSummary = async () => {
        try {
            const response = await axios.get('/writer/payment/summary');
            if (response.data.success) {
                setPaymentSummary(response.data.data);
            } else {
                setError('Failed to load payment data');
            }
        } catch (err) {
            setError('Error loading payment data');
            console.error('Payment summary fetch error:', err);
        } finally {
            setLoading(false);
        }
    };

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    if (loading) {
        return (
            <div className="flex items-center gap-2">
                <div className="animate-pulse flex items-center gap-2">
                    <div className="w-8 h-8 bg-gray-300 rounded-full"></div>
                    <div className="w-16 h-4 bg-gray-300 rounded"></div>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="flex items-center gap-2 text-red-500">
                <DollarSign size={20} />
                <span className="text-sm">Error</span>
            </div>
        );
    }

    return (
        <div className="flex items-center gap-1 sm:gap-3">
            {/* Pending Amount */}
            {paymentSummary.pending_amount > 0 && (
                <Link
                    href="/writer/payment/pending"
                    className="flex items-center gap-1 sm:gap-1.5 px-2 sm:px-3 py-1 sm:py-1.5 rounded-full bg-orange-100 text-orange-800 hover:bg-orange-200 transition-colors duration-200 group border border-orange-200"
                    title={`${paymentSummary.pending_orders_count} pending orders`}
                >
                    <Clock size={14} className="sm:hidden" />
                    <Clock size={16} className="hidden sm:block" />
                    <span className="text-xs sm:text-sm font-medium">
                        {formatCurrency(paymentSummary.pending_amount)}
                    </span>
                    <span className="hidden sm:inline text-xs opacity-75 group-hover:opacity-100">
                        Pending
                    </span>
                </Link>
            )}

            {/* Available Amount - Always show, display zero if no amount */}
            <Link
                href="/writer/payment/available"
                className={`flex items-center gap-1 sm:gap-1.5 px-2 sm:px-3 py-1 sm:py-1.5 rounded-full transition-colors duration-200 group border ${
                    paymentSummary.available_amount > 0 
                        ? 'bg-blue-100 text-blue-800 hover:bg-blue-200 border-blue-200' 
                        : 'bg-gray-100 text-gray-600 hover:bg-gray-200 border-gray-200'
                }`}
                title={`${paymentSummary.available_orders_count} orders ready for withdrawal`}
            >
                <Wallet size={14} className="sm:hidden" />
                <Wallet size={16} className="hidden sm:block" />
                <span className="text-xs sm:text-sm font-medium">
                    {formatCurrency(paymentSummary.available_amount)}
                </span>
                <span className="hidden sm:inline text-xs opacity-75 group-hover:opacity-100">
                    Available
                </span>
            </Link>

            {/* Show payment icon if no pending amounts - Only on larger screens */}
            {paymentSummary.pending_amount === 0 && (
                <Link
                    href="/writer/payment/dashboard"
                    className="hidden sm:flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-gray-100 text-gray-600 hover:bg-gray-200 transition-colors duration-200 group border border-gray-200"
                    title="View payment dashboard"
                >
                    <DollarSign size={16} />
                    <span className="text-xs opacity-75 group-hover:opacity-100">
                        Payments
                    </span>
                </Link>
            )}
        </div>
    );
}