import React, { useState } from 'react';
import { Upload, FileText, AlertCircle, CheckCircle, Clock } from 'lucide-react';
import axios from 'axios';

export default function WriterFileUpload({ orderId, onFileUploaded, onStatusUpdate }) {
    const [selectedFile, setSelectedFile] = useState(null);
    const [fileCategory, setFileCategory] = useState('draft');
    const [description, setDescription] = useState('');
    const [isUploading, setIsUploading] = useState(false);
    const [uploadStatus, setUploadStatus] = useState(null);
    const [error, setError] = useState('');

    const handleFileSelect = (event) => {
        const file = event.target.files[0];
        if (file) {
            // Validate file size (40MB limit)
            if (file.size > 40 * 1024 * 1024) {
                setError('File size must be less than 40MB');
                setSelectedFile(null);
                return;
            }
            
            setSelectedFile(file);
            setError('');
        }
    };

    const handleSubmit = async (event) => {
        event.preventDefault();
        
        if (!selectedFile) {
            setError('Please select a file to upload');
            return;
        }

        setIsUploading(true);
        setError('');
        setUploadStatus(null);

        const formData = new FormData();
        formData.append('file', selectedFile);
        formData.append('order_id', orderId);
        formData.append('file_category', fileCategory);
        formData.append('description', description);
        formData.append('uploader_user_type', 'writer');

        try {
            const response = await axios.post('/order-files/writer-upload', formData, {
                headers: {
                    'Content-Type': 'multipart/form-data',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content')
                }
            });

            if (response.data.success) {
                setUploadStatus('success');
                
                // Reset form
                setSelectedFile(null);
                setDescription('');
                setFileCategory('draft');
                
                // Clear file input
                const fileInput = document.getElementById('writer-file-input');
                if (fileInput) fileInput.value = '';

                // Notify parent component
                if (onFileUploaded) {
                    onFileUploaded(response.data.file);
                }

                // If order status was updated, notify parent
                if (response.data.order_status_updated && onStatusUpdate) {
                    onStatusUpdate('under_review');
                }

                // Auto-hide success message after 5 seconds
                setTimeout(() => {
                    setUploadStatus(null);
                }, 5000);
            }
        } catch (error) {
            console.error('Upload failed:', error);
            
            if (error.response?.data?.errors) {
                const errorMessages = Object.values(error.response.data.errors).flat();
                setError(errorMessages.join(', '));
            } else if (error.response?.data?.message) {
                setError(error.response.data.message);
            } else {
                setError('Upload failed. Please try again.');
            }
        } finally {
            setIsUploading(false);
        }
    };

    const getCategoryIcon = (category) => {
        switch (category) {
            case 'draft':
                return <Clock className="w-4 h-4" />;
            case 'complete':
                return <CheckCircle className="w-4 h-4" />;
            default:
                return <FileText className="w-4 h-4" />;
        }
    };

    const getCategoryDescription = (category) => {
        switch (category) {
            case 'draft':
                return 'Work-in-progress file (no timeline impact)';
            case 'complete':
                return 'Final deliverable (moves order to admin review)';
            default:
                return '';
        }
    };

    return (
        <div className="bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 p-6">
            <div className="flex items-center gap-2 mb-4">
                <Upload className="w-5 h-5 text-blue-600" />
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                    Upload Work File
                </h3>
            </div>

            <form onSubmit={handleSubmit} className="space-y-4">
                {/* File Category Selection */}
                <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        File Category *
                    </label>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                        {['draft', 'complete'].map((category) => (
                            <label
                                key={category}
                                className={`relative flex items-center p-3 border rounded-lg cursor-pointer transition-colors ${
                                    fileCategory === category
                                        ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20'
                                        : 'border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500'
                                }`}
                            >
                                <input
                                    type="radio"
                                    name="fileCategory"
                                    value={category}
                                    checked={fileCategory === category}
                                    onChange={(e) => setFileCategory(e.target.value)}
                                    className="sr-only"
                                />
                                <div className="flex items-center gap-2">
                                    {getCategoryIcon(category)}
                                    <span className="font-medium capitalize text-gray-900 dark:text-white">
                                        {category}
                                    </span>
                                </div>
                                {fileCategory === category && (
                                    <CheckCircle className="w-5 h-5 text-blue-600 absolute top-2 right-2" />
                                )}
                            </label>
                        ))}
                    </div>
                    <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                        {getCategoryDescription(fileCategory)}
                    </p>
                </div>

                {/* File Selection */}
                <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Select File *
                    </label>
                    <div className="flex items-center justify-center w-full">
                        <label className="flex flex-col items-center justify-center w-full h-32 border-2 border-gray-300 dark:border-gray-600 border-dashed rounded-lg cursor-pointer bg-gray-50 dark:bg-gray-700 hover:bg-gray-100 dark:hover:bg-gray-600 transition-colors">
                            <div className="flex flex-col items-center justify-center pt-5 pb-6">
                                <Upload className="w-8 h-8 mb-2 text-gray-400" />
                                <p className="mb-2 text-sm text-gray-500 dark:text-gray-400">
                                    <span className="font-semibold">Click to upload</span> or drag and drop
                                </p>
                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                    DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP (Max 40MB)
                                </p>
                            </div>
                            <input
                                id="writer-file-input"
                                type="file"
                                className="hidden"
                                onChange={handleFileSelect}
                                accept=".doc,.docx,.xls,.xlsx,.ppt,.pptx,.pdf,.zip"
                            />
                        </label>
                    </div>
                    {selectedFile && (
                        <div className="mt-2 p-2 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-md">
                            <div className="flex items-center gap-2">
                                <FileText className="w-4 h-4 text-green-600" />
                                <span className="text-sm text-green-800 dark:text-green-200">
                                    {selectedFile.name} ({(selectedFile.size / 1024 / 1024).toFixed(2)} MB)
                                </span>
                            </div>
                        </div>
                    )}
                </div>

                {/* Description */}
                <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Description (Optional)
                    </label>
                    <textarea
                        value={description}
                        onChange={(e) => setDescription(e.target.value)}
                        rows={3}
                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                        placeholder="Add any notes about this file..."
                        maxLength={1000}
                    />
                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                        {description.length}/1000 characters
                    </p>
                </div>

                {/* Error Display */}
                {error && (
                    <div className="flex items-center gap-2 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-md">
                        <AlertCircle className="w-5 h-5 text-red-600" />
                        <span className="text-sm text-red-800 dark:text-red-200">{error}</span>
                    </div>
                )}

                {/* Success Display */}
                {uploadStatus === 'success' && (
                    <div className="flex items-center gap-2 p-3 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-md">
                        <CheckCircle className="w-5 h-5 text-green-600" />
                        <span className="text-sm text-green-800 dark:text-green-200">
                            File uploaded successfully!
                            {fileCategory === 'complete' && ' Order status updated to "Under Review".'}
                        </span>
                    </div>
                )}

                {/* Submit Button */}
                <button
                    type="submit"
                    disabled={isUploading || !selectedFile}
                    className={`w-full flex items-center justify-center gap-2 px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white transition-colors ${
                        isUploading || !selectedFile
                            ? 'bg-gray-400 dark:bg-gray-600 cursor-not-allowed'
                            : 'bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500'
                    }`}
                >
                    {isUploading ? (
                        <>
                            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                            Uploading...
                        </>
                    ) : (
                        <>
                            <Upload className="w-4 h-4" />
                            Upload File
                        </>
                    )}
                </button>
            </form>
        </div>
    );
} 