import React from "react";
import { cn } from "@/lib/utils";
import { AlertCircle, AlertTriangle, CheckCircle, Info, X } from "lucide-react";

const variantStyles = {
    default:
        "bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200 border-gray-200 dark:border-gray-700",
    primary:
        "bg-blue-50 text-blue-800 dark:bg-blue-900/30 dark:text-blue-300 border-blue-200 dark:border-blue-800",
    success:
        "bg-green-50 text-green-800 dark:bg-green-900/30 dark:text-green-300 border-green-200 dark:border-green-800",
    warning:
        "bg-yellow-50 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-300 border-yellow-200 dark:border-yellow-800",
    destructive:
        "bg-red-50 text-red-800 dark:bg-red-900/30 dark:text-red-300 border-red-200 dark:border-red-800",
    info: "bg-indigo-50 text-indigo-800 dark:bg-indigo-900/30 dark:text-indigo-300 border-indigo-200 dark:border-indigo-800",
};

const variantIcons = {
    default: AlertCircle,
    primary: Info,
    success: CheckCircle,
    warning: AlertTriangle,
    destructive: AlertCircle,
    info: Info,
};

const Alert = React.forwardRef(
    (
        {
            className,
            variant = "default",
            title,
            message,
            children,
            dismissible,
            onDismiss,
            icon: Icon,
            ...props
        },
        ref
    ) => {
        const AlertIcon = Icon || variantIcons[variant];

        return (
            <div
                ref={ref}
                role="alert"
                className={cn(
                    "relative rounded-lg border p-4",
                    variantStyles[variant],
                    className
                )}
                {...props}
            >
                <div className="flex">
                    <div className="flex-shrink-0">
                        <AlertIcon className="h-5 w-5" aria-hidden="true" />
                    </div>
                    <div className="ml-3 flex-1">
                        {title && (
                            <h3 className="text-sm font-medium">{title}</h3>
                        )}
                        {message && (
                            <div className={cn("text-sm", title ? "mt-2" : "")}>
                                {message}
                            </div>
                        )}
                        {children}
                    </div>

                    {dismissible && (
                        <button
                            type="button"
                            className={cn(
                                "ml-auto -mx-1.5 -my-1.5 rounded-lg p-1.5 inline-flex items-center justify-center",
                                "hover:bg-white/20 focus:outline-none focus:ring-2 focus:ring-opacity-50",
                                {
                                    "focus:ring-blue-500":
                                        variant === "primary",
                                    "focus:ring-green-500":
                                        variant === "success",
                                    "focus:ring-yellow-500":
                                        variant === "warning",
                                    "focus:ring-red-500":
                                        variant === "destructive",
                                    "focus:ring-indigo-500": variant === "info",
                                    "focus:ring-gray-500":
                                        variant === "default",
                                }
                            )}
                            onClick={onDismiss}
                            aria-label="Dismiss"
                        >
                            <X className="h-4 w-4" />
                        </button>
                    )}
                </div>
            </div>
        );
    }
);

Alert.displayName = "Alert";

// Convenience components for different alert types
Alert.Success = (props) => <Alert variant="success" {...props} />;
Alert.Warning = (props) => <Alert variant="warning" {...props} />;
Alert.Error = (props) => <Alert variant="destructive" {...props} />;
Alert.Info = (props) => <Alert variant="info" {...props} />;
Alert.Primary = (props) => <Alert variant="primary" {...props} />;

export default Alert;
