import React, { useState, useEffect } from "react";
import { ChevronLeft, ChevronRight } from "lucide-react";
import { cn, formatDate } from "@/lib/utils";
import { Button } from "@/Components/ui/Button";
import {
    addMonths,
    subMonths,
    startOfMonth,
    endOfMonth,
    eachDayOfInterval,
    isSameMonth,
    isSameDay,
    startOfWeek,
    endOfWeek,
    isToday,
    addDays,
    format,
} from "date-fns";

const Calendar = ({
    mode = "single",
    selected,
    onSelect,
    className,
    disabled,
    initialFocus = false,
    ...props
}) => {
    const [currentMonth, setCurrentMonth] = useState(
        selected ? new Date(selected) : new Date()
    );
    const [focusDay, setFocusDay] = useState(null);

    // Set focus on init if needed
    useEffect(() => {
        if (initialFocus && focusDay) {
            document
                .getElementById(
                    `calendar-day-${format(focusDay, "yyyy-MM-dd")}`
                )
                ?.focus();
        }
    }, [initialFocus, focusDay]);

    // Navigate to previous month
    const prevMonth = () => {
        setCurrentMonth(subMonths(currentMonth, 1));
    };

    // Navigate to next month
    const nextMonth = () => {
        setCurrentMonth(addMonths(currentMonth, 1));
    };

    // Generate days for the calendar
    const generateDays = () => {
        const monthStart = startOfMonth(currentMonth);
        const monthEnd = endOfMonth(currentMonth);
        const startDate = startOfWeek(monthStart);
        const endDate = endOfWeek(monthEnd);

        const days = eachDayOfInterval({ start: startDate, end: endDate });

        // Generate weeks
        const weeks = [];
        let currentWeek = [];

        days.forEach((day) => {
            if (currentWeek.length === 7) {
                weeks.push(currentWeek);
                currentWeek = [];
            }
            currentWeek.push(day);
        });

        if (currentWeek.length > 0) {
            weeks.push(currentWeek);
        }

        return weeks;
    };

    const weekdays = ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"];
    const weeks = generateDays();

    // Handle day selection
    const handleDayClick = (day) => {
        if (disabled) return;
        onSelect?.(day);
    };

    // Handle keyboard navigation
    const handleKeyDown = (e, day) => {
        if (disabled) return;

        setFocusDay(day);

        switch (e.key) {
            case "ArrowLeft":
                e.preventDefault();
                setFocusDay(addDays(day, -1));
                document
                    .getElementById(
                        `calendar-day-${format(addDays(day, -1), "yyyy-MM-dd")}`
                    )
                    ?.focus();
                break;
            case "ArrowRight":
                e.preventDefault();
                setFocusDay(addDays(day, 1));
                document
                    .getElementById(
                        `calendar-day-${format(addDays(day, 1), "yyyy-MM-dd")}`
                    )
                    ?.focus();
                break;
            case "ArrowUp":
                e.preventDefault();
                setFocusDay(addDays(day, -7));
                document
                    .getElementById(
                        `calendar-day-${format(addDays(day, -7), "yyyy-MM-dd")}`
                    )
                    ?.focus();
                break;
            case "ArrowDown":
                e.preventDefault();
                setFocusDay(addDays(day, 7));
                document
                    .getElementById(
                        `calendar-day-${format(addDays(day, 7), "yyyy-MM-dd")}`
                    )
                    ?.focus();
                break;
            case "Enter":
            case " ":
                e.preventDefault();
                handleDayClick(day);
                break;
            default:
                break;
        }
    };

    return (
        <div className={cn("p-2", className)} {...props}>
            {/* Calendar header */}
            <div className="flex justify-between items-center mb-4">
                <Button
                    type="button"
                    variant="ghost"
                    size="sm"
                    onClick={prevMonth}
                    disabled={disabled}
                >
                    <ChevronLeft className="h-4 w-4" />
                    <span className="sr-only">Previous month</span>
                </Button>
                <h2 className="text-sm font-semibold text-center text-gray-900 dark:text-white">
                    {format(currentMonth, "MMMM yyyy")}
                </h2>
                <Button
                    type="button"
                    variant="ghost"
                    size="sm"
                    onClick={nextMonth}
                    disabled={disabled}
                >
                    <ChevronRight className="h-4 w-4" />
                    <span className="sr-only">Next month</span>
                </Button>
            </div>

            {/* Weekday headers */}
            <div className="grid grid-cols-7 gap-1 mb-1">
                {weekdays.map((day) => (
                    <div
                        key={day}
                        className="text-xs font-medium text-gray-500 dark:text-gray-400 text-center"
                    >
                        {day}
                    </div>
                ))}
            </div>

            {/* Calendar grid */}
            <div className="space-y-1">
                {weeks.map((week, weekIndex) => (
                    <div key={weekIndex} className="grid grid-cols-7 gap-1">
                        {week.map((day) => {
                            const isCurrentMonth = isSameMonth(
                                day,
                                currentMonth
                            );
                            const isSelectedDay =
                                selected && isSameDay(day, new Date(selected));
                            const isTodayDate = isToday(day);

                            return (
                                <div key={day.toString()} className="p-0">
                                    <button
                                        id={`calendar-day-${format(
                                            day,
                                            "yyyy-MM-dd"
                                        )}`}
                                        type="button"
                                        className={cn(
                                            "w-full h-8 flex items-center justify-center rounded-md text-sm",
                                            "focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-1",
                                            isCurrentMonth
                                                ? "text-gray-900 dark:text-gray-100"
                                                : "text-gray-400 dark:text-gray-600",
                                            isSelectedDay &&
                                                "bg-blue-500 text-white dark:bg-blue-600",
                                            isTodayDate &&
                                                !isSelectedDay &&
                                                "border border-blue-500 dark:border-blue-600",
                                            !isSelectedDay &&
                                                "hover:bg-gray-100 dark:hover:bg-gray-800",
                                            disabled &&
                                                "opacity-50 cursor-not-allowed"
                                        )}
                                        disabled={disabled || !isCurrentMonth}
                                        onClick={() => handleDayClick(day)}
                                        onKeyDown={(e) => handleKeyDown(e, day)}
                                        tabIndex={isCurrentMonth ? 0 : -1}
                                    >
                                        {format(day, "d")}
                                    </button>
                                </div>
                            );
                        })}
                    </div>
                ))}
            </div>
        </div>
    );
};

export { Calendar };
