import React from "react";
import { cn } from "@/lib/utils";
import { FileQuestion, Plus, RefreshCw, Search } from "lucide-react";
import { Button } from "@/Components/ui/Button";
import { Link } from "@inertiajs/react";

const EmptyState = ({
    className,
    icon: Icon = FileQuestion, // Properly capitalized to indicate a component
    title = "No items found",
    description = "There are no items to display.",
    action,
    actionText = "Create New",
    actionIcon: ActionIcon = Plus, // Properly capitalized
    actionUrl, // New prop for action URL
    secondaryAction,
    secondaryActionText = "Refresh",
    secondaryActionIcon: SecondaryIcon = RefreshCw, // Properly capitalized
    secondaryActionUrl, // New prop for secondary action URL
    compact = false,
    children,
    ...props
}) => {
    // Check if the icon is a React element instead of a component
    // This makes the component more robust against incorrect usage
    const isIconElement = React.isValidElement(Icon);
    const isActionIconElement = React.isValidElement(ActionIcon);
    const isSecondaryIconElement = React.isValidElement(SecondaryIcon);

    return (
        <div
            className={cn(
                "flex flex-col items-center justify-center text-center px-6 py-8 bg-white dark:bg-gray-800 rounded-lg shadow-sm",
                compact ? "py-6" : "py-12",
                className
            )}
            {...props}
        >
            {/* Handle both component and element cases for the main icon */}
            {isIconElement ? (
                <div
                    className={cn(
                        "text-gray-400 dark:text-gray-500",
                        compact ? "h-10 w-10 mb-3" : "h-16 w-16 mb-4"
                    )}
                >
                    {Icon}
                </div>
            ) : (
                <Icon
                    className={cn(
                        "text-gray-400 dark:text-gray-500",
                        compact ? "h-10 w-10 mb-3" : "h-16 w-16 mb-4"
                    )}
                    aria-hidden="true"
                />
            )}
            <h3
                className={cn(
                    "font-semibold text-gray-900 dark:text-white",
                    compact ? "text-lg mb-1" : "text-xl mb-2"
                )}
            >
                {title}
            </h3>
            <p className="text-gray-500 dark:text-gray-400 max-w-md">
                {description}
            </p>

            {children}

            {(action || actionUrl || secondaryAction || secondaryActionUrl) && (
                <div
                    className={cn("flex gap-3 mt-4", compact ? "mt-3" : "mt-6")}
                >
                    {/* Primary action button */}
                    {(action || actionUrl) && (
                        <>
                            {actionUrl ? (
                                // Use Button with asChild but DON'T add href to Button
                                <Button
                                    asChild
                                    className="inline-flex items-center"
                                >
                                    <Link
                                        href={actionUrl}
                                        className="flex items-center"
                                    >
                                        {isActionIconElement ? (
                                            <span className="w-4 h-4 mr-2">
                                                {ActionIcon}
                                            </span>
                                        ) : (
                                            <ActionIcon className="w-4 h-4 mr-2" />
                                        )}
                                        {actionText}
                                    </Link>
                                </Button>
                            ) : (
                                <Button
                                    onClick={action}
                                    className="inline-flex items-center"
                                >
                                    {isActionIconElement ? (
                                        <span className="w-4 h-4 mr-2">
                                            {ActionIcon}
                                        </span>
                                    ) : (
                                        <ActionIcon className="w-4 h-4 mr-2" />
                                    )}
                                    {actionText}
                                </Button>
                            )}
                        </>
                    )}

                    {/* Secondary action button */}
                    {(secondaryAction || secondaryActionUrl) && (
                        <>
                            {secondaryActionUrl ? (
                                // Same fix for secondary button
                                <Button variant="outline" asChild>
                                    <Link href={secondaryActionUrl}>
                                        {isSecondaryIconElement ? (
                                            <span className="w-4 h-4 mr-2">
                                                {SecondaryIcon}
                                            </span>
                                        ) : (
                                            <SecondaryIcon className="w-4 h-4 mr-2" />
                                        )}
                                        {secondaryActionText}
                                    </Link>
                                </Button>
                            ) : (
                                <Button
                                    variant="outline"
                                    onClick={secondaryAction}
                                >
                                    {isSecondaryIconElement ? (
                                        <span className="w-4 h-4 mr-2">
                                            {SecondaryIcon}
                                        </span>
                                    ) : (
                                        <SecondaryIcon className="w-4 h-4 mr-2" />
                                    )}
                                    {secondaryActionText}
                                </Button>
                            )}
                        </>
                    )}
                </div>
            )}
        </div>
    );
};

// Specialized variants - using proper component pattern
EmptyState.Search = (props) => (
    <EmptyState
        icon={Search}
        title="No results found"
        description="Try adjusting your search or filter criteria to find what you're looking for."
        actionText="Clear filters"
        {...props}
    />
);

EmptyState.NoData = (props) => (
    <EmptyState
        title="No data available"
        description="There's no data to display at the moment."
        actionText="Refresh"
        actionIcon={RefreshCw}
        {...props}
    />
);

EmptyState.Compact = (props) => <EmptyState compact={true} {...props} />;

export default EmptyState;
