import React, { createContext, useContext, useId } from "react";
import { cn } from "@/lib/utils";

// Create context for radio group state
const RadioGroupContext = createContext(null);

const useRadioGroupContext = () => {
    const context = useContext(RadioGroupContext);
    if (!context) {
        throw new Error(
            "RadioGroup components must be used within a RadioGroup"
        );
    }
    return context;
};

const RadioGroup = React.forwardRef(
    (
        {
            className,
            value,
            onValueChange,
            defaultValue,
            name,
            disabled,
            required,
            ...props
        },
        ref
    ) => {
        const rootId = useId();

        return (
            <RadioGroupContext.Provider
                value={{
                    value,
                    onValueChange,
                    name,
                    disabled,
                    required,
                    rootId,
                }}
            >
                <div
                    ref={ref}
                    role="radiogroup"
                    className={cn("space-y-2", className)}
                    {...props}
                />
            </RadioGroupContext.Provider>
        );
    }
);

RadioGroup.displayName = "RadioGroup";

const RadioGroupItem = React.forwardRef(
    ({ className, value, id, disabled, ...props }, ref) => {
        const context = useRadioGroupContext();
        const itemId = id || `${context.rootId}-${value}`;
        const checked = context.value === value;
        const itemDisabled = disabled || context.disabled;

        return (
            <div className="flex items-center space-x-2">
                <input
                    type="radio"
                    id={itemId}
                    ref={ref}
                    value={value}
                    name={context.name}
                    checked={checked}
                    disabled={itemDisabled}
                    required={context.required}
                    onChange={() => context.onValueChange(value)}
                    className={cn(
                        "h-4 w-4 cursor-pointer rounded-full border-gray-300 text-blue-600 focus:ring-blue-500 dark:border-gray-600 dark:bg-gray-700 dark:focus:ring-offset-gray-800",
                        checked
                            ? "border-blue-500"
                            : "border-gray-300 dark:border-gray-600",
                        itemDisabled && "cursor-not-allowed opacity-50",
                        className
                    )}
                    {...props}
                />
            </div>
        );
    }
);

RadioGroupItem.displayName = "RadioGroupItem";

const RadioGroupLabel = React.forwardRef(
    ({ className, htmlFor, ...props }, ref) => {
        return (
            <label
                ref={ref}
                htmlFor={htmlFor}
                className={cn(
                    "text-sm font-medium text-gray-700 dark:text-gray-300 cursor-pointer",
                    className
                )}
                {...props}
            />
        );
    }
);

RadioGroupLabel.displayName = "RadioGroupLabel";

const RadioGroupIndicator = React.forwardRef(({ className, ...props }, ref) => {
    return (
        <span
            ref={ref}
            className={cn(
                "absolute inset-0 flex items-center justify-center",
                className
            )}
            {...props}
        />
    );
});

RadioGroupIndicator.displayName = "RadioGroupIndicator";

export { RadioGroup, RadioGroupItem, RadioGroupLabel, RadioGroupIndicator };
