import React from "react";
import { cn } from "@/lib/utils";

const TabsContext = React.createContext({
    value: "",
    onChange: () => {},
});

const Tabs = React.forwardRef(
    (
        { children, value, onValueChange, defaultValue, className, ...props },
        ref
    ) => {
        const [selectedTab, setSelectedTab] = React.useState(
            value || defaultValue || ""
        );

        React.useEffect(() => {
            if (value !== undefined) {
                setSelectedTab(value);
            }
        }, [value]);

        const handleTabChange = (newValue) => {
            if (value === undefined) {
                setSelectedTab(newValue);
            }

            if (onValueChange) {
                onValueChange(newValue);
            }
        };

        return (
            <TabsContext.Provider
                value={{ value: selectedTab, onChange: handleTabChange }}
            >
                <div ref={ref} className={cn("w-full", className)} {...props}>
                    {children}
                </div>
            </TabsContext.Provider>
        );
    }
);

Tabs.displayName = "Tabs";

const TabsList = React.forwardRef(({ className, ...props }, ref) => {
    return (
        <div
            ref={ref}
            className={cn(
                "flex flex-wrap h-10 items-center justify-center rounded-md bg-gray-100 p-1 text-gray-500 dark:bg-gray-800 dark:text-gray-400",
                className
            )}
            role="tablist"
            {...props}
        />
    );
});

TabsList.displayName = "TabsList";

const TabsTrigger = React.forwardRef(
    ({ className, value, disabled, children, ...props }, ref) => {
        const { value: selectedValue, onChange } =
            React.useContext(TabsContext);
        const isSelected = selectedValue === value;

        return (
            <button
                ref={ref}
                role="tab"
                type="button"
                aria-selected={isSelected}
                disabled={disabled}
                data-state={isSelected ? "active" : "inactive"}
                className={cn(
                    "inline-flex items-center justify-center whitespace-nowrap rounded-sm px-3 py-1.5 text-sm font-medium ring-offset-white transition-all focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-gray-950 focus-visible:ring-offset-2 disabled:pointer-events-none disabled:opacity-50 dark:ring-offset-gray-950 dark:focus-visible:ring-gray-300",
                    isSelected
                        ? "bg-white text-gray-950 shadow-sm dark:bg-gray-700 dark:text-gray-50"
                        : "text-gray-600 hover:bg-gray-200 dark:text-gray-400 dark:hover:bg-gray-700/50",
                    className
                )}
                onClick={() => {
                    if (!disabled) {
                        onChange(value);
                    }
                }}
                {...props}
            >
                {children}
            </button>
        );
    }
);

TabsTrigger.displayName = "TabsTrigger";

const TabsContent = React.forwardRef(
    ({ className, value, children, ...props }, ref) => {
        const { value: selectedValue } = React.useContext(TabsContext);
        const isSelected = selectedValue === value;

        if (!isSelected) return null;

        return (
            <div
                ref={ref}
                role="tabpanel"
                tabIndex={0}
                data-state={isSelected ? "active" : "inactive"}
                className={cn(
                    "mt-2 ring-offset-white focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-gray-950 focus-visible:ring-offset-2 dark:ring-offset-gray-950 dark:focus-visible:ring-gray-300",
                    className
                )}
                {...props}
            >
                {children}
            </div>
        );
    }
);

TabsContent.displayName = "TabsContent";

export { Tabs, TabsList, TabsTrigger, TabsContent };
