import React, { createContext, useContext, useState, useRef, useEffect } from 'react';

const DropdownMenuContext = createContext();

const DropdownMenu = ({ children, ...props }) => {
    return <div {...props}>{children}</div>;
};

const DropdownMenuTrigger = React.forwardRef(({ asChild, children, ...props }, ref) => {
    const context = useContext(DropdownMenuContext);
    
    if (!context) {
        throw new Error('DropdownMenuTrigger must be used within a DropdownMenu');
    }

    const handleClick = (e) => {
        context.setOpen(!context.open);
        if (props.onClick) {
            props.onClick(e);
        }
    };

    if (asChild) {
        return React.cloneElement(children, {
            ref,
            onClick: handleClick,
            ...props
        });
    }

    return (
        <button
            ref={ref}
            onClick={handleClick}
            {...props}
        >
            {children}
        </button>
    );
});

const DropdownMenuContent = React.forwardRef(({ children, className = '', align = 'end', ...props }, ref) => {
    const context = useContext(DropdownMenuContext);
    const contentRef = useRef(null);

    useEffect(() => {
        const handleClickOutside = (event) => {
            if (contentRef.current && !contentRef.current.contains(event.target)) {
                context.setOpen(false);
            }
        };

        if (context.open) {
            document.addEventListener('mousedown', handleClickOutside);
        }

        return () => {
            document.removeEventListener('mousedown', handleClickOutside);
        };
    }, [context.open]);

    if (!context.open) {
        return null;
    }

    const alignmentClasses = {
        start: 'left-0',
        center: 'left-1/2 transform -translate-x-1/2',
        end: 'right-0'
    };

    return (
        <div
            ref={contentRef}
            className={`absolute top-full mt-1 w-48 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-md shadow-lg z-50 ${alignmentClasses[align]} ${className}`}
            {...props}
        >
            {children}
        </div>
    );
});

const DropdownMenuItem = React.forwardRef(({ children, className = '', onClick, asChild, ...props }, ref) => {
    const context = useContext(DropdownMenuContext);

    const handleClick = (e) => {
        if (onClick) {
            onClick(e);
        }
        context.setOpen(false);
    };

    return (
        <div
            ref={ref}
            onClick={handleClick}
            className={`px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 cursor-pointer first:rounded-t-md last:rounded-b-md ${className}`}
            {...props}
        >
            {children}
        </div>
    );
});

const DropdownMenuSeparator = React.forwardRef(({ className = '', ...props }, ref) => {
    return (
        <div
            ref={ref}
            className={`border-t border-gray-200 dark:border-gray-700 my-1 ${className}`}
            {...props}
        />
    );
});

const DropdownMenuProvider = ({ children }) => {
    const [open, setOpen] = useState(false);

    return (
        <DropdownMenuContext.Provider value={{ open, setOpen }}>
            <div className="relative">
                {children}
            </div>
        </DropdownMenuContext.Provider>
    );
};

// Main component that provides context
const DropdownMenuWithProvider = ({ children, ...props }) => {
    return (
        <DropdownMenuProvider>
            <DropdownMenu {...props}>
                {children}
            </DropdownMenu>
        </DropdownMenuProvider>
    );
};

export {
    DropdownMenuWithProvider as DropdownMenu,
    DropdownMenuTrigger,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
};

