import { useState, useEffect, useRef } from "react";
import { Link, usePage, router } from "@inertiajs/react";
import {
    LayoutDashboard,
    ListOrdered,
    Users,
    FileText,
    Settings,
    Bell,
    User,
    LogOut,
    Menu,
    X,
    Search,
    ChevronDown,
    ChevronRight,
    Sun,
    BookOpen,
    Tag,
    Moon,
    BarChart,
    GraduationCap,
    Settings2,
    Percent,
    BarChart2,
    CreditCard,
    Upload,
    FolderOpen,
    Tags,
    UserCheck,
    Award,
    CalendarRange,
    Zap,
    TicketPercent,
    DollarSign,
    ScrollText,
    Gift,
    AlertTriangle,
    Globe,
    Edit3,
    Eye,
    Search as SearchIcon,
    Layout,
    Code,
} from "lucide-react";
import { ThemeProvider, useTheme } from "@/Contexts/ThemeContext";
import Tooltip from "@/Components/ui/Tooltip";
import NotificationBell from "@/Components/Notifications/NotificationBell";
import MessagingWidget from "@/Components/Messaging/MessagingWidget";
import { Toaster } from "react-hot-toast";

export default function AdminLayout({ children }) {
    return (
        <ThemeProvider>
            <AdminLayoutContent>{children}</AdminLayoutContent>
        </ThemeProvider>
    );
}

function AdminLayoutContent({ children }) {
    const { darkMode, toggleDarkMode } = useTheme();
    const user = usePage().props.auth.user;
    const isMobile = () => window.innerWidth < 768;
    const isDesktop = !isMobile();
    const [sidebarOpen, setSidebarOpen] = useState(isDesktop);
    const [profileOpen, setProfileOpen] = useState(false);
    const [settingsOpen, setSettingsOpen] = useState(isDesktop);
    const [testsOpen, setTestsOpen] = useState(isDesktop);
    const [marketingOpen, setMarketingOpen] = useState(isDesktop);
    const [blogOpen, setBlogOpen] = useState(isDesktop);
    const [pagesOpen, setPagesOpen] = useState(isDesktop);
    const [searchQuery, setSearchQuery] = useState("");
    const sidebarRef = useRef(null);

    // Logout handler - Inertia handles CSRF automatically via cookie
    const handleLogout = () => {
        router.post(route('logout'));
    };

    useEffect(() => {
        const handleResize = () => {
            const isDesktop = !isMobile();
            setSidebarOpen(isDesktop);
            // Expand submenus on desktop, collapse on mobile
            setSettingsOpen(isDesktop);
            setTestsOpen(isDesktop);
            setMarketingOpen(isDesktop);
            setBlogOpen(isDesktop);
            setPagesOpen(isDesktop);
        };
        setSidebarOpen(!isMobile());
        window.addEventListener("resize", handleResize);
        return () => window.removeEventListener("resize", handleResize);
    }, []);

    const handleParentMenuClick = (hasSubmenus, e) => {
        if (!sidebarOpen && hasSubmenus) {
            setSidebarOpen(true);
            e.preventDefault();
        } else if (!hasSubmenus && !sidebarOpen) {
        }
    };

    // Filter menu items based on search query
    const filterMenuItems = (items) => {
        if (!searchQuery) return true;
        return items.toLowerCase().includes(searchQuery.toLowerCase());
    };

    return (
        <div
            className={`flex flex-col min-h-screen ${
                darkMode
                    ? "bg-gray-900 text-white"
                    : "bg-gray-100 text-gray-900"
            }`}
        >
            <header className="w-full flex justify-between items-center p-4 bg-gradient-to-r from-cyan-600 to-indigo-800 dark:bg-none dark:bg-gray-800 shadow-md border-b border-gray-200 dark:border-gray-700 sticky right-0 z-30">
                <button
                    onClick={() => setSidebarOpen(!sidebarOpen)}
                    className="p-1 hover:bg-gray-200/20 dark:hover:bg-gray-700 rounded-lg text-white transition-colors duration-200"
                    aria-label={sidebarOpen ? "Close sidebar" : "Open sidebar"}
                >
                    {sidebarOpen ? <X size={24} /> : <Menu size={24} />}
                </button>
                <div className="text-lg text-gray-50 mx-3 font-bold md:block">
                    <Link
                        href="/"
                        className="text-lg text-gray-50 mx-3 font-bold md:block"
                    >
                        AcademicScribe
                    </Link>
                </div>
                <div className="flex items-center gap-4 ml-auto">
                    <button
                        onClick={toggleDarkMode}
                        className="p-2 rounded-full bg-gray-200/20 dark:bg-gray-700 hover:bg-gray-200/30 dark:hover:bg-gray-600 transition-colors duration-200 text-white"
                        aria-label={
                            darkMode
                                ? "Switch to light mode"
                                : "Switch to dark mode"
                        }
                    >
                        {darkMode ? <Sun size={20} /> : <Moon size={20} />}
                    </button>
                    <NotificationBell />
                    <div className="relative">
                        <button
                            onClick={() => setProfileOpen(!profileOpen)}
                            className="flex items-center text-white gap-2 rounded-full hover:bg-gray-200/20 dark:hover:bg-gray-700 px-3 py-1.5 transition-colors duration-200"
                        >
                            {user.user_type === "admin" && (
                                <span className="bg-red-500 text-white text-xs font-bold px-1.5 py-0.5 rounded-full">
                                    AD
                                </span>
                            )}
                            <span className="hidden sm:block">
                                {user?.name?.split(" ")[0] || "User"}
                            </span>
                            <ChevronDown className="h-4 w-4" />
                        </button>
                        {profileOpen && (
                            <div className="absolute right-0 mt-2 p-2 w-48 bg-white dark:bg-gray-800 shadow-lg rounded-lg z-50 border border-gray-200 dark:border-gray-700 animate-in fade-in duration-200">
                                <div className="px-3 py-2 border-b border-gray-200 dark:border-gray-700 mb-1">
                                    <p className="text-sm font-medium text-gray-900 dark:text-white">
                                        {user?.name}
                                    </p>
                                    <p className="text-xs text-gray-500 dark:text-gray-400 truncate">
                                        {user?.email}
                                    </p>
                                </div>
                                <Link
                                    href="/profile"
                                    className="flex items-center gap-2 p-2 text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md transition-colors duration-200"
                                    onClick={() => setProfileOpen(false)}
                                >
                                    <User size={18} /> Profile
                                </Link>
                                <Link
                                    href={route("logout")}
                                    method="post"
                                    as="button"
                                    className="flex items-center gap-2 p-2 text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 w-full text-left rounded-md transition-colors duration-200"
                                    onClick={handleLogout}
                                >
                                    <LogOut size={18} /> Logout
                                </Link>
                            </div>
                        )}
                    </div>
                </div>
            </header>
            <div className="flex flex-1">
                <aside
                    ref={sidebarRef}
                    className={`fixed md:relative transition-all duration-300 h-[calc(100vh-4rem)] ${
                        sidebarOpen
                            ? "translate-x-0 w-64"
                            : "-translate-x-full md:translate-x-0 md:w-16"
                    } bg-gradient-to-r from-indigo-800 to-cyan-600 dark:bg-none text-gray-50 rounded-lg dark:bg-gray-800 shadow-xl border-r border-gray-200 dark:border-gray-700 z-20 overflow-hidden`}
                >
                    <div className="flex flex-col h-full">
                        {/* Search box - visible only when sidebar is open */}
                        {sidebarOpen && (
                            <div className="px-4 py-3 border-b border-indigo-700/50 dark:border-gray-700">
                                <div className="relative">
                                    <Search className="absolute left-2 right-2.5 h-4 w-4 text-gray-400" />
                                    <input
                                        type="text"
                                        placeholder="Search menu..."
                                        className="w-full pl-8 pr-3 py-2 bg-indigo-900/30 dark:bg-gray-700 border border-indigo-700/50 dark:border-gray-600 rounded-md text-sm text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-cyan-500 focus:border-transparent"
                                        value={searchQuery}
                                        onChange={(e) =>
                                            setSearchQuery(e.target.value)
                                        }
                                    />
                                </div>
                            </div>
                        )}
                        {/* Scrollable menu container */}
                        <nav className="flex-1 overflow-y-auto overflow-x-hidden py-4 px-3 flex flex-col space-y-1.5 scrollbar-thin scrollbar-thumb-indigo-700 dark:scrollbar-thumb-gray-700">
                            {/* Dashboard Link */}
                            {filterMenuItems("Dashboard") && (
                                <Tooltip
                                    content="Dashboard"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <Link
                                        href="/admin/dashboard"
                                        className={`flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("admin.dashboard")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={(e) =>
                                            handleParentMenuClick(false, e)
                                        }
                                    >
                                        <LayoutDashboard size={20} />
                                        {sidebarOpen && <span>Dashboard</span>}
                                    </Link>
                                </Tooltip>
                            )}
                            {/* Orders Link */}
                            {filterMenuItems("Orders") && (
                                <Tooltip
                                    content="Orders"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href="/admin/orders"
                                        className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("admin.orders.*")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={() =>
                                            handleParentMenuClick(false)
                                        }
                                    >
                                        <ListOrdered size={20} />
                                        {sidebarOpen && <span>Orders</span>}
                                    </Link>
                                </Tooltip>
                            )}
                            {/* Users Link */}
                            {filterMenuItems("Users") && (
                                <Tooltip
                                    content="Users"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href="/admin/users"
                                        className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("admin.users.*")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={() =>
                                            handleParentMenuClick(false)
                                        }
                                    >
                                        <Users size={20} />
                                        {sidebarOpen && <span>Users</span>}
                                    </Link>
                                </Tooltip>
                            )}

                            {/* Writer Infractions Link */}
                            {filterMenuItems("Writer Infractions") && (
                                <Tooltip
                                    content="Writer Infractions"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href="/admin/writer-infractions"
                                        className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("writer-infractions.*")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={() =>
                                            handleParentMenuClick(false)
                                        }
                                    >
                                        <AlertTriangle size={20} />
                                        {sidebarOpen && <span>Writer Infractions</span>}
                                    </Link>
                                </Tooltip>
                            )}

                            {/* Pages Management Dropdown */}
                            {filterMenuItems("Pages") && (
                                <Tooltip
                                    content="Pages"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setPagesOpen(!pagesOpen);
                                                handleParentMenuClick(true);
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                pagesOpen ||
                                                route().current("admin.cms.*")
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                        >
                                            <Globe size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Pages</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            pagesOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>
                                        {pagesOpen && sidebarOpen && (
                                            <div className="ml-6 mt-1 pl-2 border-l border-indigo-500/50 dark:border-gray-600 space-y-1 animate-in slide-in-from-left duration-200">
                                                {/* Page Management */}
                                                <div className="mb-2">
                                                    <p className="text-xs text-indigo-300 dark:text-gray-400 uppercase tracking-wider pl-2 mb-1.5">
                                                        Content Management
                                                    </p>
                                                    <Link
                                                        href="/admin/cms/pages"
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.cms.pages.*"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <FileText size={16} />
                                                        <span>All Pages</span>
                                                    </Link>
                                                    <Link
                                                        href="/admin/cms/pages/create"
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.cms.pages.create"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <Edit3 size={16} />
                                                        <span>Create Page</span>
                                                    </Link>
                                                    <Link
                                                        href="/admin/cms/content-blocks"
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.cms.content-blocks.*"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <ScrollText size={16} />
                                                        <span>Content Blocks</span>
                                                    </Link>
                                                    <Link
                                                        href="/admin/cms/footer"
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.cms.footer.*"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <Layout size={16} />
                                                        <span>Footer Management</span>
                                                    </Link>
                                                </div>
                                                {/* SEO & Media */}
                                                <div className="mb-2">
                                                    <p className="text-xs text-indigo-300 dark:text-gray-400 uppercase tracking-wider pl-2 mb-1.5">
                                                        SEO & Media
                                                    </p>
                                                    <Link
                                                        href="/admin/cms/media"
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.cms.media.*"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <Upload size={16} />
                                                        <span>Media Library</span>
                                                    </Link>
                                                    <Link
                                                        href="/admin/cms/pages?type=service_page"
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.cms.pages.index"
                                                            ) && new URLSearchParams(window.location.search).get('type') === 'service_page'
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <Eye size={16} />
                                                        <span>Service Pages</span>
                                                    </Link>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}

                            {/* Blog Management Dropdown */}
                            {filterMenuItems("Blog") && (
                                <Tooltip
                                    content="Blog"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setBlogOpen(!blogOpen);
                                                handleParentMenuClick(true);
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                blogOpen ||
                                                route().current("admin.blog.*")
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                        >
                                            <BookOpen size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Blog</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            blogOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>
                                        {blogOpen && sidebarOpen && (
                                            <div className="ml-6 mt-1 pl-2 border-l border-indigo-500/50 dark:border-gray-600 space-y-1 animate-in slide-in-from-left duration-200">
                                                {/* Blog Management */}
                                                <div className="mb-2">
                                                    <p className="text-xs text-indigo-300 dark:text-gray-400 uppercase tracking-wider pl-2 mb-1.5">
                                                        Content Management
                                                    </p>
                                                    <Link
                                                        href={route(
                                                            "admin.blog.dashboard"
                                                        )}
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.blog.dashboard"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <BarChart2 size={16} />
                                                        <span>Dashboard</span>
                                                    </Link>
                                                    <Link
                                                        href="/admin/blog/posts"
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.blog.posts.*"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <FileText size={16} />
                                                        <span>All Posts</span>
                                                    </Link>
                                                    <Link
                                                        href="/admin/blog/upload"
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.blog.upload"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <Upload size={16} />
                                                        <span>
                                                            Upload Files
                                                        </span>
                                                    </Link>
                                                </div>
                                                {/* Blog Organization */}
                                                <div className="mb-2">
                                                    <p className="text-xs text-indigo-300 dark:text-gray-400 uppercase tracking-wider pl-2 mb-1.5">
                                                        Organization
                                                    </p>
                                                    <Link
                                                        href={route(
                                                            "admin.blog.categories"
                                                        )}
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.blog.categories"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <FolderOpen size={16} />
                                                        <span>Categories</span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.blog.tags"
                                                        )}
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.blog.tags"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <Tags size={16} />
                                                        <span>Tags</span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.blog.authors"
                                                        )}
                                                        className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                            route().current(
                                                                "admin.blog.authors"
                                                            )
                                                                ? "bg-white/10 dark:bg-gray-600"
                                                                : ""
                                                        }`}
                                                    >
                                                        <UserCheck size={16} />
                                                        <span>Authors</span>
                                                    </Link>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}

                            {/* Marketing & Loyalty Dropdown - Enhanced */}
                            {filterMenuItems("Marketing") && (
                                <Tooltip
                                    content="Marketing"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setMarketingOpen(
                                                    !marketingOpen
                                                );
                                                handleParentMenuClick(true);
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                marketingOpen
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                        >
                                            <Tag size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Marketing</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            marketingOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>

                            {/* Writer Payment Management */}
                            {filterMenuItems("Writer Payments") && (
                                <Tooltip
                                    content="Writer Payments"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <Link
                                        href="/admin/writer-withdrawals"
                                        className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("admin.writer-withdrawals.*")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={() =>
                                            handleParentMenuClick(false)
                                        }
                                    >
                                        <DollarSign size={20} />
                                        {sidebarOpen && <span>Writer Withdrawals</span>}
                                    </Link>
                                </Tooltip>
                            )}
                                        {marketingOpen && sidebarOpen && (
                                            <div className="ml-6 mt-1 pl-2 border-l border-indigo-500/50 dark:border-gray-600 space-y-1 animate-in slide-in-from-left duration-200">
                                                {/* Coupon Management */}
                                                <div className="mb-2">
                                                    <p className="text-xs text-indigo-300 dark:text-gray-400 uppercase tracking-wider pl-2 mb-1.5">
                                                        Coupon Management
                                                    </p>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.index"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <Tag size={16} />
                                                        <span>All Coupons</span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.statistics"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <BarChart size={16} />
                                                        <span>
                                                            Coupon Statistics
                                                        </span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.default-coupons.index"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <Percent size={16} />
                                                        <span>
                                                            Default Coupons
                                                        </span>
                                                    </Link>
                                                </div>
                                                {/* Generate Coupons */}
                                                <div className="mb-2">
                                                    <p className="text-xs text-indigo-300 dark:text-gray-400 uppercase tracking-wider pl-2 mb-1.5">
                                                        Generate Coupons
                                                    </p>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.generate.loyalty.form"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <Award size={16} />
                                                        <span>
                                                            Loyalty Coupons
                                                        </span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.generate.seasonal.form"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <CalendarRange
                                                            size={16}
                                                        />
                                                        <span>
                                                            Seasonal Coupons
                                                        </span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.generate.flash.form"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <Zap size={16} />
                                                        <span>Flash Sale</span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.generate.first-order.form"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <TicketPercent
                                                            size={16}
                                                        />
                                                        <span>First Order</span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.generate.volume.form"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <ScrollText size={16} />
                                                        <span>
                                                            Volume Discount
                                                        </span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.generate.referral.form"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <User size={16} />
                                                        <span>Referral</span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.coupons.generate.anniversary.form"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <Gift size={16} />
                                                        <span>Anniversary</span>
                                                    </Link>
                                                </div>
                                                {/* Loyalty & Seasonal Settings */}
                                                <div>
                                                    <p className="text-xs text-indigo-300 dark:text-gray-400 uppercase tracking-wider pl-2 mb-1.5">
                                                        Loyalty & Seasonal
                                                    </p>
                                                    <Link
                                                        href={route(
                                                            "admin.loyalty-tiers.index"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <Award size={16} />
                                                        <span>
                                                            Loyalty Tiers
                                                        </span>
                                                    </Link>
                                                    <Link
                                                        href="/admin/seasonal-types"
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <CalendarRange
                                                            size={16}
                                                        />
                                                        <span>
                                                            Seasonal Types
                                                        </span>
                                                    </Link>
                                                    <Link
                                                        href={route(
                                                            "admin.loyalty-tiers.statistics"
                                                        )}
                                                        className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                    >
                                                        <BarChart size={16} />
                                                        <span>
                                                            Loyalty Statistics
                                                        </span>
                                                    </Link>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}
                            {/* Payment Methods Section */}
                            {filterMenuItems("Payment") && (
                                <div className="pt-2">
                                    <p
                                        className={`text-xs text-indigo-300 dark:text-gray-400 uppercase tracking-wider px-3 mb-1.5 ${
                                            sidebarOpen ? "block" : "hidden"
                                        }`}
                                    >
                                        Payments
                                    </p>
                                    <Tooltip
                                        content="Payment Methods"
                                        position={
                                            sidebarOpen ? "right" : "right"
                                        }
                                    >
                                        <Link
                                            href="/admin/payment-methods"
                                            className={`flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                route().current(
                                                    "admin.payment-methods.*"
                                                )
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                            onClick={() =>
                                                handleParentMenuClick(false)
                                            }
                                        >
                                            <CreditCard size={20} />
                                            {sidebarOpen && (
                                                <span>Payment Methods</span>
                                            )}
                                        </Link>
                                    </Tooltip>
                                    <Tooltip
                                        content="Payment Analytics"
                                        position={
                                            sidebarOpen ? "right" : "right"
                                        }
                                    >
                                        <Link
                                            href={route(
                                                "admin.payment.analytics"
                                            )}
                                            className={`flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                route().current(
                                                    "admin.payment.analytics"
                                                )
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                            onClick={() =>
                                                handleParentMenuClick(false)
                                            }
                                        >
                                            <BarChart size={20} />
                                            {sidebarOpen && (
                                                <span>Payment Analytics</span>
                                            )}
                                        </Link>
                                    </Tooltip>
                                    <Tooltip
                                        content="Payment History"
                                        position={
                                            sidebarOpen ? "right" : "right"
                                        }
                                    >
                                        <Link
                                            href={route(
                                                "admin.payments.history"
                                            )}
                                            className={`flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                route().current(
                                                    "admin.payments.*"
                                                )
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                            onClick={() =>
                                                handleParentMenuClick(false)
                                            }
                                        >
                                            <CreditCard size={20} />
                                            {sidebarOpen && (
                                                <span>Payment History</span>
                                            )}
                                        </Link>
                                    </Tooltip>
                                </div>
                            )}
                            {/* Tests and Essays Dropdown */}
                            {filterMenuItems("Tests") && (
                                <Tooltip
                                    content="Tests"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setTestsOpen(!testsOpen);
                                                handleParentMenuClick(true);
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                testsOpen
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                        >
                                            <GraduationCap size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Tests</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            testsOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>
                                        {testsOpen && sidebarOpen && (
                                            <div className="ml-6 mt-1 pl-2 border-l border-indigo-500/50 dark:border-gray-600 space-y-1 animate-in slide-in-from-left duration-200">
                                                <Link
                                                    href="/admin/test-questions"
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <FileText size={16} />
                                                    <span>Test Questions</span>
                                                </Link>
                                                <Link
                                                    href="/admin/essay-topics"
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <BookOpen size={16} />
                                                    <span>Essay Topics</span>
                                                </Link>
                                                <Link
                                                    href="/admin/essays"
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <FileText size={16} />
                                                    <span>Essays</span>
                                                </Link>
                                                <Link
                                                    href="/admin/writers/tests"
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <Users size={16} />
                                                    <span>Writer Tests</span>
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}
                            {/* Preferences Dropdown */}
                            {filterMenuItems("Preferences") && (
                                <Tooltip
                                    content="Preferences"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setSettingsOpen(!settingsOpen);
                                                handleParentMenuClick(true);
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                settingsOpen
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                        >
                                            <Settings2 size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Preferences</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            settingsOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>
                                        {settingsOpen && sidebarOpen && (
                                            <div className="ml-6 mt-1 pl-2 border-l border-indigo-500/50 dark:border-gray-600 space-y-1 animate-in slide-in-from-left duration-200">
                                                <Link
                                                    href="/academic-levels"
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <span>Academic Levels</span>
                                                </Link>
                                                <Link
                                                    href="/subjects"
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <span>Subjects</span>
                                                </Link>
                                                <Link
                                                    href="/paper-types"
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <span>Paper Types</span>
                                                </Link>
                                                <Link
                                                    href="/pricings"
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <span>Urgencies</span>
                                                </Link>
                                                <Link
                                                    href={route("currencies.index")}
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <span>Currencies</span>
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}
                            {/* Settings Link */}
                            {filterMenuItems("Settings") && (
                                <Tooltip
                                    content="Settings"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href="/admin/settings/1"
                                        className={`flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("admin.settings.*")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={() =>
                                            handleParentMenuClick(false)
                                        }
                                    >
                                        <Settings size={20} />
                                        {sidebarOpen && <span>Settings</span>}
                                    </Link>
                                </Tooltip>
                            )}
                            {/* 3rd Party Scripts Link */}
                            {filterMenuItems("3rd Party Scripts") && (
                                <Tooltip
                                    content="3rd Party Scripts"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href={route("admin.third-party-scripts.index")}
                                        className={`flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("admin.third-party-scripts.*")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={() =>
                                            handleParentMenuClick(false)
                                        }
                                    >
                                        <Code size={20} />
                                        {sidebarOpen && <span>3rd Party Scripts</span>}
                                    </Link>
                                </Tooltip>
                            )}
                        </nav>
                        {/* Sidebar footer */}
                        {sidebarOpen && (
                            <div className="p-3 mt-auto border-t border-indigo-700/50 dark:border-gray-700">
                                <div className="flex items-center justify-between text-xs text-indigo-200 dark:text-gray-400">
                                    <span>v1.0.0</span>
                                    <button
                                        type="button"
                                        className="hover:text-white transition-colors duration-200 text-xs"
                                        onClick={() => setSidebarOpen(false)}
                                    >
                                        Collapse
                                    </button>
                                </div>
                            </div>
                        )}
                    </div>
                </aside>
                <main className="flex-1 p-6 bg-gradient-to-b from-blue-50 to-white dark:from-gray-900 dark:to-gray-800 dark:bg-none overflow-auto min-h-[calc(100vh-4rem)]">
                    <div className="max-w-7xl mx-auto">{children}</div>
                </main>
            </div>
            <MessagingWidget />
            <Toaster
                position="top-right"
                toastOptions={{
                    duration: 4000,
                    style: {
                        background: darkMode ? "#1F2937" : "#ffffff",
                        color: darkMode ? "#F9FAFB" : "#111827",
                        fontSize: "14px",
                        borderRadius: "8px",
                        padding: "12px 16px",
                        boxShadow: darkMode 
                            ? "0 4px 6px -1px rgba(0, 0, 0, 0.3)" 
                            : "0 4px 6px -1px rgba(0, 0, 0, 0.1)",
                    },
                    success: {
                        style: {
                            background: "#10B981",
                            color: "#ffffff",
                        },
                        iconTheme: {
                            primary: "#fff",
                            secondary: "#10B981",
                        },
                    },
                    error: {
                        style: {
                            background: "#EF4444",
                            color: "#ffffff",
                        },
                        iconTheme: {
                            primary: "#fff",
                            secondary: "#EF4444",
                        },
                    },
                }}
            />
        </div>
    );
}
