import { Head } from "@inertiajs/react";
import { useMemo } from "react";
import MainLayout from "@/Layouts/MainLayout";
import AboutHero from "@/Components/About/AboutHero";
import AboutStory from "@/Components/About/AboutStory";
import AboutAchievements from "@/Components/About/AboutAchievements";
import AboutValues from "@/Components/About/AboutValues";
import AboutServices from "@/Components/About/AboutServices";
import AboutWhyChoose from "@/Components/About/AboutWhyChoose";
import AboutProcess from "@/Components/About/AboutProcess";
import AboutTestimonials from "@/Components/About/AboutTestimonials";
import ServiceFAQs from "@/Components/Services/ServiceFAQs";
import PriceCalculatorSection from "@/Components/CMS/PriceCalculatorSection";

export default function About(props) {
    const { seoData, contentBlocks, structuredData, isCMS } = props;

    // Sort content blocks by order field for dynamic rendering
    const sortedBlocks = useMemo(() => {
        if (!isCMS || !contentBlocks) return [];
        return [...contentBlocks].sort((a, b) => (a.order || 0) - (b.order || 0));
    }, [contentBlocks, isCMS]);

    // Render individual block based on block_type
    const renderBlock = (block) => {
        const content = block.content;
        
        switch (block.block_type) {
            case 'about_hero':
                return <AboutHero key={block.id} hero={content || fallbackHero} />;
            case 'about_story':
                return <AboutStory key={block.id} title={content?.title} description={content?.description} image={content?.image} badge={content?.badge} linkText={content?.linkText} linkUrl={content?.linkUrl} />;
            case 'about_achievements':
                return <AboutAchievements key={block.id} title={content?.title || fallbackAchievements.title} description={content?.description || fallbackAchievements.description} achievements={content?.achievements || fallbackAchievements.achievements} />;
            case 'about_values':
                return <AboutValues key={block.id} title={content?.title} description={content?.description} values={content?.values || []} />;
            case 'about_services':
                return <AboutServices key={block.id} title={content?.title} description={content?.description} services={content?.services || []} ctaText={content?.ctaText} ctaLink={content?.ctaLink} />;
            case 'about_why_choose':
                return <AboutWhyChoose key={block.id} title={content?.title} description={content?.description} reasons={content?.reasons || []} />;
            case 'about_process':
                return <AboutProcess key={block.id} title={content?.title} description={content?.description} steps={content?.steps || []} />;
            case 'about_testimonials':
                return <AboutTestimonials key={block.id} title={content?.title} description={content?.description} testimonials={content?.testimonials || []} />;
            case 'about_faqs':
                return <ServiceFAQs key={block.id} title={content?.title} description={content?.description} faqs={content?.faqs || []} />;
            case 'about_cta':
                // CTA is handled separately, skip here
                return null;
            case 'price_calculator':
                return <PriceCalculatorSection key={block.id} content={content} />;
            default:
                return null;
        }
    };

    // Extract content from CMS blocks (for fallback compatibility)
    const getContentBlock = (blockKey) => {
        if (!isCMS || !contentBlocks) return null;
        const block = contentBlocks.find((b) => b.block_key === blockKey);
        return block ? block.content : null;
    };

    // Get all content blocks from CMS
    const hero = getContentBlock('about_hero');
    const story = getContentBlock('about_story');
    const achievements = getContentBlock('about_achievements');
    const values = getContentBlock('about_values');
    const services = getContentBlock('about_services');
    const whyChoose = getContentBlock('about_why_choose');
    const process = getContentBlock('about_process');
    const testimonials = getContentBlock('about_testimonials');
    const faqs = getContentBlock('about_faqs');
    const cta = getContentBlock('about_cta');

    // FALLBACK DATA (exact copy from original for SEO if CMS not available)
    const fallbackHero = {
        title: 'About Academic Scribe',
        description: 'Your trusted partner for academic success.<br/>Professional essay writing, research papers, and dissertation help from expert writers with advanced degrees.',
        primaryButtonText: 'Get Essay Help',
        primaryButtonLink: '/services/essay-writing',
        secondaryButtonText: 'Contact Us',
        secondaryButtonLink: '/contact',
    };

    const fallbackAchievements = {
        title: 'Our Achievements',
        description: 'Numbers that reflect our commitment to academic excellence',
        achievements: [
            { number: '50,000+', label: 'Papers Delivered', description: 'Successfully completed academic assignments' },
            { number: '15,000+', label: 'Happy Students', description: 'Satisfied customers worldwide' },
            { number: '500+', label: 'Expert Writers', description: 'PhD and Masters qualified professionals' },
            { number: '98%', label: 'Success Rate', description: 'Customer satisfaction guarantee' },
            { number: '24/7', label: 'Support Available', description: 'Round-the-clock customer assistance' },
            { number: '5+', label: 'Years Experience', description: 'Trusted academic writing service' },
        ]
    };

    const fallbackCta = {
        title: 'Ready to Get Professional Academic Help?',
        description: 'Join thousands of satisfied students who have achieved academic success with Academic Scribe. Get expert writing assistance tailored to your needs.',
        primaryButtonText: 'Order Now',
        primaryButtonLink: '/order',
        secondaryButtonText: 'Contact Us',
        secondaryButtonLink: '/contact',
        trustBadges: [
            '500+ Expert Writers',
            '50,000+ Papers Delivered',
            '98% Success Rate',
            '24/7 Support',
        ]
    };

    return (
        <MainLayout>
            <Head>
                <title>{seoData?.title || 'About Us | Professional Academic Writing Service | Academic Scribe'}</title>
                <meta name="description" content={seoData?.description || 'Learn about Academic Scribe - a trusted academic writing service with 500+ expert writers, 50,000+ papers delivered, and 98% customer satisfaction. Professional essay writing, research papers, and dissertation help.'} />
                <meta name="keywords" content={seoData?.keywords || 'about academic scribe, professional essay writing service, academic writing company, expert writers, custom essays, research paper help, dissertation writing, assignment assistance, academic support, online writing service'} />
                <link rel="canonical" href={seoData?.canonicalUrl || 'https://academicscribe.com/about'} />
                <meta property="og:title" content={seoData?.ogTitle || 'About Academic Scribe | Professional Academic Writing Service'} />
                <meta property="og:description" content={seoData?.ogDescription || 'Discover Academic Scribe - your trusted partner for academic success. 500+ expert writers, 50,000+ papers delivered, 98% satisfaction rate. Professional essay writing and research assistance.'} />
                <meta property="og:type" content="website" />
                <meta property="og:url" content={seoData?.ogUrl || 'https://academicscribe.com/about'} />
                <meta property="og:image" content={seoData?.ogImage || 'https://academicscribe.com/images/about-academic-scribe.jpg'} />
                <meta name="twitter:card" content="summary_large_image" />
                <meta name="twitter:title" content={seoData?.twitterTitle || 'About Academic Scribe | Professional Academic Writing Service'} />
                <meta name="twitter:description" content={seoData?.twitterDescription || 'Trusted academic writing service with expert writers, guaranteed quality, and 24/7 support. Professional essay writing and research assistance.'} />
                <meta name="twitter:image" content={seoData?.twitterImage || 'https://academicscribe.com/images/about-academic-scribe.jpg'} />
                
                {/* Structured Data */}
                {structuredData && (
                    <script type="application/ld+json">
                        {JSON.stringify(structuredData)}
                    </script>
                )}
            </Head>

            <div className="about-page bg-white dark:bg-gray-900">
                {isCMS && sortedBlocks.length > 0 ? (
                    sortedBlocks.map((block) => renderBlock(block))
                ) : (
                    <>
                        {/* Fallback: Static sections */}
                        <AboutHero hero={fallbackHero} />
                        <AboutAchievements 
                            title={fallbackAchievements.title}
                            description={fallbackAchievements.description}
                            achievements={fallbackAchievements.achievements}
                        />
                    </>
                )}

                {/* Final CTA Section */}
                <section className="py-16 bg-gradient-to-r from-blue-600 to-purple-600 dark:from-blue-800 dark:to-purple-900">
                    <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                        <h2 className="text-3xl md:text-4xl font-bold text-white mb-4">
                            {cta?.title || fallbackCta.title}
                        </h2>
                        <p className="text-xl text-blue-100 dark:text-blue-200 mb-8 max-w-2xl mx-auto">
                            {cta?.description || fallbackCta.description}
                        </p>

                        <div className="flex flex-col sm:flex-row gap-4 justify-center mb-8">
                            <a
                                href={cta?.primaryButtonLink || fallbackCta.primaryButtonLink}
                                className="bg-white text-blue-600 hover:bg-gray-100 dark:bg-gray-100 dark:hover:bg-white px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                {cta?.primaryButtonText || fallbackCta.primaryButtonText}
                            </a>
                            <a
                                href={cta?.secondaryButtonLink || fallbackCta.secondaryButtonLink}
                                className="border-2 border-white text-white hover:bg-white hover:text-blue-600 dark:hover:bg-gray-100 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                            >
                                {cta?.secondaryButtonText || fallbackCta.secondaryButtonText}
                            </a>
                        </div>

                        {/* Trust Indicators */}
                        <div className="flex flex-wrap justify-center items-center gap-8 text-blue-100 dark:text-blue-200">
                            {(cta?.trustBadges || fallbackCta.trustBadges).map((badge, index) => (
                                <div key={index} className="flex items-center">
                                    <svg
                                        className="w-5 h-5 mr-2"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                    <span className="text-sm">{badge}</span>
                                </div>
                            ))}
                        </div>
                    </div>
                </section>

                {/* Mobile Bottom CTA */}
                <div className="fixed bottom-0 left-0 right-0 bg-white dark:bg-gray-900 border-t border-gray-200 dark:border-gray-700 p-4 lg:hidden z-50">
                    <div className="flex gap-3">
                        <a
                            href={cta?.primaryButtonLink || fallbackCta.primaryButtonLink}
                            className="flex-1 bg-blue-600 hover:bg-blue-700 dark:bg-blue-700 dark:hover:bg-blue-800 text-white text-center py-3 px-4 rounded-lg font-semibold transition-colors"
                        >
                            {cta?.primaryButtonText || fallbackCta.primaryButtonText}
                        </a>
                        <a
                            href={cta?.secondaryButtonLink || fallbackCta.secondaryButtonLink}
                            className="flex-1 border border-blue-600 dark:border-blue-400 text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/20 text-center py-3 px-4 rounded-lg font-semibold transition-colors"
                        >
                            {cta?.secondaryButtonText || fallbackCta.secondaryButtonText}
                        </a>
                    </div>
                </div>
            </div>
        </MainLayout>
    );
}
