import React, { useState, useCallback, useMemo } from "react";
import { Head, Link, router, useForm } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Plus,
    Edit,
    Trash2,
    Search,
    Filter,
    MoreHorizontal,
    X,
    Check,
    AlertCircle,
    FolderOpen,
    Tag,
    Eye,
    FileText,
    Settings,
    ArrowUpDown,
    ArrowUp,
    ArrowDown,
    CheckCircle,
    XCircle,
    Loader2,
    BookOpen,
    Archive,
    RefreshCw,
    ExternalLink,
    Users,
    TrendingUp,
    BarChart3,
    Zap,
    Star,
    Heart,
    MessageCircle,
    Share2,
    Award,
    Target,
    Briefcase,
    GraduationCap,
    Code,
    Database,
    Lightbulb,
    Shield,
    Smartphone,
    Headphones,
    Camera,
    Mic,
    Video,
    Music,
    Gamepad,
    Plane,
    Car,
    Home,
    ShoppingBag,
    CreditCard,
    Banknote,
    PieChart,
    LineChart,
    Activity,
    Wifi,
    Battery,
    Bluetooth,
    Cpu,
    HardDrive,
    Smartphone as Phone,
    Tablet,
    Laptop,
    MonitorCheck,
    Server,
    Cloud,
    Lock,
    Key,
    Mail,
    Phone as PhoneIcon,
    MapPin,
    Navigation,
    Compass,
    Globe as GlobeIcon,
    Sun,
    Moon,
    CloudRain,
    CloudSnow,
    Wind,
    Thermometer,
    Zap as Lightning,
} from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";

// Available icons for categories
const CATEGORY_ICONS = {
    // General
    BookOpen,
    FileText,
    FolderOpen,
    Tag,
    Archive,
    Settings,
    // Education
    GraduationCap,
    Award,
    Target,
    Lightbulb,
    // Business
    Briefcase,
    Users,
    TrendingUp,
    BarChart3,
    PieChart,
    LineChart,
    // Technology
    Code,
    Database,
    Cpu,
    HardDrive,
    Server,
    Cloud,
    Wifi,
    // Devices
    Smartphone,
    Tablet,
    Laptop,
    MonitorCheck,

    // Communication
    Mail,
    PhoneIcon,
    MessageCircle,
    Share2,
    Headphones,
    // Media
    Video,
    Music,
    Mic,
    Camera,
    Gamepad,
    // Shopping
    ShoppingBag,
    CreditCard,
    Banknote,
    // Travel
    Plane,
    Car,
    MapPin,
    Navigation,
    Compass,
    GlobeIcon,
    // Weather
    Sun,
    Moon,
    CloudRain,
    CloudSnow,
    Wind,
    Thermometer,
    // Security
    Shield,
    Lock,
    Key,
    // Other
    Heart,
    Star,
    Zap,
    Activity,
    Battery,
    Bluetooth,
    Home,
};

// Available colors for categories
const CATEGORY_COLORS = [
    { name: "Blue", value: "#3B82F6" },
    { name: "Green", value: "#10B981" },
    { name: "Purple", value: "#8B5CF6" },
    { name: "Red", value: "#EF4444" },
    { name: "Yellow", value: "#F59E0B" },
    { name: "Pink", value: "#EC4899" },
    { name: "Indigo", value: "#6366F1" },
    { name: "Teal", value: "#14B8A6" },
    { name: "Orange", value: "#F97316" },
    { name: "Gray", value: "#6B7280" },
    { name: "Cyan", value: "#06B6D4" },
    { name: "Emerald", value: "#059669" },
    { name: "Violet", value: "#7C3AED" },
    { name: "Rose", value: "#F43F5E" },
    { name: "Amber", value: "#D97706" },
    { name: "Lime", value: "#65A30D" },
    { name: "Sky", value: "#0EA5E9" },
    { name: "Fuchsia", value: "#C026D3" },
    { name: "Slate", value: "#475569" },
    { name: "Zinc", value: "#52525B" },
];

export default function BlogCategories({ auth, categories, flash }) {
    // State management
    const [searchQuery, setSearchQuery] = useState("");
    const [sortBy, setSortBy] = useState("name");
    const [sortOrder, setSortOrder] = useState("asc");
    const [selectedCategories, setSelectedCategories] = useState([]);
    const [showCreateModal, setShowCreateModal] = useState(false);
    const [showEditModal, setShowEditModal] = useState(false);
    const [showDeleteModal, setShowDeleteModal] = useState(false);
    const [editingCategory, setEditingCategory] = useState(null);
    const [deletingCategory, setDeletingCategory] = useState(null);
    const [loading, setLoading] = useState(false);
    const [actionLoading, setActionLoading] = useState({});

    // Form for creating/editing categories
    const { data, setData, post, patch, processing, errors, reset } = useForm({
        name: "",
        slug: "",
        description: "",
        color: "#3B82F6",
        icon: "BookOpen",
        meta_title: "",
        meta_description: "",
    });

    // Animation variants
    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const itemVariants = {
        hidden: { opacity: 0, y: 20 },
        visible: {
            opacity: 1,
            y: 0,
            transition: {
                duration: 0.5,
            },
        },
    };

    const modalVariants = {
        hidden: { opacity: 0, scale: 0.95 },
        visible: { opacity: 1, scale: 1 },
        exit: { opacity: 0, scale: 0.95 },
    };

    // Filter and sort categories
    const filteredCategories = useMemo(() => {
        let filtered = categories || [];

        // Search filter
        if (searchQuery) {
            filtered = filtered.filter(
                (category) =>
                    category.name
                        .toLowerCase()
                        .includes(searchQuery.toLowerCase()) ||
                    category.description
                        .toLowerCase()
                        .includes(searchQuery.toLowerCase())
            );
        }

        // Sort categories
        filtered.sort((a, b) => {
            let aValue = a[sortBy];
            let bValue = b[sortBy];

            if (sortBy === "posts_count") {
                aValue = a.posts_count || 0;
                bValue = b.posts_count || 0;
            }

            if (sortOrder === "asc") {
                return aValue > bValue ? 1 : -1;
            } else {
                return aValue < bValue ? 1 : -1;
            }
        });

        return filtered;
    }, [categories, searchQuery, sortBy, sortOrder]);

    // Handle sort
    const handleSort = useCallback(
        (field) => {
            if (sortBy === field) {
                setSortOrder(sortOrder === "asc" ? "desc" : "asc");
            } else {
                setSortBy(field);
                setSortOrder("asc");
            }
        },
        [sortBy, sortOrder]
    );

    // Generate slug from name
    const generateSlug = useCallback((name) => {
        return name
            .toLowerCase()
            .replace(/[^a-z0-9\s-]/g, "")
            .replace(/\s+/g, "-")
            .replace(/-+/g, "-")
            .trim();
    }, []);

    // Handle name change and auto-generate slug
    const handleNameChange = useCallback(
        (name) => {
            setData("name", name);
            if (!editingCategory) {
                setData("slug", generateSlug(name));
            }
        },
        [editingCategory, generateSlug, setData]
    );

    // Handle create category
    const handleCreateCategory = useCallback(() => {
        post("/admin/blog/categories", {
            onSuccess: () => {
                setShowCreateModal(false);
                reset();
            },
        });
    }, [post, reset]);

    // Handle edit category
    const handleEditCategory = useCallback(
        (category) => {
            setEditingCategory(category);
            setData({
                name: category.name,
                slug: category.slug,
                description: category.description,
                color: category.color,
                icon: category.icon,
                meta_title: category.meta_title || "",
                meta_description: category.meta_description || "",
            });
            setShowEditModal(true);
        },
        [setData]
    );

    // Handle update category
    const handleUpdateCategory = useCallback(() => {
        if (!editingCategory) return;

        patch(`/admin/blog/categories/${editingCategory.id}`, {
            onSuccess: () => {
                setShowEditModal(false);
                setEditingCategory(null);
                reset();
            },
        });
    }, [editingCategory, patch, reset]);

    // Handle delete category
    const handleDeleteCategory = useCallback((category) => {
        setDeletingCategory(category);
        setShowDeleteModal(true);
    }, []);

    // Confirm delete category
    const confirmDeleteCategory = useCallback(() => {
        if (!deletingCategory) return;

        setActionLoading((prev) => ({ ...prev, [deletingCategory.id]: true }));

        router.delete(`/admin/blog/categories/${deletingCategory.id}`, {
            onSuccess: () => {
                setShowDeleteModal(false);
                setDeletingCategory(null);
            },
            onFinish: () => {
                setActionLoading((prev) => ({
                    ...prev,
                    [deletingCategory.id]: false,
                }));
            },
        });
    }, [deletingCategory]);

    // Handle category selection
    const handleCategorySelect = useCallback((categoryId) => {
        setSelectedCategories((prev) =>
            prev.includes(categoryId)
                ? prev.filter((id) => id !== categoryId)
                : [...prev, categoryId]
        );
    }, []);

    // Handle select all
    const handleSelectAll = useCallback(() => {
        if (selectedCategories.length === filteredCategories.length) {
            setSelectedCategories([]);
        } else {
            setSelectedCategories(
                filteredCategories.map((category) => category.id)
            );
        }
    }, [selectedCategories, filteredCategories]);

    // Get sort icon
    const getSortIcon = (field) => {
        if (sortBy !== field) return <ArrowUpDown className="w-4 h-4" />;
        return sortOrder === "asc" ? (
            <ArrowUp className="w-4 h-4" />
        ) : (
            <ArrowDown className="w-4 h-4" />
        );
    };

    // Get icon component
    const getIconComponent = (iconName) => {
        const IconComponent = CATEGORY_ICONS[iconName] || BookOpen;
        return <IconComponent className="w-5 h-5" />;
    };

    // Get category stats
    const getCategoryStats = () => {
        const totalCategories = categories?.length || 0;
        const totalPosts =
            categories?.reduce((sum, cat) => sum + (cat.posts_count || 0), 0) ||
            0;
        const avgPostsPerCategory =
            totalCategories > 0 ? Math.round(totalPosts / totalCategories) : 0;
        const mostUsedCategory = categories?.reduce(
            (prev, current) =>
                (current.posts_count || 0) > (prev.posts_count || 0)
                    ? current
                    : prev,
            categories[0]
        );

        return {
            totalCategories,
            totalPosts,
            avgPostsPerCategory,
            mostUsedCategory,
        };
    };

    const stats = getCategoryStats();

    return (
        <AdminLayout>
            <Head title="Blog Categories" />

            <motion.div
                variants={containerVariants}
                initial="hidden"
                animate="visible"
                className="max-w-7xl mx-auto space-y-6"
            >
                {/* Header */}
                <motion.div
                    variants={itemVariants}
                    className="flex items-center justify-between"
                >
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                            Blog Categories
                        </h1>
                        <p className="text-gray-600 dark:text-gray-400 mt-1">
                            Organize your blog content with categories
                        </p>
                    </div>

                    <div className="flex items-center gap-3">
                        <button
                            onClick={() => setShowCreateModal(true)}
                            className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                        >
                            <Plus className="w-4 h-4" />
                            <span>New Category</span>
                        </button>
                    </div>
                </motion.div>

                {/* Stats Cards */}
                <motion.div
                    variants={itemVariants}
                    className="grid grid-cols-1 md:grid-cols-4 gap-4"
                >
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Total Categories
                                </p>
                                <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                    {stats.totalCategories}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center">
                                <FolderOpen className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Total Posts
                                </p>
                                <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                                    {stats.totalPosts}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-green-100 dark:bg-green-900/30 rounded-lg flex items-center justify-center">
                                <FileText className="w-6 h-6 text-green-600 dark:text-green-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Avg Posts/Category
                                </p>
                                <p className="text-2xl font-bold text-purple-600 dark:text-purple-400">
                                    {stats.avgPostsPerCategory}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-purple-100 dark:bg-purple-900/30 rounded-lg flex items-center justify-center">
                                <BarChart3 className="w-6 h-6 text-purple-600 dark:text-purple-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Most Used
                                </p>
                                <p className="text-lg font-bold text-orange-600 dark:text-orange-400">
                                    {stats.mostUsedCategory?.name || "None"}
                                </p>
                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                    {stats.mostUsedCategory?.posts_count || 0}{" "}
                                    posts
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-orange-100 dark:bg-orange-900/30 rounded-lg flex items-center justify-center">
                                <TrendingUp className="w-6 h-6 text-orange-600 dark:text-orange-400" />
                            </div>
                        </div>
                    </div>
                </motion.div>

                {/* Search and Controls */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700"
                >
                    <div className="flex flex-col lg:flex-row gap-4 items-start lg:items-center justify-between">
                        <div className="flex-1 max-w-md">
                            <div className="relative">
                                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
                                <input
                                    type="text"
                                    placeholder="Search categories..."
                                    value={searchQuery}
                                    onChange={(e) =>
                                        setSearchQuery(e.target.value)
                                    }
                                    className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400"
                                />
                            </div>
                        </div>

                        <div className="flex items-center gap-3">
                            <div className="text-sm text-gray-600 dark:text-gray-400">
                                {filteredCategories.length} of{" "}
                                {categories?.length || 0} categories
                            </div>
                        </div>
                    </div>
                </motion.div>

                {/* Bulk Actions */}
                <AnimatePresence>
                    {selectedCategories.length > 0 && (
                        <motion.div
                            initial={{ opacity: 0, y: -10 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -10 }}
                            className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4"
                        >
                            <div className="flex items-center justify-between">
                                <div className="flex items-center gap-3">
                                    <span className="text-sm text-blue-700 dark:text-blue-300">
                                        {selectedCategories.length} categor
                                        {selectedCategories.length !== 1
                                            ? "ies"
                                            : "y"}{" "}
                                        selected
                                    </span>
                                    <button
                                        onClick={() =>
                                            setSelectedCategories([])
                                        }
                                        className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-200"
                                    >
                                        Clear selection
                                    </button>
                                </div>

                                <div className="flex items-center gap-2">
                                    <button
                                        onClick={() => {
                                            // Implement bulk delete
                                            if (
                                                confirm(
                                                    `Are you sure you want to delete ${selectedCategories.length} categories?`
                                                )
                                            ) {
                                                // Add bulk delete logic here
                                            }
                                        }}
                                        className="px-3 py-1 bg-red-600 hover:bg-red-700 text-white rounded-lg text-sm transition-colors"
                                    >
                                        Delete Selected
                                    </button>
                                </div>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>

                {/* Categories Table */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden"
                >
                    <div className="overflow-x-auto">
                        <table className="w-full">
                            <thead className="bg-gray-50 dark:bg-gray-900 border-b border-gray-200 dark:border-gray-700">
                                <tr>
                                    <th className="w-12 px-6 py-3 text-left">
                                        <input
                                            type="checkbox"
                                            checked={
                                                selectedCategories.length ===
                                                filteredCategories.length
                                            }
                                            onChange={handleSelectAll}
                                            className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                        />
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                        <button
                                            onClick={() => handleSort("name")}
                                            className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                        >
                                            Category
                                            {getSortIcon("name")}
                                        </button>
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                        Description
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                        <button
                                            onClick={() =>
                                                handleSort("posts_count")
                                            }
                                            className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                        >
                                            Posts
                                            {getSortIcon("posts_count")}
                                        </button>
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                        <button
                                            onClick={() =>
                                                handleSort("created_at")
                                            }
                                            className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                        >
                                            Created
                                            {getSortIcon("created_at")}
                                        </button>
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                        Actions
                                    </th>
                                </tr>
                            </thead>
                            <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                {filteredCategories.map((category, index) => (
                                    <motion.tr
                                        key={category.id}
                                        initial={{ opacity: 0, y: 20 }}
                                        animate={{ opacity: 1, y: 0 }}
                                        transition={{ delay: index * 0.05 }}
                                        className="hover:bg-gray-50 dark:hover:bg-gray-900 transition-colors"
                                    >
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <input
                                                type="checkbox"
                                                checked={selectedCategories.includes(
                                                    category.id
                                                )}
                                                onChange={() =>
                                                    handleCategorySelect(
                                                        category.id
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="flex items-center">
                                                <div className="flex-shrink-0 mr-3">
                                                    <div
                                                        className="w-8 h-8 rounded-lg flex items-center justify-center"
                                                        style={{
                                                            backgroundColor:
                                                                category.color +
                                                                "20",
                                                        }}
                                                    >
                                                        <div
                                                            style={{
                                                                color: category.color,
                                                            }}
                                                        >
                                                            {getIconComponent(
                                                                category.icon
                                                            )}
                                                        </div>
                                                    </div>
                                                </div>
                                                <div>
                                                    <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                        {category.name}
                                                    </div>
                                                    <div className="text-sm text-gray-500 dark:text-gray-400">
                                                        /{category.slug}
                                                    </div>
                                                </div>
                                            </div>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className="text-sm text-gray-900 dark:text-white max-w-xs truncate">
                                                {category.description}
                                            </div>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="flex items-center">
                                                <span className="text-sm text-gray-900 dark:text-white">
                                                    {category.posts_count || 0}
                                                </span>
                                                {category.posts_count > 0 && (
                                                    <Link
                                                        href={`/admin/blog/posts?category=${category.slug}`}
                                                        className="ml-2 text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                                    >
                                                        <Eye className="w-4 h-4" />
                                                    </Link>
                                                )}
                                            </div>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                            {new Date(
                                                category.created_at
                                            ).toLocaleDateString()}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                            <div className="flex items-center gap-2">
                                                <Link
                                                    href={`/blog/category/${category.slug}`}
                                                    target="_blank"
                                                    className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                                >
                                                    <ExternalLink className="w-4 h-4" />
                                                </Link>
                                                <button
                                                    onClick={() =>
                                                        handleEditCategory(
                                                            category
                                                        )
                                                    }
                                                    className="text-yellow-600 hover:text-yellow-800 dark:text-yellow-400 dark:hover:text-yellow-200"
                                                >
                                                    <Edit className="w-4 h-4" />
                                                </button>
                                                <button
                                                    onClick={() =>
                                                        handleDeleteCategory(
                                                            category
                                                        )
                                                    }
                                                    disabled={
                                                        actionLoading[
                                                            category.id
                                                        ]
                                                    }
                                                    className="text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-200 disabled:opacity-50"
                                                >
                                                    {actionLoading[
                                                        category.id
                                                    ] ? (
                                                        <Loader2 className="w-4 h-4 animate-spin" />
                                                    ) : (
                                                        <Trash2 className="w-4 h-4" />
                                                    )}
                                                </button>
                                            </div>
                                        </td>
                                    </motion.tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </motion.div>

                {/* Empty State */}
                {filteredCategories.length === 0 && (
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-12 text-center"
                    >
                        <div className="w-16 h-16 mx-auto mb-4 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center">
                            <FolderOpen className="w-8 h-8 text-gray-400" />
                        </div>
                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                            No categories found
                        </h3>
                        <p className="text-gray-500 dark:text-gray-400 mb-6">
                            {searchQuery
                                ? "Try adjusting your search criteria."
                                : "Get started by creating your first category."}
                        </p>
                        <div className="flex items-center justify-center gap-3">
                            {searchQuery ? (
                                <button
                                    onClick={() => setSearchQuery("")}
                                    className="flex items-center gap-2 px-4 py-2 text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                >
                                    <RefreshCw className="w-4 h-4" />
                                    Clear search
                                </button>
                            ) : (
                                <button
                                    onClick={() => setShowCreateModal(true)}
                                    className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                                >
                                    <Plus className="w-4 h-4" />
                                    Create Category
                                </button>
                            )}
                        </div>
                    </motion.div>
                )}

                {/* Create Category Modal */}
                <AnimatePresence>
                    {showCreateModal && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                variants={modalVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto"
                            >
                                <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center justify-between">
                                        <h2 className="text-xl font-semibold text-gray-900 dark:text-white">
                                            Create New Category
                                        </h2>
                                        <button
                                            onClick={() => {
                                                setShowCreateModal(false);
                                                reset();
                                            }}
                                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                        >
                                            <X className="w-6 h-6" />
                                        </button>
                                    </div>
                                </div>

                                <form
                                    onSubmit={(e) => {
                                        e.preventDefault();
                                        handleCreateCategory();
                                    }}
                                    className="p-6 space-y-6"
                                >
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Category Name *
                                            </label>
                                            <input
                                                type="text"
                                                value={data.name}
                                                onChange={(e) =>
                                                    handleNameChange(
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Enter category name"
                                                required
                                            />
                                            {errors.name && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.name}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Slug *
                                            </label>
                                            <input
                                                type="text"
                                                value={data.slug}
                                                onChange={(e) =>
                                                    setData(
                                                        "slug",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="category-slug"
                                                required
                                            />
                                            {errors.slug && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.slug}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Description *
                                        </label>
                                        <textarea
                                            value={data.description}
                                            onChange={(e) =>
                                                setData(
                                                    "description",
                                                    e.target.value
                                                )
                                            }
                                            rows={3}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            placeholder="Enter category description"
                                            required
                                        />
                                        {errors.description && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.description}
                                            </p>
                                        )}
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Color
                                            </label>
                                            <div className="grid grid-cols-5 gap-2 mb-2">
                                                {CATEGORY_COLORS.map(
                                                    (color) => (
                                                        <button
                                                            key={color.value}
                                                            type="button"
                                                            onClick={() =>
                                                                setData(
                                                                    "color",
                                                                    color.value
                                                                )
                                                            }
                                                            className={`w-8 h-8 rounded-full border-2 ${
                                                                data.color ===
                                                                color.value
                                                                    ? "border-gray-900 dark:border-white"
                                                                    : "border-gray-300 dark:border-gray-600"
                                                            }`}
                                                            style={{
                                                                backgroundColor:
                                                                    color.value,
                                                            }}
                                                            title={color.name}
                                                        />
                                                    )
                                                )}
                                            </div>
                                            <input
                                                type="color"
                                                value={data.color}
                                                onChange={(e) =>
                                                    setData(
                                                        "color",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full h-10 border border-gray-300 dark:border-gray-600 rounded-lg"
                                            />
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Icon
                                            </label>
                                            <select
                                                value={data.icon}
                                                onChange={(e) =>
                                                    setData(
                                                        "icon",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            >
                                                {Object.keys(
                                                    CATEGORY_ICONS
                                                ).map((iconName) => (
                                                    <option
                                                        key={iconName}
                                                        value={iconName}
                                                    >
                                                        {iconName}
                                                    </option>
                                                ))}
                                            </select>
                                            <div className="mt-2 flex items-center gap-2">
                                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                                    Preview:
                                                </span>
                                                <div
                                                    className="w-8 h-8 rounded-lg flex items-center justify-center"
                                                    style={{
                                                        backgroundColor:
                                                            data.color + "20",
                                                    }}
                                                >
                                                    <div
                                                        style={{
                                                            color: data.color,
                                                        }}
                                                    >
                                                        {getIconComponent(
                                                            data.icon
                                                        )}
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            SEO Title
                                        </label>
                                        <input
                                            type="text"
                                            value={data.meta_title}
                                            onChange={(e) =>
                                                setData(
                                                    "meta_title",
                                                    e.target.value
                                                )
                                            }
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            placeholder="SEO title for this category"
                                        />
                                        {errors.meta_title && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.meta_title}
                                            </p>
                                        )}
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            SEO Description
                                        </label>
                                        <textarea
                                            value={data.meta_description}
                                            onChange={(e) =>
                                                setData(
                                                    "meta_description",
                                                    e.target.value
                                                )
                                            }
                                            rows={3}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            placeholder="SEO description for this category"
                                        />
                                        {errors.meta_description && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.meta_description}
                                            </p>
                                        )}
                                    </div>

                                    <div className="flex items-center justify-end gap-3 pt-4 border-t border-gray-200 dark:border-gray-700">
                                        <button
                                            type="button"
                                            onClick={() => {
                                                setShowCreateModal(false);
                                                reset();
                                            }}
                                            className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            type="submit"
                                            disabled={processing}
                                            className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors disabled:opacity-50"
                                        >
                                            {processing ? (
                                                <>
                                                    <Loader2 className="w-4 h-4 animate-spin" />
                                                    Creating...
                                                </>
                                            ) : (
                                                <>
                                                    <Check className="w-4 h-4" />
                                                    Create Category
                                                </>
                                            )}
                                        </button>
                                    </div>
                                </form>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>

                {/* Edit Category Modal */}
                <AnimatePresence>
                    {showEditModal && editingCategory && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                variants={modalVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto"
                            >
                                <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center justify-between">
                                        <h2 className="text-xl font-semibold text-gray-900 dark:text-white">
                                            Edit Category
                                        </h2>
                                        <button
                                            onClick={() => {
                                                setShowEditModal(false);
                                                setEditingCategory(null);
                                                reset();
                                            }}
                                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                        >
                                            <X className="w-6 h-6" />
                                        </button>
                                    </div>
                                </div>

                                <form
                                    onSubmit={(e) => {
                                        e.preventDefault();
                                        handleUpdateCategory();
                                    }}
                                    className="p-6 space-y-6"
                                >
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Category Name *
                                            </label>
                                            <input
                                                type="text"
                                                value={data.name}
                                                onChange={(e) =>
                                                    setData(
                                                        "name",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Enter category name"
                                                required
                                            />
                                            {errors.name && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.name}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Slug *
                                            </label>
                                            <input
                                                type="text"
                                                value={data.slug}
                                                onChange={(e) =>
                                                    setData(
                                                        "slug",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="category-slug"
                                                required
                                            />
                                            {errors.slug && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.slug}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Description *
                                        </label>
                                        <textarea
                                            value={data.description}
                                            onChange={(e) =>
                                                setData(
                                                    "description",
                                                    e.target.value
                                                )
                                            }
                                            rows={3}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            placeholder="Enter category description"
                                            required
                                        />
                                        {errors.description && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.description}
                                            </p>
                                        )}
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Color
                                            </label>
                                            <div className="grid grid-cols-5 gap-2 mb-2">
                                                {CATEGORY_COLORS.map(
                                                    (color) => (
                                                        <button
                                                            key={color.value}
                                                            type="button"
                                                            onClick={() =>
                                                                setData(
                                                                    "color",
                                                                    color.value
                                                                )
                                                            }
                                                            className={`w-8 h-8 rounded-full border-2 ${
                                                                data.color ===
                                                                color.value
                                                                    ? "border-gray-900 dark:border-white"
                                                                    : "border-gray-300 dark:border-gray-600"
                                                            }`}
                                                            style={{
                                                                backgroundColor:
                                                                    color.value,
                                                            }}
                                                            title={color.name}
                                                        />
                                                    )
                                                )}
                                            </div>
                                            <input
                                                type="color"
                                                value={data.color}
                                                onChange={(e) =>
                                                    setData(
                                                        "color",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full h-10 border border-gray-300 dark:border-gray-600 rounded-lg"
                                            />
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Icon
                                            </label>
                                            <select
                                                value={data.icon}
                                                onChange={(e) =>
                                                    setData(
                                                        "icon",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            >
                                                {Object.keys(
                                                    CATEGORY_ICONS
                                                ).map((iconName) => (
                                                    <option
                                                        key={iconName}
                                                        value={iconName}
                                                    >
                                                        {iconName}
                                                    </option>
                                                ))}
                                            </select>
                                            <div className="mt-2 flex items-center gap-2">
                                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                                    Preview:
                                                </span>
                                                <div
                                                    className="w-8 h-8 rounded-lg flex items-center justify-center"
                                                    style={{
                                                        backgroundColor:
                                                            data.color + "20",
                                                    }}
                                                >
                                                    <div
                                                        style={{
                                                            color: data.color,
                                                        }}
                                                    >
                                                        {getIconComponent(
                                                            data.icon
                                                        )}
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            SEO Title
                                        </label>
                                        <input
                                            type="text"
                                            value={data.meta_title}
                                            onChange={(e) =>
                                                setData(
                                                    "meta_title",
                                                    e.target.value
                                                )
                                            }
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            placeholder="SEO title for this category"
                                        />
                                        {errors.meta_title && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.meta_title}
                                            </p>
                                        )}
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            SEO Description
                                        </label>
                                        <textarea
                                            value={data.meta_description}
                                            onChange={(e) =>
                                                setData(
                                                    "meta_description",
                                                    e.target.value
                                                )
                                            }
                                            rows={3}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            placeholder="SEO description for this category"
                                        />
                                        {errors.meta_description && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.meta_description}
                                            </p>
                                        )}
                                    </div>

                                    <div className="flex items-center justify-end gap-3 pt-4 border-t border-gray-200 dark:border-gray-700">
                                        <button
                                            type="button"
                                            onClick={() => {
                                                setShowEditModal(false);
                                                setEditingCategory(null);
                                                reset();
                                            }}
                                            className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            type="submit"
                                            disabled={processing}
                                            className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors disabled:opacity-50"
                                        >
                                            {processing ? (
                                                <>
                                                    <Loader2 className="w-4 h-4 animate-spin" />
                                                    Updating...
                                                </>
                                            ) : (
                                                <>
                                                    <Check className="w-4 h-4" />
                                                    Update Category
                                                </>
                                            )}
                                        </button>
                                    </div>
                                </form>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>

                {/* Delete Category Modal */}
                <AnimatePresence>
                    {showDeleteModal && deletingCategory && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                variants={modalVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full"
                            >
                                <div className="p-6">
                                    <div className="flex items-center justify-center w-12 h-12 mx-auto mb-4 bg-red-100 dark:bg-red-900/30 rounded-full">
                                        <AlertCircle className="w-6 h-6 text-red-600 dark:text-red-400" />
                                    </div>
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white text-center mb-2">
                                        Delete Category
                                    </h3>
                                    <p className="text-gray-500 dark:text-gray-400 text-center mb-6">
                                        Are you sure you want to delete the
                                        category "{deletingCategory.name}"? This
                                        action cannot be undone.
                                    </p>
                                    {deletingCategory.posts_count > 0 && (
                                        <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4 mb-4">
                                            <div className="flex items-center">
                                                <AlertCircle className="w-5 h-5 text-yellow-600 dark:text-yellow-400 mr-2" />
                                                <span className="text-sm text-yellow-700 dark:text-yellow-300">
                                                    This category has{" "}
                                                    {
                                                        deletingCategory.posts_count
                                                    }{" "}
                                                    post(s). They will be moved
                                                    to "Uncategorized".
                                                </span>
                                            </div>
                                        </div>
                                    )}
                                    <div className="flex items-center justify-center gap-3">
                                        <button
                                            onClick={() => {
                                                setShowDeleteModal(false);
                                                setDeletingCategory(null);
                                            }}
                                            className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            onClick={confirmDeleteCategory}
                                            disabled={
                                                actionLoading[
                                                    deletingCategory.id
                                                ]
                                            }
                                            className="flex items-center gap-2 px-4 py-2 bg-red-600 hover:bg-red-700 dark:bg-red-500 dark:hover:bg-red-600 text-white rounded-lg transition-colors disabled:opacity-50"
                                        >
                                            {actionLoading[
                                                deletingCategory.id
                                            ] ? (
                                                <>
                                                    <Loader2 className="w-4 h-4 animate-spin" />
                                                    Deleting...
                                                </>
                                            ) : (
                                                <>
                                                    <Trash2 className="w-4 h-4" />
                                                    Delete Category
                                                </>
                                            )}
                                        </button>
                                    </div>
                                </div>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>

                {/* Flash Messages */}
                <AnimatePresence>
                    {flash?.success && (
                        <motion.div
                            initial={{ opacity: 0, y: 50 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -50 }}
                            className="fixed bottom-4 right-4 bg-green-500 text-white px-6 py-3 rounded-lg shadow-lg z-50"
                        >
                            <div className="flex items-center gap-2">
                                <CheckCircle className="w-5 h-5" />
                                <span>{flash.success}</span>
                            </div>
                        </motion.div>
                    )}
                    {flash?.error && (
                        <motion.div
                            initial={{ opacity: 0, y: 50 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -50 }}
                            className="fixed bottom-4 right-4 bg-red-500 text-white px-6 py-3 rounded-lg shadow-lg z-50"
                        >
                            <div className="flex items-center gap-2">
                                <XCircle className="w-5 h-5" />
                                <span>{flash.error}</span>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>
            </motion.div>
        </AdminLayout>
    );
}
