import React, { useState, useEffect } from "react";
import { Head, Link, router } from "@inertiajs/react";
import { motion } from "framer-motion";
import {
    FileText,
    FolderOpen,
    Tags,
    UserCheck,
    Upload,
    Eye,
    Edit,
    Trash2,
    TrendingUp,
    Calendar,
    Clock,
    BarChart3,
    Plus,
    Search,
    Filter,
    RefreshCw,
    Download,
    Settings,
    Activity,
    Users,
    MessageCircle,
    Heart,
    Share2,
    ExternalLink,
    AlertCircle,
    CheckCircle,
    XCircle,
    Loader2,
} from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";

export default function BlogDashboard({
    stats,
    recentPosts,
    recentUploads,
    popularCategories,
    topAuthors,
    analytics,
    auth,
}) {
    const [dateRange, setDateRange] = useState("7d");
    const [refreshing, setRefreshing] = useState(false);
    const [searchTerm, setSearchTerm] = useState("");

    // Animation variants
    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const itemVariants = {
        hidden: { opacity: 0, y: 20 },
        visible: {
            opacity: 1,
            y: 0,
            transition: {
                duration: 0.5,
            },
        },
    };

    // Handle refresh
    const handleRefresh = () => {
        setRefreshing(true);
        router.reload({
            only: ["stats", "recentPosts", "recentUploads", "analytics"],
            onFinish: () => setRefreshing(false),
        });
    };

    // Format date
    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString("en-US", {
            year: "numeric",
            month: "short",
            day: "numeric",
        });
    };

    // Format number
    const formatNumber = (num) => {
        if (num >= 1000000) {
            return (num / 1000000).toFixed(1) + "M";
        } else if (num >= 1000) {
            return (num / 1000).toFixed(1) + "K";
        }
        return num.toString();
    };

    // Get status color
    const getStatusColor = (status) => {
        switch (status) {
            case "published":
                return "text-green-600 bg-green-100 dark:text-green-400 dark:bg-green-900/30";
            case "draft":
                return "text-yellow-600 bg-yellow-100 dark:text-yellow-400 dark:bg-yellow-900/30";
            case "archived":
                return "text-gray-600 bg-gray-100 dark:text-gray-400 dark:bg-gray-900/30";
            default:
                return "text-gray-600 bg-gray-100 dark:text-gray-400 dark:bg-gray-900/30";
        }
    };

    return (
        <AdminLayout>
            <Head title="Blog Dashboard" />

            <motion.div
                variants={containerVariants}
                initial="hidden"
                animate="visible"
                className="space-y-6"
            >
                {/* Header */}
                <motion.div
                    variants={itemVariants}
                    className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
                >
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                            Blog Dashboard
                        </h1>
                        <p className="text-gray-600 dark:text-gray-400 mt-1">
                            Manage your blog content and monitor performance
                        </p>
                    </div>

                    <div className="flex items-center gap-3">
                        <button
                            onClick={handleRefresh}
                            disabled={refreshing}
                            className="flex items-center gap-2 px-4 py-2 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                        >
                            <RefreshCw
                                className={`w-4 h-4 ${
                                    refreshing ? "animate-spin" : ""
                                }`}
                            />
                            <span>Refresh</span>
                        </button>

                        <Link
                            href="/admin/blog/upload"
                            className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                        >
                            <Upload className="w-4 h-4" />
                            <span>Upload Files</span>
                        </Link>
                    </div>
                </motion.div>

                {/* Stats Cards */}
                <motion.div
                    variants={itemVariants}
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6"
                >
                    {/* Total Posts */}
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm font-medium text-gray-600 dark:text-gray-400">
                                    Total Posts
                                </p>
                                <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                    {formatNumber(stats?.total_posts || 0)}
                                </p>
                                <p className="text-xs text-green-600 dark:text-green-400 mt-1">
                                    +{stats?.posts_this_week || 0} this week
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center">
                                <FileText className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                            </div>
                        </div>
                    </div>

                    {/* Categories */}
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm font-medium text-gray-600 dark:text-gray-400">
                                    Categories
                                </p>
                                <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                    {stats?.total_categories || 0}
                                </p>
                                <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    {stats?.active_categories || 0} active
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-green-100 dark:bg-green-900/30 rounded-lg flex items-center justify-center">
                                <FolderOpen className="w-6 h-6 text-green-600 dark:text-green-400" />
                            </div>
                        </div>
                    </div>

                    {/* Tags */}
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm font-medium text-gray-600 dark:text-gray-400">
                                    Tags
                                </p>
                                <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                    {stats?.total_tags || 0}
                                </p>
                                <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    {stats?.popular_tags || 0} popular
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-purple-100 dark:bg-purple-900/30 rounded-lg flex items-center justify-center">
                                <Tags className="w-6 h-6 text-purple-600 dark:text-purple-400" />
                            </div>
                        </div>
                    </div>

                    {/* Authors */}
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm font-medium text-gray-600 dark:text-gray-400">
                                    Authors
                                </p>
                                <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                    {stats?.total_authors || 0}
                                </p>
                                <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    {stats?.active_authors || 0} active
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-orange-100 dark:bg-orange-900/30 rounded-lg flex items-center justify-center">
                                <UserCheck className="w-6 h-6 text-orange-600 dark:text-orange-400" />
                            </div>
                        </div>
                    </div>
                </motion.div>

                {/* Analytics and Quick Actions */}
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Analytics Chart */}
                    <motion.div
                        variants={itemVariants}
                        className="lg:col-span-2 bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                    >
                        <div className="flex items-center justify-between mb-6">
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                Content Performance
                            </h3>
                            <div className="flex items-center gap-2">
                                <select
                                    value={dateRange}
                                    onChange={(e) =>
                                        setDateRange(e.target.value)
                                    }
                                    className="text-sm border border-gray-300 dark:border-gray-600 rounded-md px-3 py-1 bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                >
                                    <option value="7d">Last 7 days</option>
                                    <option value="30d">Last 30 days</option>
                                    <option value="90d">Last 90 days</option>
                                </select>
                            </div>
                        </div>

                        {/* Simple Analytics Display */}
                        <div className="space-y-4">
                            <div className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                <div className="flex items-center gap-3">
                                    <Eye className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                                    <span className="text-sm font-medium text-gray-900 dark:text-white">
                                        Total Views
                                    </span>
                                </div>
                                <span className="text-xl font-bold text-gray-900 dark:text-white">
                                    {formatNumber(analytics?.total_views || 0)}
                                </span>
                            </div>

                            <div className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                <div className="flex items-center gap-3">
                                    <Users className="w-5 h-5 text-green-600 dark:text-green-400" />
                                    <span className="text-sm font-medium text-gray-900 dark:text-white">
                                        Unique Visitors
                                    </span>
                                </div>
                                <span className="text-xl font-bold text-gray-900 dark:text-white">
                                    {formatNumber(
                                        analytics?.unique_visitors || 0
                                    )}
                                </span>
                            </div>

                            <div className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                <div className="flex items-center gap-3">
                                    <Clock className="w-5 h-5 text-purple-600 dark:text-purple-400" />
                                    <span className="text-sm font-medium text-gray-900 dark:text-white">
                                        Avg. Read Time
                                    </span>
                                </div>
                                <span className="text-xl font-bold text-gray-900 dark:text-white">
                                    {analytics?.avg_read_time || "0m"}
                                </span>
                            </div>
                        </div>
                    </motion.div>

                    {/* Quick Actions */}
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                    >
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                            Quick Actions
                        </h3>

                        <div className="space-y-3">
                            <Link
                                href="/admin/blog/upload"
                                className="flex items-center gap-3 w-full p-3 text-left bg-blue-50 dark:bg-blue-900/20 hover:bg-blue-100 dark:hover:bg-blue-900/30 rounded-lg transition-colors"
                            >
                                <Upload className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                    Upload Blog Files
                                </span>
                            </Link>

                            <Link
                                href="/admin/blog/posts"
                                className="flex items-center gap-3 w-full p-3 text-left bg-gray-50 dark:bg-gray-900 hover:bg-gray-100 dark:hover:bg-gray-800 rounded-lg transition-colors"
                            >
                                <FileText className="w-5 h-5 text-gray-600 dark:text-gray-400" />
                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                    View All Posts
                                </span>
                            </Link>

                            <Link
                                href="/admin/blog/categories"
                                className="flex items-center gap-3 w-full p-3 text-left bg-gray-50 dark:bg-gray-900 hover:bg-gray-100 dark:hover:bg-gray-800 rounded-lg transition-colors"
                            >
                                <FolderOpen className="w-5 h-5 text-gray-600 dark:text-gray-400" />
                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                    Manage Categories
                                </span>
                            </Link>

                            <Link
                                href="/admin/blog/tags"
                                className="flex items-center gap-3 w-full p-3 text-left bg-gray-50 dark:bg-gray-900 hover:bg-gray-100 dark:hover:bg-gray-800 rounded-lg transition-colors"
                            >
                                <Tags className="w-5 h-5 text-gray-600 dark:text-gray-400" />
                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                    Manage Tags
                                </span>
                            </Link>

                            <Link
                                href="/admin/blog/authors"
                                className="flex items-center gap-3 w-full p-3 text-left bg-gray-50 dark:bg-gray-900 hover:bg-gray-100 dark:hover:bg-gray-800 rounded-lg transition-colors"
                            >
                                <UserCheck className="w-5 h-5 text-gray-600 dark:text-gray-400" />
                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                    Manage Authors
                                </span>
                            </Link>
                        </div>
                    </motion.div>
                </div>

                {/* Recent Posts and Popular Categories */}
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {/* Recent Posts */}
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                    >
                        <div className="flex items-center justify-between mb-6">
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                Recent Posts
                            </h3>
                            <Link
                                href="/admin/blog/posts"
                                className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 font-medium"
                            >
                                View all
                            </Link>
                        </div>

                        <div className="space-y-4">
                            {recentPosts && recentPosts.length > 0 ? (
                                recentPosts.map((post, index) => (
                                    <div
                                        key={index}
                                        className="flex items-start gap-4 p-4 bg-gray-50 dark:bg-gray-900 rounded-lg"
                                    >
                                        <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center flex-shrink-0">
                                            <FileText className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                                        </div>
                                        <div className="flex-1 min-w-0">
                                            <h4 className="text-sm font-medium text-gray-900 dark:text-white truncate">
                                                {post.title}
                                            </h4>
                                            <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                {post.author?.name} •{" "}
                                                {formatDate(post.published_at)}
                                            </p>
                                            <div className="flex items-center gap-2 mt-2">
                                                <span
                                                    className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(
                                                        post.status
                                                    )}`}
                                                >
                                                    {post.status}
                                                </span>
                                                <span className="text-xs text-gray-500 dark:text-gray-400">
                                                    {post.category?.name}
                                                </span>
                                            </div>
                                        </div>
                                        <div className="flex items-center gap-1">
                                            <Link
                                                href={`/admin/blog/posts/${post.id}/edit`}
                                                className="p-1 text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                            >
                                                <Edit className="w-4 h-4" />
                                            </Link>
                                            <button className="p-1 text-gray-400 hover:text-green-600 dark:hover:text-green-400 transition-colors">
                                                <Eye className="w-4 h-4" />
                                            </button>
                                        </div>
                                    </div>
                                ))
                            ) : (
                                <div className="text-center py-8">
                                    <FileText className="w-12 h-12 text-gray-400 mx-auto mb-3" />
                                    <p className="text-gray-500 dark:text-gray-400">
                                        No posts yet. Upload some blog files to
                                        get started!
                                    </p>
                                </div>
                            )}
                        </div>
                    </motion.div>

                    {/* Popular Categories */}
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                    >
                        <div className="flex items-center justify-between mb-6">
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                Popular Categories
                            </h3>
                            <Link
                                href="/admin/blog/categories"
                                className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 font-medium"
                            >
                                Manage
                            </Link>
                        </div>

                        <div className="space-y-4">
                            {popularCategories &&
                            popularCategories.length > 0 ? (
                                popularCategories.map((category, index) => (
                                    <div
                                        key={index}
                                        className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg"
                                    >
                                        <div className="flex items-center gap-3">
                                            <div
                                                className={`w-3 h-3 rounded-full`}
                                                style={{
                                                    backgroundColor:
                                                        category.color ||
                                                        "#3B82F6",
                                                }}
                                            ></div>
                                            <div>
                                                <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                                                    {category.name}
                                                </h4>
                                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                                    {category.posts_count} posts
                                                </p>
                                            </div>
                                        </div>
                                        <div className="text-right">
                                            <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                {category.posts_count}
                                            </div>
                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                posts
                                            </div>
                                        </div>
                                    </div>
                                ))
                            ) : (
                                <div className="text-center py-8">
                                    <FolderOpen className="w-12 h-12 text-gray-400 mx-auto mb-3" />
                                    <p className="text-gray-500 dark:text-gray-400">
                                        No categories found.
                                    </p>
                                </div>
                            )}
                        </div>
                    </motion.div>
                </div>

                {/* Recent Uploads and Top Authors */}
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {/* Recent Uploads */}
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                    >
                        <div className="flex items-center justify-between mb-6">
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                Recent Uploads
                            </h3>
                            <Link
                                href="/admin/blog/upload"
                                className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 font-medium"
                            >
                                Upload Files
                            </Link>
                        </div>

                        <div className="space-y-4">
                            {recentUploads && recentUploads.length > 0 ? (
                                recentUploads.map((upload, index) => (
                                    <div
                                        key={index}
                                        className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg"
                                    >
                                        <div className="flex items-center gap-3">
                                            <div className="w-10 h-10 bg-green-100 dark:bg-green-900/30 rounded-lg flex items-center justify-center">
                                                <Upload className="w-5 h-5 text-green-600 dark:text-green-400" />
                                            </div>
                                            <div>
                                                <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                                                    {upload.files_count} files
                                                    uploaded
                                                </h4>
                                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                                    {formatDate(
                                                        upload.created_at
                                                    )}
                                                </p>
                                            </div>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            <span
                                                className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                                                    upload.status ===
                                                    "completed"
                                                        ? "text-green-600 bg-green-100 dark:text-green-400 dark:bg-green-900/30"
                                                        : upload.status ===
                                                          "processing"
                                                        ? "text-yellow-600 bg-yellow-100 dark:text-yellow-400 dark:bg-yellow-900/30"
                                                        : "text-red-600 bg-red-100 dark:text-red-400 dark:bg-red-900/30"
                                                }`}
                                            >
                                                {upload.status}
                                            </span>
                                        </div>
                                    </div>
                                ))
                            ) : (
                                <div className="text-center py-8">
                                    <Upload className="w-12 h-12 text-gray-400 mx-auto mb-3" />
                                    <p className="text-gray-500 dark:text-gray-400">
                                        No recent uploads.
                                    </p>
                                </div>
                            )}
                        </div>
                    </motion.div>

                    {/* Top Authors */}
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                    >
                        <div className="flex items-center justify-between mb-6">
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                Top Authors
                            </h3>
                            <Link
                                href="/admin/blog/authors"
                                className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 font-medium"
                            >
                                View all
                            </Link>
                        </div>

                        <div className="space-y-4">
                            {topAuthors && topAuthors.length > 0 ? (
                                topAuthors.map((author, index) => (
                                    <div
                                        key={index}
                                        className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg"
                                    >
                                        <div className="flex items-center gap-3">
                                            <div className="w-10 h-10 bg-orange-100 dark:bg-orange-900/30 rounded-lg flex items-center justify-center">
                                                <UserCheck className="w-5 h-5 text-orange-600 dark:text-orange-400" />
                                            </div>
                                            <div>
                                                <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                                                    {author.name}
                                                </h4>
                                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                                    {author.email}
                                                </p>
                                            </div>
                                        </div>
                                        <div className="text-right">
                                            <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                {author.posts_count}
                                            </div>
                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                posts
                                            </div>
                                        </div>
                                    </div>
                                ))
                            ) : (
                                <div className="text-center py-8">
                                    <UserCheck className="w-12 h-12 text-gray-400 mx-auto mb-3" />
                                    <p className="text-gray-500 dark:text-gray-400">
                                        No authors found.
                                    </p>
                                </div>
                            )}
                        </div>
                    </motion.div>
                </div>

                {/* Performance Metrics */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                >
                    <div className="flex items-center justify-between mb-6">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                            Performance Metrics
                        </h3>
                        <div className="flex items-center gap-2">
                            <button className="p-2 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 rounded-lg transition-colors">
                                <Settings className="w-4 h-4" />
                            </button>
                            <button className="p-2 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 rounded-lg transition-colors">
                                <Download className="w-4 h-4" />
                            </button>
                        </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                        <div className="text-center">
                            <div className="w-16 h-16 bg-blue-100 dark:bg-blue-900/30 rounded-full flex items-center justify-center mx-auto mb-3">
                                <TrendingUp className="w-8 h-8 text-blue-600 dark:text-blue-400" />
                            </div>
                            <div className="text-2xl font-bold text-gray-900 dark:text-white">
                                {analytics?.engagement_rate || "0%"}
                            </div>
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                Engagement Rate
                            </div>
                        </div>

                        <div className="text-center">
                            <div className="w-16 h-16 bg-green-100 dark:bg-green-900/30 rounded-full flex items-center justify-center mx-auto mb-3">
                                <Share2 className="w-8 h-8 text-green-600 dark:text-green-400" />
                            </div>
                            <div className="text-2xl font-bold text-gray-900 dark:text-white">
                                {formatNumber(analytics?.total_shares || 0)}
                            </div>
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                Social Shares
                            </div>
                        </div>

                        <div className="text-center">
                            <div className="w-16 h-16 bg-purple-100 dark:bg-purple-900/30 rounded-full flex items-center justify-center mx-auto mb-3">
                                <MessageCircle className="w-8 h-8 text-purple-600 dark:text-purple-400" />
                            </div>
                            <div className="text-2xl font-bold text-gray-900 dark:text-white">
                                {formatNumber(analytics?.total_comments || 0)}
                            </div>
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                Comments
                            </div>
                        </div>

                        <div className="text-center">
                            <div className="w-16 h-16 bg-red-100 dark:bg-red-900/30 rounded-full flex items-center justify-center mx-auto mb-3">
                                <Heart className="w-8 h-8 text-red-600 dark:text-red-400" />
                            </div>
                            <div className="text-2xl font-bold text-gray-900 dark:text-white">
                                {formatNumber(analytics?.total_likes || 0)}
                            </div>
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                Likes
                            </div>
                        </div>
                    </div>
                </motion.div>

                {/* System Status */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                >
                    <div className="flex items-center justify-between mb-6">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                            System Status
                        </h3>
                        <div className="flex items-center gap-2">
                            <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                            <span className="text-sm text-gray-600 dark:text-gray-400">
                                All systems operational
                            </span>
                        </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div className="flex items-center gap-3 p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                            <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400" />
                            <div>
                                <div className="text-sm font-medium text-gray-900 dark:text-white">
                                    File Upload System
                                </div>
                                <div className="text-xs text-gray-500 dark:text-gray-400">
                                    Operational
                                </div>
                            </div>
                        </div>

                        <div className="flex items-center gap-3 p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                            <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400" />
                            <div>
                                <div className="text-sm font-medium text-gray-900 dark:text-white">
                                    Content Processing
                                </div>
                                <div className="text-xs text-gray-500 dark:text-gray-400">
                                    Operational
                                </div>
                            </div>
                        </div>

                        <div className="flex items-center gap-3 p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                            <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400" />
                            <div>
                                <div className="text-sm font-medium text-gray-900 dark:text-white">
                                    Database Import
                                </div>
                                <div className="text-xs text-gray-500 dark:text-gray-400">
                                    Operational
                                </div>
                            </div>
                        </div>
                    </div>
                </motion.div>
            </motion.div>
        </AdminLayout>
    );
}
