import React, { useState, useMemo } from "react";
import { Head, router } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Plus,
    Edit,
    Trash2,
    Search,
    MoreVertical,
    Tag,
    Hash,
    FileText,
    CheckCircle,
    AlertCircle,
    X,
} from "lucide-react";

export default function BlogTags({ auth, tags, flash }) {
    // Add safety checks
    const safeTags = tags || [];

    // State management
    const [searchQuery, setSearchQuery] = useState("");
    const [selectedTags, setSelectedTags] = useState([]);
    const [showCreateModal, setShowCreateModal] = useState(false);
    const [showEditModal, setShowEditModal] = useState(false);
    const [editingTag, setEditingTag] = useState(null);
    const [loading, setLoading] = useState(false);
    const [formData, setFormData] = useState({
        name: "",
        slug: "",
    });

    // Filter tags based on search
    const filteredTags = useMemo(() => {
        if (!searchQuery) return safeTags;

        return safeTags.filter(
            (tag) =>
                tag.name?.toLowerCase().includes(searchQuery.toLowerCase()) ||
                tag.slug?.toLowerCase().includes(searchQuery.toLowerCase())
        );
    }, [safeTags, searchQuery]);

    // Handle form submission
    const handleSubmit = async (e) => {
        e.preventDefault();
        setLoading(true);

        try {
            if (editingTag) {
                router.put(
                    route("admin.blog.tags.update", editingTag.id),
                    formData,
                    {
                        onSuccess: () => {
                            setShowEditModal(false);
                            setEditingTag(null);
                            resetForm();
                        },
                        onFinish: () => setLoading(false),
                    }
                );
            } else {
                router.post(route("admin.blog.tags.store"), formData, {
                    onSuccess: () => {
                        setShowCreateModal(false);
                        resetForm();
                    },
                    onFinish: () => setLoading(false),
                });
            }
        } catch (error) {
            console.error("Error submitting form:", error);
            setLoading(false);
        }
    };

    // Reset form
    const resetForm = () => {
        setFormData({
            name: "",
            slug: "",
        });
    };

    // Handle edit
    const handleEdit = (tag) => {
        setEditingTag(tag);
        setFormData({
            name: tag.name || "",
            slug: tag.slug || "",
        });
        setShowEditModal(true);
    };

    // Handle delete
    const handleDelete = (tag) => {
        if (confirm(`Are you sure you want to delete the tag "${tag.name}"?`)) {
            router.delete(route("admin.blog.tags.delete", tag.id));
        }
    };

    // Generate slug from name
    const generateSlug = (name) => {
        return name
            .toLowerCase()
            .replace(/[^a-z0-9]+/g, "-")
            .replace(/^-+|-+$/g, "");
    };

    // Handle name change and auto-generate slug
    const handleNameChange = (e) => {
        const name = e.target.value;
        setFormData({
            ...formData,
            name,
            slug: generateSlug(name),
        });
    };

    // Animation variants
    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const itemVariants = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0 },
    };

    return (
        <AdminLayout user={auth.user}>
            <Head title="Blog Tags" />

            <motion.div
                variants={containerVariants}
                initial="hidden"
                animate="visible"
                className="space-y-6"
            >
                {/* Header */}
                <motion.div
                    variants={itemVariants}
                    className="flex items-center justify-between"
                >
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                            Blog Tags
                        </h1>
                        <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                            Manage your blog post tags
                        </p>
                    </div>

                    <button
                        onClick={() => setShowCreateModal(true)}
                        className="flex items-center px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg transition-colors"
                    >
                        <Plus className="w-4 h-4 mr-2" />
                        New Tag
                    </button>
                </motion.div>

                {/* Flash Messages */}
                {flash?.success && (
                    <motion.div
                        initial={{ opacity: 0, y: -10 }}
                        animate={{ opacity: 1, y: 0 }}
                        className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4"
                    >
                        <div className="flex items-center">
                            <CheckCircle className="w-5 h-5 text-green-500 mr-3" />
                            <p className="text-sm text-green-700 dark:text-green-300">
                                {flash.success}
                            </p>
                        </div>
                    </motion.div>
                )}

                {flash?.error && (
                    <motion.div
                        initial={{ opacity: 0, y: -10 }}
                        animate={{ opacity: 1, y: 0 }}
                        className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-4"
                    >
                        <div className="flex items-center">
                            <AlertCircle className="w-5 h-5 text-red-500 mr-3" />
                            <p className="text-sm text-red-700 dark:text-red-300">
                                {flash.error}
                            </p>
                        </div>
                    </motion.div>
                )}

                {/* Stats */}
                <motion.div
                    variants={itemVariants}
                    className="grid grid-cols-1 md:grid-cols-3 gap-4"
                >
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Total Tags
                                </p>
                                <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                    {safeTags.length}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center">
                                <Tag className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Active Tags
                                </p>
                                <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                                    {
                                        safeTags.filter(
                                            (tag) => tag.posts_count > 0
                                        ).length
                                    }
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-green-100 dark:bg-green-900/30 rounded-lg flex items-center justify-center">
                                <CheckCircle className="w-6 h-6 text-green-600 dark:text-green-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Unused Tags
                                </p>
                                <p className="text-2xl font-bold text-yellow-600 dark:text-yellow-400">
                                    {
                                        safeTags.filter(
                                            (tag) => tag.posts_count === 0
                                        ).length
                                    }
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-yellow-100 dark:bg-yellow-900/30 rounded-lg flex items-center justify-center">
                                <AlertCircle className="w-6 h-6 text-yellow-600 dark:text-yellow-400" />
                            </div>
                        </div>
                    </div>
                </motion.div>

                {/* Search */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700"
                >
                    <div className="relative">
                        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                        <input
                            type="text"
                            placeholder="Search tags..."
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                        />
                    </div>
                </motion.div>

                {/* Tags Grid */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden"
                >
                    {filteredTags.length > 0 ? (
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 p-6">
                            {filteredTags.map((tag) => (
                                <motion.div
                                    key={tag.id}
                                    layout
                                    className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg border border-gray-200 dark:border-gray-600 hover:shadow-md transition-shadow"
                                >
                                    <div className="flex items-start justify-between">
                                        <div className="flex-1">
                                            <div className="flex items-center space-x-2">
                                                <Hash className="w-4 h-4 text-gray-400" />
                                                <h3 className="font-medium text-gray-900 dark:text-white">
                                                    {tag.name}
                                                </h3>
                                            </div>
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                                Slug: {tag.slug}
                                            </p>
                                            <div className="flex items-center mt-2">
                                                <FileText className="w-4 h-4 text-gray-400 mr-1" />
                                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                                    {tag.posts_count || 0} posts
                                                </span>
                                            </div>
                                        </div>
                                        <div className="flex items-center space-x-2">
                                            <button
                                                onClick={() => handleEdit(tag)}
                                                className="p-1 text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                            >
                                                <Edit className="w-4 h-4" />
                                            </button>
                                            <button
                                                onClick={() =>
                                                    handleDelete(tag)
                                                }
                                                className="p-1 text-gray-400 hover:text-red-600 dark:hover:text-red-400 transition-colors"
                                            >
                                                <Trash2 className="w-4 h-4" />
                                            </button>
                                        </div>
                                    </div>
                                </motion.div>
                            ))}
                        </div>
                    ) : (
                        <div className="p-12 text-center">
                            <Tag className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                            <p className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                                No tags found
                            </p>
                            <p className="text-gray-500 dark:text-gray-400">
                                {searchQuery
                                    ? "Try adjusting your search"
                                    : "Create your first tag to get started"}
                            </p>
                        </div>
                    )}
                </motion.div>

                {/* Create Modal */}
                <AnimatePresence>
                    {showCreateModal && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                initial={{ opacity: 0, scale: 0.95 }}
                                animate={{ opacity: 1, scale: 1 }}
                                exit={{ opacity: 0, scale: 0.95 }}
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full"
                            >
                                <div className="p-6">
                                    <div className="flex items-center justify-between mb-6">
                                        <h2 className="text-lg font-semibold text-gray-900 dark:text-white">
                                            Create New Tag
                                        </h2>
                                        <button
                                            onClick={() =>
                                                setShowCreateModal(false)
                                            }
                                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                        >
                                            <X className="w-5 h-5" />
                                        </button>
                                    </div>

                                    <form
                                        onSubmit={handleSubmit}
                                        className="space-y-4"
                                    >
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Tag Name
                                            </label>
                                            <input
                                                type="text"
                                                value={formData.name}
                                                onChange={handleNameChange}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                                placeholder="Enter tag name"
                                                required
                                            />
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Slug
                                            </label>
                                            <input
                                                type="text"
                                                value={formData.slug}
                                                onChange={(e) =>
                                                    setFormData({
                                                        ...formData,
                                                        slug: e.target.value,
                                                    })
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                                placeholder="tag-slug"
                                                required
                                            />
                                            <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                Auto-generated from name, but
                                                you can customize it
                                            </p>
                                        </div>

                                        <div className="flex space-x-3 pt-4">
                                            <button
                                                type="button"
                                                onClick={() =>
                                                    setShowCreateModal(false)
                                                }
                                                className="flex-1 px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                            >
                                                Cancel
                                            </button>
                                            <button
                                                type="submit"
                                                disabled={loading}
                                                className="flex-1 px-4 py-2 text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 rounded-lg transition-colors disabled:opacity-50"
                                            >
                                                {loading
                                                    ? "Creating..."
                                                    : "Create Tag"}
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>

                {/* Edit Modal */}
                <AnimatePresence>
                    {showEditModal && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                initial={{ opacity: 0, scale: 0.95 }}
                                animate={{ opacity: 1, scale: 1 }}
                                exit={{ opacity: 0, scale: 0.95 }}
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full"
                            >
                                <div className="p-6">
                                    <div className="flex items-center justify-between mb-6">
                                        <h2 className="text-lg font-semibold text-gray-900 dark:text-white">
                                            Edit Tag
                                        </h2>
                                        <button
                                            onClick={() => {
                                                setShowEditModal(false);
                                                setEditingTag(null);
                                                resetForm();
                                            }}
                                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                        >
                                            <X className="w-5 h-5" />
                                        </button>
                                    </div>

                                    <form
                                        onSubmit={handleSubmit}
                                        className="space-y-4"
                                    >
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Tag Name
                                            </label>
                                            <input
                                                type="text"
                                                value={formData.name}
                                                onChange={handleNameChange}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                                placeholder="Enter tag name"
                                                required
                                            />
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Slug
                                            </label>
                                            <input
                                                type="text"
                                                value={formData.slug}
                                                onChange={(e) =>
                                                    setFormData({
                                                        ...formData,
                                                        slug: e.target.value,
                                                    })
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                                placeholder="tag-slug"
                                                required
                                            />
                                        </div>

                                        {editingTag && (
                                            <div className="bg-gray-50 dark:bg-gray-700 p-3 rounded-lg">
                                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                                    This tag is used in{" "}
                                                    {editingTag.posts_count ||
                                                        0}{" "}
                                                    posts
                                                </p>
                                            </div>
                                        )}

                                        <div className="flex space-x-3 pt-4">
                                            <button
                                                type="button"
                                                onClick={() => {
                                                    setShowEditModal(false);
                                                    setEditingTag(null);
                                                    resetForm();
                                                }}
                                                className="flex-1 px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                            >
                                                Cancel
                                            </button>
                                            <button
                                                type="submit"
                                                disabled={loading}
                                                className="flex-1 px-4 py-2 text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 rounded-lg transition-colors disabled:opacity-50"
                                            >
                                                {loading
                                                    ? "Updating..."
                                                    : "Update Tag"}
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>
            </motion.div>
        </AdminLayout>
    );
}
