import React, { useState, useEffect, useCallback } from "react";
import { Head, Link, router } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Upload,
    FileText,
    CheckCircle,
    AlertCircle,
    XCircle,
    Loader2,
    RefreshCw,
    Eye,
    Trash2,
    Download,
    Activity,
    Database,
    Settings,
    ArrowLeft,
    Info,
    PlayCircle,
    StopCircle,
    BarChart3,
    Clock,
    Users,
    FolderOpen,
    Tag,
    BookOpen,
    Plus,
    ExternalLink,
} from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";
import BlogFileUpload from "@/Components/Blog/BlogFileUpload";
import BlogFilePreview from "@/Components/Blog/BlogFilePreview";
import BlogUploadProgress from "@/Components/Blog/BlogUploadProgress";

export default function BlogUpload({ auth, flash }) {
    // State management
    const [currentStep, setCurrentStep] = useState(1); // 1: Upload, 2: Preview, 3: Process, 4: Complete
    const [uploadId, setUploadId] = useState(null);
    const [uploadedFiles, setUploadedFiles] = useState([]);
    const [selectedFile, setSelectedFile] = useState(null);
    const [showPreview, setShowPreview] = useState(false);
    const [processing, setProcessing] = useState(false);
    const [importing, setImporting] = useState(false);
    const [processResults, setProcessResults] = useState(null);
    const [importResults, setImportResults] = useState(null);
    const [errors, setErrors] = useState([]);
    const [warnings, setWarnings] = useState([]);
    const [showAdvancedOptions, setShowAdvancedOptions] = useState(false);
    const [autoProcess, setAutoProcess] = useState(true);
    const [autoImport, setAutoImport] = useState(true);
    const [processingOptions, setProcessingOptions] = useState({
        validateContent: true,
        generateThumbnails: false,
        optimizeImages: false,
        extractMetadata: true,
        createBackups: true,
    });

    // Animation variants
    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const itemVariants = {
        hidden: { opacity: 0, y: 20 },
        visible: {
            opacity: 1,
            y: 0,
            transition: {
                duration: 0.5,
            },
        },
    };

    const stepVariants = {
        hidden: { opacity: 0, x: 20 },
        visible: { opacity: 1, x: 0 },
        exit: { opacity: 0, x: -20 },
    };

    // Handle upload completion
    const handleUploadComplete = useCallback(
        (result) => {
            setUploadId(result.upload_id);
            setUploadedFiles(result.files);
            setCurrentStep(2);
            setErrors([]);
            setWarnings([]);

            // Auto-process if enabled
            if (autoProcess) {
                setTimeout(() => {
                    handleProcessFiles();
                }, 1000);
            }
        },
        [autoProcess]
    );

    // Handle upload error
    const handleUploadError = useCallback((error) => {
        setErrors([error.error || "Upload failed"]);
        console.error("Upload error:", error);
    }, []);

    // Handle file processing
    const handleProcessFiles = useCallback(async () => {
        if (!uploadId) return;

        setProcessing(true);
        setCurrentStep(3);
        setErrors([]);

        try {
            const response = await fetch("/blog/process/files", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": document
                        .querySelector('meta[name="csrf-token"]')
                        .getAttribute("content"),
                },
                body: JSON.stringify({
                    upload_id: uploadId,
                    options: processingOptions,
                }),
            });

            const result = await response.json();

            if (result.success) {
                setProcessResults(result);
                setWarnings(result.warnings || []);

                // Auto-import if enabled
                if (autoImport) {
                    setTimeout(() => {
                        handleImportFiles();
                    }, 1000);
                }
            } else {
                setErrors([result.error || "Processing failed"]);
            }
        } catch (error) {
            setErrors([error.message || "Processing failed"]);
            console.error("Processing error:", error);
        } finally {
            setProcessing(false);
        }
    }, [uploadId, processingOptions, autoImport]);

    // Handle file import
    const handleImportFiles = useCallback(async () => {
        if (!uploadId) return;

        setImporting(true);
        setErrors([]);

        try {
            const response = await fetch("/blog/process/import", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": document
                        .querySelector('meta[name="csrf-token"]')
                        .getAttribute("content"),
                },
                body: JSON.stringify({
                    upload_id: uploadId,
                }),
            });

            const result = await response.json();

            if (result.success) {
                setImportResults(result);
                setCurrentStep(4);
            } else {
                setErrors([result.error || "Import failed"]);
            }
        } catch (error) {
            setErrors([error.message || "Import failed"]);
            console.error("Import error:", error);
        } finally {
            setImporting(false);
        }
    }, [uploadId]);

    // Process and import in one step
    const handleProcessAndImport = useCallback(async () => {
        if (!uploadId) return;

        setProcessing(true);
        setCurrentStep(3);
        setErrors([]);

        try {
            const response = await fetch("/blog/process/process-and-import", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": document
                        .querySelector('meta[name="csrf-token"]')
                        .getAttribute("content"),
                },
                body: JSON.stringify({
                    upload_id: uploadId,
                    options: processingOptions,
                }),
            });

            const result = await response.json();

            if (result.success) {
                setProcessResults(result.data.processing);
                setImportResults(result.data.import);
                setCurrentStep(4);
            } else {
                setErrors([result.error || "Process and import failed"]);
            }
        } catch (error) {
            setErrors([error.message || "Process and import failed"]);
            console.error("Process and import error:", error);
        } finally {
            setProcessing(false);
        }
    }, [uploadId, processingOptions]);

    // Reset upload
    const handleReset = useCallback(() => {
        setCurrentStep(1);
        setUploadId(null);
        setUploadedFiles([]);
        setSelectedFile(null);
        setShowPreview(false);
        setProcessing(false);
        setImporting(false);
        setProcessResults(null);
        setImportResults(null);
        setErrors([]);
        setWarnings([]);
    }, []);

    // Preview file
    const handlePreviewFile = useCallback((file) => {
        setSelectedFile(file);
        setShowPreview(true);
    }, []);

    // Get step status
    const getStepStatus = (step) => {
        if (step < currentStep) return "complete";
        if (step === currentStep) return "active";
        return "pending";
    };

    // Get step icon
    const getStepIcon = (step, status) => {
        if (status === "complete") {
            return <CheckCircle className="w-5 h-5 text-green-600" />;
        }
        if (status === "active") {
            if (step === 3 && (processing || importing)) {
                return (
                    <Loader2 className="w-5 h-5 text-blue-600 animate-spin" />
                );
            }
            return (
                <div className="w-5 h-5 bg-blue-600 rounded-full flex items-center justify-center text-white text-xs font-bold">
                    {step}
                </div>
            );
        }
        return (
            <div className="w-5 h-5 bg-gray-300 rounded-full flex items-center justify-center text-gray-600 text-xs font-bold">
                {step}
            </div>
        );
    };

    return (
        <AdminLayout>
            <Head title="Blog Upload" />

            <motion.div
                variants={containerVariants}
                initial="hidden"
                animate="visible"
                className="max-w-6xl mx-auto space-y-6"
            >
                {/* Header */}
                <motion.div
                    variants={itemVariants}
                    className="flex items-center justify-between"
                >
                    <div className="flex items-center gap-4">
                        <Link
                            href="/admin/blog"
                            className="p-2 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 rounded-lg transition-colors"
                        >
                            <ArrowLeft className="w-5 h-5" />
                        </Link>
                        <div>
                            <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                Upload Blog Files
                            </h1>
                            <p className="text-gray-600 dark:text-gray-400 mt-1">
                                Upload JavaScript files containing blog post
                                data
                            </p>
                        </div>
                    </div>

                    <div className="flex items-center gap-3">
                        <button
                            onClick={() =>
                                setShowAdvancedOptions(!showAdvancedOptions)
                            }
                            className="flex items-center gap-2 px-4 py-2 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                        >
                            <Settings className="w-4 h-4" />
                            <span>Options</span>
                        </button>

                        {currentStep > 1 && (
                            <button
                                onClick={handleReset}
                                className="flex items-center gap-2 px-4 py-2 bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
                            >
                                <RefreshCw className="w-4 h-4" />
                                <span>Reset</span>
                            </button>
                        )}
                    </div>
                </motion.div>

                {/* Progress Steps */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                >
                    <div className="flex items-center justify-between">
                        {[
                            { step: 1, title: "Upload Files", icon: Upload },
                            { step: 2, title: "Preview & Validate", icon: Eye },
                            {
                                step: 3,
                                title: "Process & Import",
                                icon: Activity,
                            },
                            { step: 4, title: "Complete", icon: CheckCircle },
                        ].map((item, index) => (
                            <div key={item.step} className="flex items-center">
                                <div className="flex items-center gap-3">
                                    {getStepIcon(
                                        item.step,
                                        getStepStatus(item.step)
                                    )}
                                    <div>
                                        <div
                                            className={`text-sm font-medium ${
                                                getStepStatus(item.step) ===
                                                "active"
                                                    ? "text-blue-600 dark:text-blue-400"
                                                    : getStepStatus(
                                                          item.step
                                                      ) === "complete"
                                                    ? "text-green-600 dark:text-green-400"
                                                    : "text-gray-500 dark:text-gray-400"
                                            }`}
                                        >
                                            {item.title}
                                        </div>
                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                            Step {item.step}
                                        </div>
                                    </div>
                                </div>
                                {index < 3 && (
                                    <div
                                        className={`w-16 h-0.5 mx-4 ${
                                            getStepStatus(item.step + 1) ===
                                                "complete" ||
                                            getStepStatus(item.step + 1) ===
                                                "active"
                                                ? "bg-blue-600"
                                                : "bg-gray-300"
                                        }`}
                                    />
                                )}
                            </div>
                        ))}
                    </div>
                </motion.div>

                {/* Advanced Options */}
                <AnimatePresence>
                    {showAdvancedOptions && (
                        <motion.div
                            initial={{ opacity: 0, height: 0 }}
                            animate={{ opacity: 1, height: "auto" }}
                            exit={{ opacity: 0, height: 0 }}
                            className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700"
                        >
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                Advanced Options
                            </h3>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-3">
                                        Automation Settings
                                    </h4>
                                    <div className="space-y-3">
                                        <label className="flex items-center gap-3">
                                            <input
                                                type="checkbox"
                                                checked={autoProcess}
                                                onChange={(e) =>
                                                    setAutoProcess(
                                                        e.target.checked
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <span className="text-sm text-gray-700 dark:text-gray-300">
                                                Auto-process files after upload
                                            </span>
                                        </label>

                                        <label className="flex items-center gap-3">
                                            <input
                                                type="checkbox"
                                                checked={autoImport}
                                                onChange={(e) =>
                                                    setAutoImport(
                                                        e.target.checked
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <span className="text-sm text-gray-700 dark:text-gray-300">
                                                Auto-import after processing
                                            </span>
                                        </label>
                                    </div>
                                </div>

                                <div>
                                    <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-3">
                                        Processing Options
                                    </h4>
                                    <div className="space-y-3">
                                        <label className="flex items-center gap-3">
                                            <input
                                                type="checkbox"
                                                checked={
                                                    processingOptions.validateContent
                                                }
                                                onChange={(e) =>
                                                    setProcessingOptions(
                                                        (prev) => ({
                                                            ...prev,
                                                            validateContent:
                                                                e.target
                                                                    .checked,
                                                        })
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <span className="text-sm text-gray-700 dark:text-gray-300">
                                                Validate content structure
                                            </span>
                                        </label>

                                        <label className="flex items-center gap-3">
                                            <input
                                                type="checkbox"
                                                checked={
                                                    processingOptions.extractMetadata
                                                }
                                                onChange={(e) =>
                                                    setProcessingOptions(
                                                        (prev) => ({
                                                            ...prev,
                                                            extractMetadata:
                                                                e.target
                                                                    .checked,
                                                        })
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <span className="text-sm text-gray-700 dark:text-gray-300">
                                                Extract metadata
                                            </span>
                                        </label>

                                        <label className="flex items-center gap-3">
                                            <input
                                                type="checkbox"
                                                checked={
                                                    processingOptions.createBackups
                                                }
                                                onChange={(e) =>
                                                    setProcessingOptions(
                                                        (prev) => ({
                                                            ...prev,
                                                            createBackups:
                                                                e.target
                                                                    .checked,
                                                        })
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <span className="text-sm text-gray-700 dark:text-gray-300">
                                                Create backups
                                            </span>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>

                {/* Error Display */}
                <AnimatePresence>
                    {errors.length > 0 && (
                        <motion.div
                            initial={{ opacity: 0, y: -10 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -10 }}
                            className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-4"
                        >
                            <div className="flex items-center mb-3">
                                <AlertCircle className="w-5 h-5 text-red-500 dark:text-red-400 mr-2" />
                                <h3 className="text-sm font-medium text-red-800 dark:text-red-200">
                                    Errors ({errors.length})
                                </h3>
                            </div>
                            <div className="space-y-1">
                                {errors.map((error, index) => (
                                    <div
                                        key={index}
                                        className="text-sm text-red-700 dark:text-red-300"
                                    >
                                        {error}
                                    </div>
                                ))}
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>

                {/* Warning Display */}
                <AnimatePresence>
                    {warnings.length > 0 && (
                        <motion.div
                            initial={{ opacity: 0, y: -10 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -10 }}
                            className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4"
                        >
                            <div className="flex items-center mb-3">
                                <AlertCircle className="w-5 h-5 text-yellow-500 dark:text-yellow-400 mr-2" />
                                <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                                    Warnings ({warnings.length})
                                </h3>
                            </div>
                            <div className="space-y-1">
                                {warnings.map((warning, index) => (
                                    <div
                                        key={index}
                                        className="text-sm text-yellow-700 dark:text-yellow-300"
                                    >
                                        {warning}
                                    </div>
                                ))}
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>

                {/* Main Content */}
                <div className="space-y-6">
                    {/* Step 1: Upload Files */}
                    <AnimatePresence mode="wait">
                        {currentStep === 1 && (
                            <motion.div
                                key="upload"
                                variants={stepVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-6"
                            >
                                <div className="mb-6">
                                    <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                                        Upload Blog Files
                                    </h2>
                                    <p className="text-gray-600 dark:text-gray-400">
                                        Select JavaScript files containing your
                                        blog post data. Files will be validated
                                        before processing.
                                    </p>
                                </div>

                                <BlogFileUpload
                                    onUploadComplete={handleUploadComplete}
                                    onUploadError={handleUploadError}
                                    maxFiles={20}
                                    maxFileSize={5 * 1024 * 1024} // 5MB
                                />

                                {/* Upload Instructions */}
                                <div className="mt-6 p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
                                    <div className="flex items-start gap-3">
                                        <Info className="w-5 h-5 text-blue-600 dark:text-blue-400 mt-0.5" />
                                        <div>
                                            <h3 className="text-sm font-medium text-blue-900 dark:text-blue-300 mb-2">
                                                File Format Requirements
                                            </h3>
                                            <ul className="text-sm text-blue-700 dark:text-blue-300 space-y-1">
                                                <li>
                                                    • Files must be JavaScript
                                                    (.js) format
                                                </li>
                                                <li>
                                                    • Each file should export a
                                                    default object with blog
                                                    post data
                                                </li>
                                                <li>
                                                    • Required fields: title,
                                                    slug, excerpt, content,
                                                    author, category
                                                </li>
                                                <li>
                                                    • Optional fields: tags,
                                                    meta, featured image,
                                                    publish date
                                                </li>
                                                <li>
                                                    • Files are validated before
                                                    processing
                                                </li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </motion.div>
                        )}

                        {/* Step 2: Preview & Validate */}
                        {currentStep === 2 && (
                            <motion.div
                                key="preview"
                                variants={stepVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-6"
                            >
                                <div className="mb-6">
                                    <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                                        Preview & Validate Files
                                    </h2>
                                    <p className="text-gray-600 dark:text-gray-400">
                                        Review uploaded files before processing.
                                        Click on any file to preview its
                                        content.
                                    </p>
                                </div>

                                {/* File List */}
                                <div className="space-y-3 mb-6">
                                    {uploadedFiles.map((file, index) => (
                                        <div
                                            key={index}
                                            className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg"
                                        >
                                            <div className="flex items-center gap-3">
                                                <div className="w-10 h-10 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center">
                                                    <FileText className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                                                </div>
                                                <div>
                                                    <h3 className="text-sm font-medium text-gray-900 dark:text-white">
                                                        {file.original_name}
                                                    </h3>
                                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                                        {(
                                                            file.file_size /
                                                            1024
                                                        ).toFixed(1)}{" "}
                                                        KB • {file.mime_type}
                                                    </p>
                                                </div>
                                            </div>

                                            <div className="flex items-center gap-2">
                                                <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400">
                                                    {file.status}
                                                </span>
                                                <button
                                                    onClick={() =>
                                                        handlePreviewFile(file)
                                                    }
                                                    className="p-2 text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                                >
                                                    <Eye className="w-4 h-4" />
                                                </button>
                                            </div>
                                        </div>
                                    ))}
                                </div>

                                {/* Action Buttons */}
                                <div className="flex items-center justify-between pt-4 border-t border-gray-200 dark:border-gray-700">
                                    <button
                                        onClick={handleReset}
                                        className="flex items-center gap-2 px-4 py-2 text-gray-700 dark:text-gray-300 bg-gray-100 dark:bg-gray-700 rounded-lg hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
                                    >
                                        <ArrowLeft className="w-4 h-4" />
                                        <span>Back to Upload</span>
                                    </button>

                                    <div className="flex items-center gap-3">
                                        <button
                                            onClick={handleProcessFiles}
                                            disabled={processing}
                                            className="flex items-center gap-2 px-6 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors disabled:opacity-50"
                                        >
                                            {processing ? (
                                                <Loader2 className="w-4 h-4 animate-spin" />
                                            ) : (
                                                <PlayCircle className="w-4 h-4" />
                                            )}
                                            <span>Process Files</span>
                                        </button>

                                        <button
                                            onClick={handleProcessAndImport}
                                            disabled={processing}
                                            className="flex items-center gap-2 px-6 py-2 bg-green-600 hover:bg-green-700 dark:bg-green-500 dark:hover:bg-green-600 text-white rounded-lg transition-colors disabled:opacity-50"
                                        >
                                            {processing ? (
                                                <Loader2 className="w-4 h-4 animate-spin" />
                                            ) : (
                                                <Database className="w-4 h-4" />
                                            )}
                                            <span>Process & Import</span>
                                        </button>
                                    </div>
                                </div>
                            </motion.div>
                        )}

                        {/* Step 3: Processing */}
                        {currentStep === 3 && (
                            <motion.div
                                key="processing"
                                variants={stepVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="space-y-6"
                            >
                                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                                    <div className="mb-6">
                                        <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">
                                            Processing Files
                                        </h2>
                                        <p className="text-gray-600 dark:text-gray-400">
                                            Files are being processed and
                                            imported into the database. This may
                                            take a few minutes.
                                        </p>
                                    </div>

                                    {/* Progress Component */}
                                    <BlogUploadProgress
                                        uploadId={uploadId}
                                        onComplete={(result) => {
                                            setImportResults(result);
                                            setCurrentStep(4);
                                        }}
                                        onError={(error) => {
                                            setErrors([
                                                error.message ||
                                                    "Processing failed",
                                            ]);
                                        }}
                                        autoRefresh={true}
                                        refreshInterval={2000}
                                    />
                                </div>
                            </motion.div>
                        )}

                        {/* Step 4: Complete */}
                        {currentStep === 4 && (
                            <motion.div
                                key="complete"
                                variants={stepVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="space-y-6"
                            >
                                {/* Success Message */}
                                <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-6">
                                    <div className="flex items-center mb-4">
                                        <CheckCircle className="w-8 h-8 text-green-600 dark:text-green-400 mr-3" />
                                        <div>
                                            <h2 className="text-xl font-semibold text-green-900 dark:text-green-300">
                                                Upload Complete!
                                            </h2>
                                            <p className="text-green-700 dark:text-green-400 mt-1">
                                                Your blog files have been
                                                successfully processed and
                                                imported.
                                            </p>
                                        </div>
                                    </div>
                                </div>

                                {/* Results Summary */}
                                <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                        Import Summary
                                    </h3>

                                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                                        <div className="text-center p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                                            <div className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                                                {importResults?.total_imported ||
                                                    0}
                                            </div>
                                            <div className="text-sm text-blue-700 dark:text-blue-300">
                                                Posts Imported
                                            </div>
                                        </div>

                                        <div className="text-center p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                                            <div className="text-2xl font-bold text-green-600 dark:text-green-400">
                                                {importResults?.categories_created ||
                                                    0}
                                            </div>
                                            <div className="text-sm text-green-700 dark:text-green-300">
                                                Categories Created
                                            </div>
                                        </div>

                                        <div className="text-center p-4 bg-purple-50 dark:bg-purple-900/20 rounded-lg">
                                            <div className="text-2xl font-bold text-purple-600 dark:text-purple-400">
                                                {importResults?.tags_created ||
                                                    0}
                                            </div>
                                            <div className="text-sm text-purple-700 dark:text-purple-300">
                                                Tags Created
                                            </div>
                                        </div>

                                        <div className="text-center p-4 bg-orange-50 dark:bg-orange-900/20 rounded-lg">
                                            <div className="text-2xl font-bold text-orange-600 dark:text-orange-400">
                                                {importResults?.authors_created ||
                                                    0}
                                            </div>
                                            <div className="text-sm text-orange-700 dark:text-orange-300">
                                                Authors Created
                                            </div>
                                        </div>
                                    </div>

                                    {/* Imported Files List */}
                                    {importResults?.imported_files && (
                                        <div className="space-y-3">
                                            <h4 className="font-medium text-gray-900 dark:text-white">
                                                Imported Posts:
                                            </h4>
                                            <div className="space-y-2">
                                                {importResults.imported_files.map(
                                                    (file, index) => (
                                                        <div
                                                            key={index}
                                                            className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-900 rounded-lg"
                                                        >
                                                            <div className="flex items-center gap-3">
                                                                <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400" />
                                                                <div>
                                                                    <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                                        {
                                                                            file.filename
                                                                        }
                                                                    </div>
                                                                    <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                        Slug:{" "}
                                                                        {
                                                                            file.blog_post_slug
                                                                        }
                                                                    </div>
                                                                </div>
                                                            </div>
                                                            <div className="flex items-center gap-2">
                                                                <span className="text-xs text-gray-500 dark:text-gray-400">
                                                                    {
                                                                        file.category
                                                                    }{" "}
                                                                    •{" "}
                                                                    {
                                                                        file.tags_count
                                                                    }{" "}
                                                                    tags
                                                                </span>
                                                                <Link
                                                                    href={`/blog/${file.blog_post_slug}`}
                                                                    target="_blank"
                                                                    className="p-1 text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                                                >
                                                                    <ExternalLink className="w-4 h-4" />
                                                                </Link>
                                                            </div>
                                                        </div>
                                                    )
                                                )}
                                            </div>
                                        </div>
                                    )}
                                </div>

                                {/* Action Buttons */}
                                <div className="flex items-center justify-between">
                                    <button
                                        onClick={handleReset}
                                        className="flex items-center gap-2 px-4 py-2 text-gray-700 dark:text-gray-300 bg-gray-100 dark:bg-gray-700 rounded-lg hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
                                    >
                                        <Plus className="w-4 h-4" />
                                        <span>Upload More Files</span>
                                    </button>

                                    <div className="flex items-center gap-3">
                                        <Link
                                            href="/admin/blog/posts"
                                            className="flex items-center gap-2 px-4 py-2 text-blue-600 dark:text-blue-400 bg-blue-50 dark:bg-blue-900/20 rounded-lg hover:bg-blue-100 dark:hover:bg-blue-900/30 transition-colors"
                                        >
                                            <FileText className="w-4 h-4" />
                                            <span>Manage Posts</span>
                                        </Link>

                                        <Link
                                            href="/admin/blog"
                                            className="flex items-center gap-2 px-6 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                                        >
                                            <BarChart3 className="w-4 h-4" />
                                            <span>View Dashboard</span>
                                        </Link>
                                    </div>
                                </div>
                            </motion.div>
                        )}
                    </AnimatePresence>
                </div>

                {/* Help Section */}
                <motion.div
                    variants={itemVariants}
                    className="bg-gray-50 dark:bg-gray-900 rounded-xl p-6 border border-gray-200 dark:border-gray-700"
                >
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                        Need Help?
                    </h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <h4 className="font-medium text-gray-900 dark:text-white mb-2">
                                File Format Example:
                            </h4>
                            <pre className="text-xs bg-gray-100 dark:bg-gray-800 p-3 rounded-lg overflow-x-auto">
                                {`export default {
  title: "Your Blog Post Title",
  slug: "your-blog-post-slug",
  excerpt: "Brief description...",
  content: "Full blog post content...",
  author: "Author Name",
  category: "category-slug",
  tags: ["tag1", "tag2"],
  publishedAt: "2024-01-01T00:00:00Z",
  meta: {
    title: "SEO Title",
    description: "SEO Description"
  }
};`}
                            </pre>
                        </div>
                        <div>
                            <h4 className="font-medium text-gray-900 dark:text-white mb-2">
                                Common Issues:
                            </h4>
                            <ul className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                                <li>• Make sure files have .js extension</li>
                                <li>
                                    • Verify all required fields are present
                                </li>
                                <li>• Check that slugs are unique</li>
                                <li>
                                    • Ensure valid date format for publishedAt
                                </li>
                                <li>• Tags should be an array of strings</li>
                            </ul>
                        </div>
                    </div>
                </motion.div>
            </motion.div>

            {/* File Preview Modal */}
            <AnimatePresence>
                {showPreview && selectedFile && (
                    <BlogFilePreview
                        file={selectedFile}
                        onClose={() => {
                            setShowPreview(false);
                            setSelectedFile(null);
                        }}
                    />
                )}
            </AnimatePresence>
        </AdminLayout>
    );
}
