import { useMemo, useState } from 'react';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, ChevronUp, ChevronDown, Trash2, Save } from 'lucide-react';
import { ensureBoolean, ensureRichText, ensureString } from './editorUtils';

const iconOptions = [
    { value: 'Award', label: '🏆 Award' },
    { value: 'Shield', label: '🛡️ Shield' },
    { value: 'BookOpen', label: '📖 Expertise' },
    { value: 'CheckCircle', label: '✓ Verified' },
    { value: 'Star', label: '⭐ Rating' },
    { value: 'Users', label: '👥 Team' },
    { value: 'Target', label: '🎯 Precision' },
    { value: 'Heart', label: '❤️ Care' },
    { value: 'Zap', label: '⚡ Speed' },
    { value: 'Clock', label: '⏰ Timing' },
    { value: 'FileText', label: '📄 Documentation' },
    { value: 'Search', label: '🔍 Research' },
    { value: 'DollarSign', label: '💲 Pricing' },
    { value: 'ShieldCheck', label: '🛡️ Shield Check' },
    { value: 'Percent', label: '％ Savings' },
    { value: 'Headset', label: '🎧 Support' },
    { value: 'MessageCircle', label: '💬 Messaging' },
    { value: 'Phone', label: '📞 Phone' },
];

const supportChannelTypes = [
    { value: 'chat', label: 'Live chat' },
    { value: 'email', label: 'Email' },
    { value: 'phone', label: 'Phone' },
    { value: 'support', label: 'Support queue' },
];

export default function ServiceBenefitsBlockEditor({ block, onUpdate }) {
    const initialContent = useMemo(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return {};
        }
        return block.content;
    }, [block]);

    const [content, setContent] = useState({
        eyebrow: ensureString(initialContent.eyebrow),
        title: ensureString(initialContent.title),
        description: ensureRichText(initialContent.description_html || initialContent.description || ''),
        ctaText: ensureString(initialContent.ctaText),
        ctaLink: ensureString(initialContent.ctaLink),
    });

    const [benefits, setBenefits] = useState(() => {
        if (!Array.isArray(initialContent.benefits)) {
            return [];
        }

        return initialContent.benefits.map((benefit) => ({
            icon: ensureString(benefit?.icon, 'Award') || 'Award',
            title: ensureString(benefit?.title),
            description: ensureRichText(benefit?.description_html || benefit?.description || ''),
            is_active: ensureBoolean(benefit?.is_active),
        }));
    });

    const [guarantees, setGuarantees] = useState(() => {
        const source = initialContent.guarantees;
        if (source && !Array.isArray(source) && typeof source === 'object') {
            return {
                title: ensureString(source.title),
                description: ensureString(source.description),
                items: Array.isArray(source.items)
                    ? source.items.map((item) => ({
                          title: ensureString(item?.title),
                          description: ensureRichText(item?.description_html || item?.description || ''),
                          is_active: ensureBoolean(item?.is_active),
                      }))
                    : [],
            };
        }

        if (Array.isArray(source)) {
            return {
                title: ensureString(initialContent.guarantees_title || 'Our Guarantees'),
                description: ensureString(initialContent.guarantees_description || ''),
                items: source.map((item) => ({
                    title: ensureString(item?.title),
                    description: ensureRichText(item?.description_html || item?.description || ''),
                    is_active: ensureBoolean(item?.is_active),
                })),
            };
        }

        return {
            title: 'Our Guarantees',
            description: '',
            items: [],
        };
    });

    const [quickStart, setQuickStart] = useState(() => {
        const source = initialContent.quickStart || {};
        return {
            title: ensureString(source.title),
            description: ensureRichText(source.description_html || source.description || ''),
            footnote: ensureString(source.footnote),
            steps: Array.isArray(source.steps)
                ? source.steps.map((step) => ({
                      title: ensureString(step?.title),
                      description: ensureRichText(step?.description_html || step?.description || ''),
                      is_active: ensureBoolean(step?.is_active),
                  }))
                : [],
        };
    });

    const [offer, setOffer] = useState(() => {
        const source = initialContent.offer || {};
        return {
            badge: ensureString(source.badge),
            title: ensureString(source.title),
            description: ensureRichText(source.description_html || source.description || ''),
            code: ensureString(source.code),
            note: ensureString(source.note),
            ctaText: ensureString(source.ctaText),
            ctaLink: ensureString(source.ctaLink),
            sidebarTitle: ensureString(source.sidebarTitle),
            sidebarDescription: ensureString(source.sidebarDescription),
            sidebarFootnote: ensureString(source.sidebarFootnote),
            sidebarCtaText: ensureString(source.sidebarCtaText),
            sidebarCtaLink: ensureString(source.sidebarCtaLink),
        };
    });

    const [support, setSupport] = useState(() => {
        const source = initialContent.support || {};
        return {
            title: ensureString(source.title),
            description: ensureRichText(source.description_html || source.description || ''),
            channels: Array.isArray(source.channels)
                ? source.channels.map((channel) => ({
                      label: ensureString(channel?.label),
                      value: ensureString(channel?.value),
                      href: ensureString(channel?.href),
                      type: ensureString(channel?.type || 'chat'),
                      is_active: ensureBoolean(channel?.is_active),
                  }))
                : [],
        };
    });

    const [trustSignals, setTrustSignals] = useState(() => {
        if (!Array.isArray(initialContent.trustSignals)) {
            return [];
        }

        return initialContent.trustSignals.map((signal) => ({
            label: ensureString(signal?.label),
            value: ensureString(signal?.value),
            description: ensureString(signal?.description),
            is_active: ensureBoolean(signal?.is_active),
        }));
    });

    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        const payload = {
            eyebrow: ensureString(content.eyebrow),
            title: ensureString(content.title),
            description: content.description || '',
            description_html: content.description || '',
            ctaText: ensureString(content.ctaText),
            ctaLink: ensureString(content.ctaLink),
            benefits: benefits.map((benefit) => ({
                icon: ensureString(benefit.icon, 'Award') || 'Award',
                title: ensureString(benefit.title),
                description: benefit.description || '',
                description_html: benefit.description || '',
                is_active: ensureBoolean(benefit.is_active),
            })),
            guarantees: {
                title: ensureString(guarantees.title),
                description: ensureString(guarantees.description),
                items: guarantees.items.map((item) => ({
                    title: ensureString(item.title),
                    description: item.description || '',
                    description_html: item.description || '',
                    is_active: ensureBoolean(item.is_active),
                })),
            },
            quickStart: {
                title: ensureString(quickStart.title),
                description: quickStart.description || '',
                description_html: quickStart.description || '',
                footnote: ensureString(quickStart.footnote),
                steps: quickStart.steps.map((step) => ({
                    title: ensureString(step.title),
                    description: step.description || '',
                    description_html: step.description || '',
                    is_active: ensureBoolean(step.is_active),
                })),
            },
            offer: {
                badge: ensureString(offer.badge),
                title: ensureString(offer.title),
                description: offer.description || '',
                description_html: offer.description || '',
                code: ensureString(offer.code),
                note: ensureString(offer.note),
                ctaText: ensureString(offer.ctaText),
                ctaLink: ensureString(offer.ctaLink),
                sidebarTitle: ensureString(offer.sidebarTitle),
                sidebarDescription: ensureString(offer.sidebarDescription),
                sidebarFootnote: ensureString(offer.sidebarFootnote),
                sidebarCtaText: ensureString(offer.sidebarCtaText),
                sidebarCtaLink: ensureString(offer.sidebarCtaLink),
            },
            support: {
                title: ensureString(support.title),
                description: support.description || '',
                description_html: support.description || '',
                channels: support.channels.map((channel) => ({
                    label: ensureString(channel.label),
                    value: ensureString(channel.value),
                    href: ensureString(channel.href),
                    type: ensureString(channel.type || 'chat'),
                    is_active: ensureBoolean(channel.is_active),
                })),
            },
            trustSignals: trustSignals.map((signal) => ({
                label: ensureString(signal.label),
                value: ensureString(signal.value),
                description: ensureString(signal.description),
                is_active: ensureBoolean(signal.is_active),
            })),
        };

        onUpdate(block.id, payload);
    };

    const addBenefit = () => {
        setBenefits((current) => [
            ...current,
            { icon: 'Award', title: '', description: '', is_active: true },
        ]);
    };

    const updateBenefit = (index, partial) => {
        setBenefits((current) => {
            const clone = [...current];
            clone[index] = { ...clone[index], ...partial };
            return clone;
        });
    };

    const removeBenefit = (index) => {
        setBenefits((current) => current.filter((_, idx) => idx !== index));
    };

    const moveBenefitUp = (index) => {
        if (index === 0) return;
        setBenefits((current) => {
            const clone = [...current];
            [clone[index - 1], clone[index]] = [clone[index], clone[index - 1]];
            return clone;
        });
    };

    const moveBenefitDown = (index) => {
        if (index === benefits.length - 1) return;
        setBenefits((current) => {
            const clone = [...current];
            [clone[index], clone[index + 1]] = [clone[index + 1], clone[index]];
            return clone;
        });
    };

    const addGuarantee = () => {
        setGuarantees((current) => ({
            ...current,
            items: [
                ...current.items,
                { title: '', description: '', is_active: true },
            ],
        }));
    };

    const updateGuarantee = (index, partial) => {
        setGuarantees((current) => {
            const items = [...current.items];
            items[index] = { ...items[index], ...partial };
            return { ...current, items };
        });
    };

    const removeGuarantee = (index) => {
        setGuarantees((current) => ({
            ...current,
            items: current.items.filter((_, idx) => idx !== index),
        }));
    };

    const moveGuaranteeUp = (index) => {
        if (index === 0) return;
        setGuarantees((current) => {
            const items = [...current.items];
            [items[index - 1], items[index]] = [items[index], items[index - 1]];
            return { ...current, items };
        });
    };

    const moveGuaranteeDown = (index) => {
        setGuarantees((current) => {
            const items = [...current.items];
            if (index >= items.length - 1) {
                return current;
            }
            [items[index], items[index + 1]] = [items[index + 1], items[index]];
            return { ...current, items };
        });
    };

    const addQuickStartStep = () => {
        setQuickStart((current) => ({
            ...current,
            steps: [
                ...current.steps,
                { title: '', description: '', is_active: true },
            ],
        }));
    };

    const updateQuickStartStep = (index, partial) => {
        setQuickStart((current) => {
            const steps = [...current.steps];
            steps[index] = { ...steps[index], ...partial };
            return { ...current, steps };
        });
    };

    const removeQuickStartStep = (index) => {
        setQuickStart((current) => ({
            ...current,
            steps: current.steps.filter((_, idx) => idx !== index),
        }));
    };

    const moveQuickStartStep = (index, direction) => {
        setQuickStart((current) => {
            const steps = [...current.steps];
            const targetIndex = index + direction;
            if (targetIndex < 0 || targetIndex >= steps.length) return current;
            [steps[index], steps[targetIndex]] = [steps[targetIndex], steps[index]];
            return { ...current, steps };
        });
    };

    const addSupportChannel = () => {
        setSupport((current) => ({
            ...current,
            channels: [
                ...current.channels,
                { label: '', value: '', href: '', type: 'chat', is_active: true },
            ],
        }));
    };

    const updateSupportChannel = (index, partial) => {
        setSupport((current) => {
            const channels = [...current.channels];
            channels[index] = { ...channels[index], ...partial };
            return { ...current, channels };
        });
    };

    const removeSupportChannel = (index) => {
        setSupport((current) => ({
            ...current,
            channels: current.channels.filter((_, idx) => idx !== index),
        }));
    };

    const moveSupportChannelUp = (index) => {
        if (index === 0) return;
        setSupport((current) => {
            const channels = [...current.channels];
            [channels[index - 1], channels[index]] = [channels[index], channels[index - 1]];
            return { ...current, channels };
        });
    };

    const moveSupportChannelDown = (index) => {
        setSupport((current) => {
            const channels = [...current.channels];
            if (index >= channels.length - 1) {
                return current;
            }
            [channels[index], channels[index + 1]] = [channels[index + 1], channels[index]];
            return { ...current, channels };
        });
    };

    const addTrustSignal = () => {
        setTrustSignals((current) => [
            ...current,
            { label: '', value: '', description: '', is_active: true },
        ]);
    };

    const updateTrustSignal = (index, partial) => {
        setTrustSignals((current) => {
            const signals = [...current];
            signals[index] = { ...signals[index], ...partial };
            return signals;
        });
    };

    const removeTrustSignal = (index) => {
        setTrustSignals((current) => current.filter((_, idx) => idx !== index));
    };

    const moveTrustSignalUp = (index) => {
        if (index === 0) return;
        setTrustSignals((current) => {
            const signals = [...current];
            [signals[index - 1], signals[index]] = [signals[index], signals[index - 1]];
            return signals;
        });
    };

    const moveTrustSignalDown = (index) => {
        setTrustSignals((current) => {
            const signals = [...current];
            if (index >= signals.length - 1) {
                return current;
            }
            [signals[index], signals[index + 1]] = [signals[index + 1], signals[index]];
            return signals;
        });
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'benefits', label: 'Benefits', icon: '✨' },
        { id: 'guarantees', label: 'Guarantees', icon: '🛡️' },
        { id: 'quickStart', label: 'Quick Start', icon: '⚡' },
        { id: 'offer', label: 'Promotion', icon: '🎟️' },
        { id: 'support', label: 'Support', icon: '📞' },
        { id: 'trust', label: 'Trust Signals', icon: '📊' },
        { id: 'cta', label: 'Call to Action', icon: '🔘' },
    ];

    return (
        <div className="space-y-6">
            <div className="flex flex-wrap gap-2 border-b border-gray-200 pb-3 dark:border-gray-700">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 text-gray-700 hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Benefits Overview</h3>

                    <div>
                        <Label htmlFor="service-benefits-eyebrow">Eyebrow</Label>
                        <Input
                            id="service-benefits-eyebrow"
                            value={content.eyebrow}
                            onChange={(event) => setContent((prev) => ({ ...prev, eyebrow: event.target.value }))}
                            placeholder="e.g., Why students trust us"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-benefits-title">Section Title</Label>
                        <Input
                            id="service-benefits-title"
                            value={content.title}
                            onChange={(event) => setContent((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="Why Students Choose Our Essay Writing Service"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-benefits-description">Section Intro (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description}
                            onChange={(value) => setContent((prev) => ({ ...prev, description: value }))}
                            height={220}
                            placeholder="Summarise the key advantages of your service."
                        />
                    </div>
                </div>
            )}

            {activeTab === 'benefits' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Benefits Grid</h3>
                        <Button type="button" onClick={addBenefit} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Benefit
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Each benefit appears as a card with icon, title, and supporting copy.
                    </p>

                    {benefits.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No benefits yet.
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {benefits.map((benefit, index) => (
                                <div
                                    key={`service-benefit-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-blue-600 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-center justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Benefit Details
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={benefit.is_active}
                                                            onChange={(event) => updateBenefit(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeBenefit(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove benefit"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <div className="grid gap-4 md:grid-cols-2">
                                                <div>
                                                    <Label>Icon</Label>
                                                    <select
                                                        value={benefit.icon || 'Award'}
                                                        onChange={(event) => updateBenefit(index, { icon: event.target.value })}
                                                        className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                                                    >
                                                        {iconOptions.map((opt) => (
                                                            <option key={opt.value} value={opt.value}>
                                                                {opt.label}
                                                            </option>
                                                        ))}
                                                    </select>
                                                </div>
                                                <div>
                                                    <Label>Benefit Title</Label>
                                                    <Input
                                                        value={benefit.title}
                                                        onChange={(event) => updateBenefit(index, { title: event.target.value })}
                                                        placeholder="e.g., 100% Original Content"
                                                    />
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Benefit Description</Label>
                                                <RichTextEditor
                                                    value={benefit.description}
                                                    onChange={(value) => updateBenefit(index, { description: value })}
                                                    height={160}
                                                    placeholder="Share how this benefit helps buyers succeed."
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveBenefitUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveBenefitDown(index)}
                                                disabled={index === benefits.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {activeTab === 'guarantees' && (
                <div className="space-y-4">
                    <div className="space-y-3 rounded-lg border border-gray-200 bg-gray-50 p-4 dark:border-gray-700 dark:bg-gray-800">
                        <Label className="text-sm font-semibold text-gray-900 dark:text-white">Guarantees Header</Label>
                        <Input
                            value={guarantees.title}
                            onChange={(event) => setGuarantees((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="Our Guarantees"
                        />
                        <Input
                            value={guarantees.description}
                            onChange={(event) => setGuarantees((prev) => ({ ...prev, description: event.target.value }))}
                            placeholder="Optional short subheading"
                        />
                    </div>

                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Guarantee Items</h3>
                        <Button type="button" onClick={addGuarantee} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Guarantee
                        </Button>
                    </div>

                    {guarantees.items.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No guarantees yet.
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {guarantees.items.map((guarantee, index) => (
                                <div
                                    key={`guarantee-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-purple-600 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-center justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Guarantee Details
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={guarantee.is_active}
                                                            onChange={(event) => updateGuarantee(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeGuarantee(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove guarantee"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Guarantee Title</Label>
                                                <Input
                                                    value={guarantee.title}
                                                    onChange={(event) => updateGuarantee(index, { title: event.target.value })}
                                                    placeholder="e.g., 100% Original Content"
                                                />
                                            </div>
                                            <div>
                                                <Label>Guarantee Description</Label>
                                                <RichTextEditor
                                                    value={guarantee.description}
                                                    onChange={(value) => updateGuarantee(index, { description: value })}
                                                    height={160}
                                                    placeholder="Explain the guarantee in detail."
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveGuaranteeUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveGuaranteeDown(index)}
                                                disabled={index === guarantees.items.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {activeTab === 'quickStart' && (
                <div className="space-y-4">
                    <div className="space-y-3 rounded-lg border border-gray-200 bg-gray-50 p-4 dark:border-gray-700 dark:bg-gray-800">
                        <Label className="text-sm font-semibold text-gray-900 dark:text-white">Quick Start Header</Label>
                        <Input
                            value={quickStart.title}
                            onChange={(event) => setQuickStart((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="Quick Start"
                        />
                        <RichTextEditor
                            value={quickStart.description}
                            onChange={(value) => setQuickStart((prev) => ({ ...prev, description: value }))}
                            height={160}
                            placeholder="Intro copy for the quick start steps."
                        />
                        <Input
                            value={quickStart.footnote}
                            onChange={(event) => setQuickStart((prev) => ({ ...prev, footnote: event.target.value }))}
                            placeholder="e.g., No registration required for quotes."
                        />
                    </div>

                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Quick Start Steps</h3>
                        <Button type="button" onClick={addQuickStartStep} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Step
                        </Button>
                    </div>

                    {quickStart.steps.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No steps yet.
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {quickStart.steps.map((step, index) => (
                                <div
                                    key={`quick-start-step-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-yellow-600 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-center justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Step Details
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={step.is_active}
                                                            onChange={(event) => updateQuickStartStep(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeQuickStartStep(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove step"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Step Title</Label>
                                                <Input
                                                    value={step.title}
                                                    onChange={(event) => updateQuickStartStep(index, { title: event.target.value })}
                                                    placeholder="e.g., Order your first essay"
                                                />
                                            </div>
                                            <div>
                                                <Label>Step Description</Label>
                                                <RichTextEditor
                                                    value={step.description}
                                                    onChange={(value) => updateQuickStartStep(index, { description: value })}
                                                    height={160}
                                                    placeholder="Explain the step in detail."
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveQuickStartStep(index, -1)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveQuickStartStep(index, 1)}
                                                disabled={index === quickStart.steps.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {activeTab === 'offer' && (
                <div className="space-y-4">
                    <div className="space-y-3 rounded-lg border border-gray-200 bg-gray-50 p-4 dark:border-gray-700 dark:bg-gray-800">
                        <Label className="text-sm font-semibold text-gray-900 dark:text-white">Promotion Header</Label>
                        <Input
                            value={offer.badge}
                            onChange={(event) => setOffer((prev) => ({ ...prev, badge: event.target.value }))}
                            placeholder="e.g., Limited Time Offer"
                        />
                        <Input
                            value={offer.title}
                            onChange={(event) => setOffer((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="e.g., Save 10% on your first order"
                        />
                        <RichTextEditor
                            value={offer.description}
                            onChange={(value) => setOffer((prev) => ({ ...prev, description: value }))}
                            height={180}
                            placeholder="Detail the promotion and why it matters."
                        />
                        <Input
                            value={offer.code}
                            onChange={(event) => setOffer((prev) => ({ ...prev, code: event.target.value }))}
                            placeholder="e.g., 10OFF"
                        />
                        <Input
                            value={offer.note}
                            onChange={(event) => setOffer((prev) => ({ ...prev, note: event.target.value }))}
                            placeholder="e.g., Expires in 24 hours"
                        />
                    </div>

                    <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                        <div>
                            <Label htmlFor="service-benefits-offer-cta-text">Button Text</Label>
                            <Input
                                id="service-benefits-offer-cta-text"
                                value={offer.ctaText}
                                onChange={(event) => setOffer((prev) => ({ ...prev, ctaText: event.target.value }))}
                                placeholder="Order now"
                            />
                        </div>
                        <div>
                            <Label htmlFor="service-benefits-offer-cta-link">Button Link</Label>
                            <Input
                                id="service-benefits-offer-cta-link"
                                value={offer.ctaLink}
                                onChange={(event) => setOffer((prev) => ({ ...prev, ctaLink: event.target.value }))}
                                placeholder="/place-order"
                            />
                        </div>
                    </div>

                    <div className="space-y-3 rounded-lg border border-gray-200 bg-gray-50 p-4 dark:border-gray-700 dark:bg-gray-800">
                        <Label className="text-sm font-semibold text-gray-900 dark:text-white">Promotion Sidebar</Label>
                        <Input
                            value={offer.sidebarTitle}
                            onChange={(event) => setOffer((prev) => ({ ...prev, sidebarTitle: event.target.value }))}
                            placeholder="e.g., Save 10% on your first order"
                        />
                        <Input
                            value={offer.sidebarDescription}
                            onChange={(event) => setOffer((prev) => ({ ...prev, sidebarDescription: event.target.value }))}
                            placeholder="Optional short description"
                        />
                        <Input
                            value={offer.sidebarFootnote}
                            onChange={(event) => setOffer((prev) => ({ ...prev, sidebarFootnote: event.target.value }))}
                            placeholder="e.g., Expires in 24 hours"
                        />
                        <Input
                            value={offer.sidebarCtaText}
                            onChange={(event) => setOffer((prev) => ({ ...prev, sidebarCtaText: event.target.value }))}
                            placeholder="Order now"
                        />
                        <Input
                            value={offer.sidebarCtaLink}
                            onChange={(event) => setOffer((prev) => ({ ...prev, sidebarCtaLink: event.target.value }))}
                            placeholder="/place-order"
                        />
                    </div>
                </div>
            )}

            {activeTab === 'support' && (
                <div className="space-y-4">
                    <div className="space-y-3 rounded-lg border border-gray-200 bg-gray-50 p-4 dark:border-gray-700 dark:bg-gray-800">
                        <Label className="text-sm font-semibold text-gray-900 dark:text-white">Support Header</Label>
                        <Input
                            value={support.title}
                            onChange={(event) => setSupport((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="Support"
                        />
                        <RichTextEditor
                            value={support.description}
                            onChange={(value) => setSupport((prev) => ({ ...prev, description: value }))}
                            height={160}
                            placeholder="Explain how and when customers can get help."
                        />
                    </div>

                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Support Channels</h3>
                        <Button type="button" onClick={addSupportChannel} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Channel
                        </Button>
                    </div>

                    {support.channels.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No support channels yet.
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {support.channels.map((channel, index) => (
                                <div
                                    key={`support-channel-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-teal-600 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-center justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Channel Details
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={channel.is_active}
                                                            onChange={(event) => updateSupportChannel(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeSupportChannel(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove channel"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Channel Label</Label>
                                                <Input
                                                    value={channel.label}
                                                    onChange={(event) => updateSupportChannel(index, { label: event.target.value })}
                                                    placeholder="e.g., Live Chat"
                                                />
                                            </div>
                                            <div>
                                                <Label>Channel Value</Label>
                                                <Input
                                                    value={channel.value}
                                                    onChange={(event) => updateSupportChannel(index, { value: event.target.value })}
                                                    placeholder="e.g., chat"
                                                />
                                            </div>
                                            <div>
                                                <Label>Channel Href</Label>
                                                <Input
                                                    value={channel.href}
                                                    onChange={(event) => updateSupportChannel(index, { href: event.target.value })}
                                                    placeholder="e.g., /chat"
                                                />
                                            </div>
                                            <div>
                                                <Label>Channel Type</Label>
                                                <select
                                                    value={channel.type}
                                                    onChange={(event) => updateSupportChannel(index, { type: event.target.value })}
                                                    className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                                                >
                                                    {supportChannelTypes.map((type) => (
                                                        <option key={type.value} value={type.value}>
                                                            {type.label}
                                                        </option>
                                                    ))}
                                                </select>
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveSupportChannelUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveSupportChannelDown(index)}
                                                disabled={index === support.channels.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {activeTab === 'trust' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Trust Signals</h3>
                        <Button type="button" onClick={addTrustSignal} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Signal
                        </Button>
                    </div>

                    {trustSignals.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No trust signals yet.
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {trustSignals.map((signal, index) => (
                                <div
                                    key={`trust-signal-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-purple-600 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-center justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Signal Details
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={signal.is_active}
                                                            onChange={(event) => updateTrustSignal(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeTrustSignal(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove signal"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Signal Label</Label>
                                                <Input
                                                    value={signal.label}
                                                    onChange={(event) => updateTrustSignal(index, { label: event.target.value })}
                                                    placeholder="e.g., 100% Original Content"
                                                />
                                            </div>
                                            <div>
                                                <Label>Signal Value</Label>
                                                <Input
                                                    value={signal.value}
                                                    onChange={(event) => updateTrustSignal(index, { value: event.target.value })}
                                                    placeholder="e.g., 100% Original Content"
                                                />
                                            </div>
                                            <div>
                                                <Label>Signal Description</Label>
                                                <RichTextEditor
                                                    value={signal.description}
                                                    onChange={(value) => updateTrustSignal(index, { description: value })}
                                                    height={160}
                                                    placeholder="Explain the signal in detail."
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveTrustSignalUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveTrustSignalDown(index)}
                                                disabled={index === trustSignals.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {activeTab === 'cta' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Optional CTA</h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Display a call-to-action button beneath the benefits grid.
                    </p>

                    <div className="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-800 dark:bg-blue-900/20">
                        <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                            <div>
                                <Label htmlFor="service-benefits-cta-text">Button Text</Label>
                                <Input
                                    id="service-benefits-cta-text"
                                    value={content.ctaText}
                                    onChange={(event) => setContent((prev) => ({ ...prev, ctaText: event.target.value }))}
                                    placeholder="Order professional help now"
                                />
                            </div>
                            <div>
                                <Label htmlFor="service-benefits-cta-link">Button Link</Label>
                                <Input
                                    id="service-benefits-cta-link"
                                    value={content.ctaLink}
                                    onChange={(event) => setContent((prev) => ({ ...prev, ctaLink: event.target.value }))}
                                    placeholder="/place-order"
                                />
                            </div>
                        </div>
                    </div>
                </div>
            )}

            <div className="sticky bottom-0 bg-white pt-4 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Service Benefits
                </button>
            </div>
        </div>
    );
}
