import { useMemo, useState } from 'react';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Textarea } from '@/Components/ui/Textarea';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, ChevronUp, ChevronDown, Trash2, Save } from 'lucide-react';
import { ensureBoolean, ensureRichText, ensureString } from './editorUtils';

const TABS = [
    { id: 'main', label: 'Main Content', icon: '📝' },
    { id: 'keyPoints', label: 'Key Points', icon: '✓' },
    { id: 'cta', label: 'Call to Action', icon: '🔘' },
    { id: 'trust', label: 'Trust Signals', icon: '⭐' },
];

export default function ServiceHeroBlockEditor({ block, onUpdate }) {
    const initialContent = useMemo(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return {};
        }
        return block.content;
    }, [block]);

    const [content, setContent] = useState({
        eyebrow: ensureString(initialContent.eyebrow),
        title: ensureString(initialContent.title),
        description: ensureRichText(initialContent.description_html || initialContent.description || ''),
        ctaText: ensureString(initialContent.ctaText, 'Get Started Now'),
        ctaLink: ensureString(initialContent.ctaLink, '/place-order'),
    });

    const [keyPoints, setKeyPoints] = useState(() => {
        if (!Array.isArray(initialContent.keyPoints)) {
            return [];
        }

        return initialContent.keyPoints.map((point) => ({
            text: ensureString(point?.text ?? point),
            is_active: ensureBoolean(point?.is_active),
        }));
    });

    const [trustSignals, setTrustSignals] = useState(() => {
        if (!Array.isArray(initialContent.trustSignals)) {
            return [];
        }

        return initialContent.trustSignals.map((signal) => ({
            text: ensureString(signal?.text ?? signal),
            is_active: ensureBoolean(signal?.is_active),
        }));
    });

    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        const payload = {
            ...content,
            eyebrow: ensureString(content.eyebrow),
            title: ensureString(content.title),
            description: content.description || '',
            ctaText: ensureString(content.ctaText),
            ctaLink: ensureString(content.ctaLink),
            keyPoints: keyPoints.map((point) => ({
                text: ensureString(point.text),
                is_active: ensureBoolean(point.is_active),
            })),
            trustSignals: trustSignals.map((signal) => ({
                text: ensureString(signal.text),
                is_active: ensureBoolean(signal.is_active),
            })),
        };

        onUpdate(block.id, payload);
    };

    const addKeyPoint = () => {
        setKeyPoints((current) => [
            ...current,
            { text: '', is_active: true },
        ]);
    };

    const updateKeyPoint = (index, partial) => {
        setKeyPoints((current) => {
            const clone = [...current];
            clone[index] = { ...clone[index], ...partial };
            return clone;
        });
    };

    const removeKeyPoint = (index) => {
        setKeyPoints((current) => current.filter((_, idx) => idx !== index));
    };

    const moveKeyPointUp = (index) => {
        if (index === 0) return;
        setKeyPoints((current) => {
            const clone = [...current];
            [clone[index - 1], clone[index]] = [clone[index], clone[index - 1]];
            return clone;
        });
    };

    const moveKeyPointDown = (index) => {
        if (index === keyPoints.length - 1) return;
        setKeyPoints((current) => {
            const clone = [...current];
            [clone[index], clone[index + 1]] = [clone[index + 1], clone[index]];
            return clone;
        });
    };

    const addTrustSignal = () => {
        setTrustSignals((current) => [
            ...current,
            { text: '', is_active: true },
        ]);
    };

    const updateTrustSignal = (index, partial) => {
        setTrustSignals((current) => {
            const clone = [...current];
            clone[index] = { ...clone[index], ...partial };
            return clone;
        });
    };

    const removeTrustSignal = (index) => {
        setTrustSignals((current) => current.filter((_, idx) => idx !== index));
    };

    const moveTrustSignalUp = (index) => {
        if (index === 0) return;
        setTrustSignals((current) => {
            const clone = [...current];
            [clone[index - 1], clone[index]] = [clone[index], clone[index - 1]];
            return clone;
        });
    };

    const moveTrustSignalDown = (index) => {
        if (index === trustSignals.length - 1) return;
        setTrustSignals((current) => {
            const clone = [...current];
            [clone[index], clone[index + 1]] = [clone[index + 1], clone[index]];
            return clone;
        });
    };

    return (
        <div className="space-y-6">
            <div className="flex flex-wrap gap-2 border-b border-gray-200 pb-3 dark:border-gray-700">
                {TABS.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 text-gray-700 hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Service Hero Content</h3>

                    <div>
                        <Label htmlFor="service-hero-eyebrow">Eyebrow / Subheading</Label>
                        <Input
                            id="service-hero-eyebrow"
                            value={content.eyebrow}
                            onChange={(event) => setContent((prev) => ({ ...prev, eyebrow: event.target.value }))}
                            placeholder="e.g., Trusted by 10,000+ students each year"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-hero-title">Headline *</Label>
                        <Input
                            id="service-hero-title"
                            value={content.title}
                            onChange={(event) => setContent((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="Professional Essay Writing Service | Expert Custom Essay Help"
                            className="text-base"
                        />
                        <p className="mt-1 text-xs text-gray-500">Displayed as the primary headline on the service page.</p>
                    </div>

                    <div>
                        <Label htmlFor="service-hero-description">Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description}
                            onChange={(value) => setContent((prev) => ({ ...prev, description: value }))}
                            height={240}
                            placeholder="Craft a compelling description. Use formatting to highlight differentiators."
                        />
                        <p className="mt-1 text-xs text-gray-500">Supports headings, bullet lists, and links.</p>
                    </div>
                </div>
            )}

            {activeTab === 'keyPoints' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Service Key Points</h3>
                        <Button type="button" onClick={addKeyPoint} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Key Point
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Short bullet items rendered beneath the hero description.
                    </p>

                    {keyPoints.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No key points yet. Add your first key point to highlight value props.
                        </div>
                    ) : (
                        <div className="space-y-3">
                            {keyPoints.map((point, index) => (
                                <div
                                    key={`service-key-point-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-blue-600 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-center justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Key Point
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={point.is_active}
                                                            onChange={(event) => updateKeyPoint(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeKeyPoint(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove key point"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <Textarea
                                                value={point.text}
                                                onChange={(event) => updateKeyPoint(index, { text: event.target.value })}
                                                placeholder={`Key point ${index + 1}...`}
                                                rows={2}
                                            />
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveKeyPointUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveKeyPointDown(index)}
                                                disabled={index === keyPoints.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {activeTab === 'cta' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Primary Call to Action</h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400">Configure the main CTA button displayed in the hero.</p>

                    <div className="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-800 dark:bg-blue-900/20">
                        <div className="space-y-4">
                            <div>
                                <Label htmlFor="service-hero-cta-text">Primary Button Text</Label>
                                <Input
                                    id="service-hero-cta-text"
                                    value={content.ctaText}
                                    onChange={(event) => setContent((prev) => ({ ...prev, ctaText: event.target.value }))}
                                    placeholder="Get Professional Essay Help Now"
                                />
                            </div>
                            <div>
                                <Label htmlFor="service-hero-cta-link">Primary Button Link</Label>
                                <Input
                                    id="service-hero-cta-link"
                                    value={content.ctaLink}
                                    onChange={(event) => setContent((prev) => ({ ...prev, ctaLink: event.target.value }))}
                                    placeholder="/place-order"
                                />
                                <p className="mt-1 text-xs text-gray-600 dark:text-gray-400">Examples: /place-order, /pricing, /contact</p>
                            </div>
                            <div className="border-t border-blue-200 pt-4 dark:border-blue-800">
                                <Label>Live Preview</Label>
                                <div className="mt-3 flex flex-wrap gap-3">
                                    <span className="rounded-md bg-white px-5 py-2.5 font-medium text-blue-700 shadow">{content.ctaText || 'Get Started'}</span>
                                    <span className="rounded-md border-2 border-white px-5 py-2.5 font-medium text-gray-700">Place Order</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {activeTab === 'trust' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Trust Signals</h3>
                        <Button type="button" onClick={addTrustSignal} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Trust Signal
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Short badges shown above the headline (ratings, guarantees, support hours, etc.).
                    </p>

                    {trustSignals.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No trust signals yet.
                        </div>
                    ) : (
                        <div className="space-y-3">
                            {trustSignals.map((signal, index) => (
                                <div
                                    key={`service-trust-signal-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-center gap-3">
                                        <span className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-yellow-500 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <Input
                                            value={signal.text}
                                            onChange={(event) => updateTrustSignal(index, { text: event.target.value })}
                                            placeholder={`e.g., ${
                                                index === 0
                                                    ? '10,000+ satisfied students'
                                                    : index === 1
                                                    ? '4.8/5 average rating'
                                                    : '24/7 customer care'
                                            }`}
                                            className="flex-1"
                                        />
                                        <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                            <input
                                                type="checkbox"
                                                className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                checked={signal.is_active}
                                                onChange={(event) => updateTrustSignal(index, { is_active: event.target.checked })}
                                            />
                                            Active
                                        </label>
                                        <div className="flex gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveTrustSignalUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move left"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveTrustSignalDown(index)}
                                                disabled={index === trustSignals.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move right"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => removeTrustSignal(index)}
                                                className="p-1 text-red-500 hover:text-red-600"
                                                title="Remove"
                                            >
                                                <Trash2 className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            <div className="sticky bottom-0 bg-white pt-4 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Service Hero
                </button>
            </div>
        </div>
    );
}
