import { useMemo, useState } from 'react';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, ChevronUp, ChevronDown, Trash2, Save } from 'lucide-react';
import { ensureBoolean, ensureRichText, ensureString } from './editorUtils';

export default function ServiceTestimonialsBlockEditor({ block, onUpdate }) {
    const initialContent = useMemo(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return {};
        }
        return block.content;
    }, [block]);

    const [content, setContent] = useState({
        eyebrow: ensureString(initialContent.eyebrow),
        title: ensureString(initialContent.title),
        description: ensureRichText(initialContent.description_html || initialContent.description || ''),
    });

    const [testimonials, setTestimonials] = useState(() => {
        if (!Array.isArray(initialContent.testimonials)) {
            return [];
        }

        return initialContent.testimonials.map((testimonial) => ({
            quote: ensureRichText(testimonial?.quote_html || testimonial?.quote || ''),
            author: ensureString(testimonial?.author),
            role: ensureString(testimonial?.role),
            rating: typeof testimonial?.rating === 'number' && testimonial.rating > 0 ? testimonial.rating : 5,
            is_active: ensureBoolean(testimonial?.is_active),
        }));
    });

    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        const payload = {
            ...content,
            eyebrow: ensureString(content.eyebrow),
            title: ensureString(content.title),
            description: content.description || '',
            testimonials: testimonials.map((testimonial) => ({
                quote: testimonial.quote || '',
                author: ensureString(testimonial.author),
                role: ensureString(testimonial.role),
                rating:
                    typeof testimonial.rating === 'number' && testimonial.rating >= 1 && testimonial.rating <= 5
                        ? Math.round(testimonial.rating)
                        : 5,
                is_active: ensureBoolean(testimonial.is_active),
            })),
        };

        onUpdate(block.id, payload);
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'testimonials', label: 'Testimonials', icon: '💬' },
    ];

    const addTestimonial = () => {
        setTestimonials((current) => [
            ...current,
            { quote: '', author: '', role: '', rating: 5, is_active: true },
        ]);
    };

    const updateTestimonial = (index, partial) => {
        setTestimonials((current) => {
            const clone = [...current];
            clone[index] = { ...clone[index], ...partial };
            return clone;
        });
    };

    const removeTestimonial = (index) => {
        setTestimonials((current) => current.filter((_, idx) => idx !== index));
    };

    const moveTestimonialUp = (index) => {
        if (index === 0) return;
        setTestimonials((current) => {
            const clone = [...current];
            [clone[index - 1], clone[index]] = [clone[index], clone[index - 1]];
            return clone;
        });
    };

    const moveTestimonialDown = (index) => {
        if (index === testimonials.length - 1) return;
        setTestimonials((current) => {
            const clone = [...current];
            [clone[index], clone[index + 1]] = [clone[index + 1], clone[index]];
            return clone;
        });
    };

    return (
        <div className="space-y-6">
            <div className="flex flex-wrap gap-2 border-b border-gray-200 pb-3 dark:border-gray-700">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 text-gray-700 hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Testimonials Section</h3>

                    <div>
                        <Label htmlFor="service-testimonials-eyebrow">Eyebrow</Label>
                        <Input
                            id="service-testimonials-eyebrow"
                            value={content.eyebrow}
                            onChange={(event) => setContent((prev) => ({ ...prev, eyebrow: event.target.value }))}
                            placeholder="e.g., Real stories from satisfied students"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-testimonials-title">Section Title</Label>
                        <Input
                            id="service-testimonials-title"
                            value={content.title}
                            onChange={(event) => setContent((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="Trusted by students worldwide"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-testimonials-description">Section Description (Rich Text)</Label>
                        <RichTextEditor
                            id="service-testimonials-description"
                            value={content.description}
                            onChange={(value) => setContent((prev) => ({ ...prev, description: value }))}
                            height={200}
                            placeholder="Introduce the testimonials and reinforce credibility."
                        />
                    </div>
                </div>
            )}

            {activeTab === 'testimonials' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Testimonials</h3>
                        <Button type="button" onClick={addTestimonial} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Testimonial
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">Testimonials rotate in a carousel on the service page.</p>

                    {testimonials.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No testimonials yet.
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {testimonials.map((testimonial, index) => (
                                <div
                                    key={`service-testimonial-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-9 w-9 shrink-0 items-center justify-center rounded-full bg-purple-600 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-start justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Quote & Attribution
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={testimonial.is_active}
                                                            onChange={(event) => updateTestimonial(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeTestimonial(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove testimonial"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Quote</Label>
                                                <RichTextEditor
                                                    value={testimonial.quote}
                                                    onChange={(value) => updateTestimonial(index, { quote: value })}
                                                    height={160}
                                                    placeholder="Insert the student's commentary or results."
                                                />
                                            </div>
                                            <div className="grid grid-cols-1 gap-3 md:grid-cols-2">
                                                <div>
                                                    <Label>Author</Label>
                                                    <Input
                                                        value={testimonial.author}
                                                        onChange={(event) => updateTestimonial(index, { author: event.target.value })}
                                                        placeholder="e.g., Emily from UK"
                                                    />
                                                </div>
                                                <div>
                                                    <Label>Role / Degree</Label>
                                                    <Input
                                                        value={testimonial.role}
                                                        onChange={(event) => updateTestimonial(index, { role: event.target.value })}
                                                        placeholder="e.g., MSc Nursing Student"
                                                    />
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Rating (1-5)</Label>
                                                <input
                                                    type="number"
                                                    min="1"
                                                    max="5"
                                                    value={testimonial.rating}
                                                    onChange={(event) => updateTestimonial(index, { rating: Number(event.target.value) })}
                                                    className="w-24 rounded-md border border-gray-300 px-3 py-2 text-sm text-gray-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveTestimonialUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveTestimonialDown(index)}
                                                disabled={index === testimonials.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            <div className="sticky bottom-0 bg-white pt-4 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Service Testimonials
                </button>
            </div>
        </div>
    );
}
