import React, { useState, useEffect } from 'react';
import { Head, useForm, router, usePage } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { toast } from 'react-hot-toast';
import { ArrowLeft, Save, Trash2 } from 'lucide-react';

export default function FooterEdit({ setting, sectionTypes }) {
    const { flash } = usePage().props;
    const { data, setData, put, processing, errors } = useForm({
        section_type: setting.section_type,
        data: setting.data,
        order: setting.order,
        is_active: setting.is_active,
    });

    // Handle flash messages
    useEffect(() => {
        if (flash?.success) {
            toast.success(flash.success, { 
                duration: 4000,
                icon: '✅'
            });
        }
        if (flash?.error) {
            toast.error(flash.error, { 
                duration: 4000,
                icon: '❌'
            });
        }
    }, [flash]);

    const handleSubmit = (e) => {
        e.preventDefault();
        put(route('admin.cms.footer.update', setting.id), {
            onSuccess: () => {
                toast.success('Footer setting updated successfully!', {
                    icon: '💾',
                    duration: 3000
                });
            },
            onError: () => {
                toast.error('Failed to update footer setting. Please check the form.', {
                    icon: '❌',
                    duration: 4000
                });
            }
        });
    };

    const handleDelete = () => {
        if (confirm('Are you sure you want to delete this footer setting?')) {
            router.delete(route('admin.cms.footer.destroy', setting.id), {
                onSuccess: () => {
                    toast.success('Footer setting deleted successfully!', {
                        icon: '🗑️',
                        duration: 3000
                    });
                },
                onError: () => {
                    toast.error('Failed to delete footer setting', {
                        icon: '❌',
                        duration: 4000
                    });
                }
            });
        }
    };

    const updateDataField = (field, value) => {
        setData('data', { ...data.data, [field]: value });
    };

    const renderDataFields = () => {
        switch (data.section_type) {
            case 'company_info':
                return (
                    <>
                        <div className="space-y-2">
                            <Label htmlFor="title">Title</Label>
                            <Input
                                id="title"
                                value={data.data.title || ''}
                                onChange={(e) => updateDataField('title', e.target.value)}
                                placeholder="e.g., Academic Scribe"
                            />
                            {errors['data.title'] && (
                                <p className="text-sm text-red-600">{errors['data.title']}</p>
                            )}
                        </div>
                        <div className="space-y-2">
                            <Label htmlFor="description">Description (Rich Text)</Label>
                            <RichTextEditor
                                value={data.data.description || ''}
                                onChange={(content) => updateDataField('description', content)}
                                height={250}
                                placeholder="Professional academic writing services..."
                            />
                            {errors['data.description'] && (
                                <p className="text-sm text-red-600">{errors['data.description']}</p>
                            )}
                        </div>
                        {data.data.text !== undefined && (
                            <div className="space-y-2">
                                <Label htmlFor="text">Text (for copyright)</Label>
                                <Input
                                    id="text"
                                    value={data.data.text || ''}
                                    onChange={(e) => updateDataField('text', e.target.value)}
                                    placeholder="Academic Scribe. All rights reserved."
                                />
                            </div>
                        )}
                    </>
                );

            case 'service_link':
            case 'quick_link':
            case 'legal_link':
                return (
                    <>
                        <div className="space-y-2">
                            <Label htmlFor="name">Link Name</Label>
                            <Input
                                id="name"
                                value={data.data.name || ''}
                                onChange={(e) => updateDataField('name', e.target.value)}
                                placeholder="e.g., Privacy Policy"
                                required
                            />
                            {errors['data.name'] && (
                                <p className="text-sm text-red-600">{errors['data.name']}</p>
                            )}
                        </div>
                        <div className="space-y-2">
                            <Label htmlFor="url">URL</Label>
                            <Input
                                id="url"
                                value={data.data.url || ''}
                                onChange={(e) => updateDataField('url', e.target.value)}
                                placeholder="e.g., /privacy-policy or https://..."
                                required
                            />
                            {errors['data.url'] && (
                                <p className="text-sm text-red-600">{errors['data.url']}</p>
                            )}
                        </div>
                        {data.section_type === 'legal_link' && (
                            <div className="flex items-center gap-2">
                                <input
                                    type="checkbox"
                                    id="external"
                                    checked={data.data.external || false}
                                    onChange={(e) => updateDataField('external', e.target.checked)}
                                    className="w-4 h-4 rounded border-gray-300"
                                />
                                <Label htmlFor="external" className="cursor-pointer">
                                    External Link (opens in new tab)
                                </Label>
                            </div>
                        )}
                    </>
                );

            case 'contact_info':
                return (
                    <>
                        <div className="space-y-2">
                            <Label htmlFor="email">Email Address</Label>
                            <Input
                                id="email"
                                type="email"
                                value={data.data.email || ''}
                                onChange={(e) => updateDataField('email', e.target.value)}
                                placeholder="support@example.com"
                            />
                            {errors['data.email'] && (
                                <p className="text-sm text-red-600">{errors['data.email']}</p>
                            )}
                        </div>
                        <div className="space-y-2">
                            <Label htmlFor="phone">Phone Number</Label>
                            <Input
                                id="phone"
                                value={data.data.phone || ''}
                                onChange={(e) => updateDataField('phone', e.target.value)}
                                placeholder="+1 (555) 123-4567"
                            />
                            {errors['data.phone'] && (
                                <p className="text-sm text-red-600">{errors['data.phone']}</p>
                            )}
                        </div>
                        <div className="space-y-2">
                            <Label htmlFor="phone_description">Phone Description</Label>
                            <Input
                                id="phone_description"
                                value={data.data.phone_description || ''}
                                onChange={(e) => updateDataField('phone_description', e.target.value)}
                                placeholder="24/7 Support"
                            />
                        </div>
                        <div className="space-y-4 border-t pt-4">
                            <Label>Address</Label>
                            <div className="space-y-2">
                                <Input
                                    placeholder="Line 1"
                                    value={data.data.address?.line1 || ''}
                                    onChange={(e) => updateDataField('address', { 
                                        ...data.data.address, 
                                        line1: e.target.value 
                                    })}
                                />
                                <Input
                                    placeholder="Line 2"
                                    value={data.data.address?.line2 || ''}
                                    onChange={(e) => updateDataField('address', { 
                                        ...data.data.address, 
                                        line2: e.target.value 
                                    })}
                                />
                                <Input
                                    placeholder="Line 3"
                                    value={data.data.address?.line3 || ''}
                                    onChange={(e) => updateDataField('address', { 
                                        ...data.data.address, 
                                        line3: e.target.value 
                                    })}
                                />
                            </div>
                        </div>
                    </>
                );

            default:
                return <p className="text-gray-500">Unknown section type</p>;
        }
    };

    return (
        <AdminLayout>
            <Head title="Edit Footer Setting" />

            <div className="space-y-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div className="flex items-center gap-4">
                        <Button
                            variant="outline"
                            onClick={() => router.get(route('admin.cms.footer.index'))}
                        >
                            <ArrowLeft className="h-4 w-4 mr-2" />
                            Back
                        </Button>
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                                Edit Footer Setting
                            </h1>
                            <p className="mt-1 text-gray-600 dark:text-gray-400">
                                Update the footer content
                            </p>
                        </div>
                    </div>
                    <Button
                        variant="destructive"
                        onClick={handleDelete}
                    >
                        <Trash2 className="h-4 w-4 mr-2" />
                        Delete
                    </Button>
                </div>

                {/* Form */}
                <form onSubmit={handleSubmit}>
                    <Card>
                        <CardHeader>
                            <CardTitle>Footer Settings</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-6">
                            {/* Section Type */}
                            <div className="space-y-2">
                                <Label htmlFor="section_type">Section Type</Label>
                                <select
                                    id="section_type"
                                    value={data.section_type}
                                    onChange={(e) => setData('section_type', e.target.value)}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white"
                                    disabled
                                >
                                    {Object.entries(sectionTypes).map(([value, label]) => (
                                        <option key={value} value={value}>
                                            {label}
                                        </option>
                                    ))}
                                </select>
                                <p className="text-xs text-gray-500">
                                    Section type cannot be changed. Delete and create a new item to change it.
                                </p>
                                {errors.section_type && (
                                    <p className="text-sm text-red-600">{errors.section_type}</p>
                                )}
                            </div>

                            {/* Dynamic Data Fields */}
                            {renderDataFields()}

                            {/* Order */}
                            <div className="space-y-2">
                                <Label htmlFor="order">Display Order</Label>
                                <Input
                                    id="order"
                                    type="number"
                                    value={data.order}
                                    onChange={(e) => setData('order', parseInt(e.target.value))}
                                    min="0"
                                />
                                <p className="text-xs text-gray-500">
                                    Lower numbers appear first
                                </p>
                                {errors.order && (
                                    <p className="text-sm text-red-600">{errors.order}</p>
                                )}
                            </div>

                            {/* Active Status */}
                            <div className="flex items-center gap-2">
                                <input
                                    type="checkbox"
                                    id="is_active"
                                    checked={data.is_active}
                                    onChange={(e) => setData('is_active', e.target.checked)}
                                    className="w-4 h-4 rounded border-gray-300"
                                />
                                <Label htmlFor="is_active" className="cursor-pointer">
                                    Active (visible on website)
                                </Label>
                            </div>

                            {/* Submit Button */}
                            <div className="flex items-center gap-3 pt-4 border-t">
                                <Button
                                    type="submit"
                                    disabled={processing}
                                    className="flex items-center gap-2"
                                >
                                    <Save className="h-4 w-4" />
                                    {processing ? 'Saving...' : 'Save Changes'}
                                </Button>
                                <Button
                                    type="button"
                                    variant="outline"
                                    onClick={() => router.get(route('admin.cms.footer.index'))}
                                >
                                    Cancel
                                </Button>
                            </div>
                        </CardContent>
                    </Card>
                </form>
            </div>
        </AdminLayout>
    );
}

