import React, { useState, useEffect, useRef } from 'react';
import { Head, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Badge } from '@/Components/ui/Badge';
import { Input } from '@/Components/ui/Input';
import { 
    Plus, 
    FileText, 
    Image, 
    Search, 
    TrendingUp, 
    Users, 
    Eye,
    Edit,
    Trash2,
    Copy,
    MoreVertical,
    Filter,
    Calendar,
    Globe,
    BarChart3,
    Sparkles,
    Zap,
    Shield,
    Clock,
    ArrowUpRight,
    ChevronDown
} from 'lucide-react';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
    DropdownMenuTrigger,
} from '@/Components/ui/dropdown-menu';

export default function CMSDashboard({ pages = { data: [] }, stats = {}, filters: initialFilters = {} }) {
    const [searchTerm, setSearchTerm] = useState(initialFilters.search || '');
    const [filterStatus, setFilterStatus] = useState(initialFilters.status || 'all');
    const [filterType, setFilterType] = useState(initialFilters.page_type || 'all');
    const [showFilters, setShowFilters] = useState(false);

    // Apply filters with server-side request
    const applyFilters = (newSearch = searchTerm, newStatus = filterStatus, newType = filterType) => {
        router.get('/admin/cms/pages', {
            search: newSearch || undefined,
            status: newStatus !== 'all' ? newStatus : undefined,
            page_type: newType !== 'all' ? newType : undefined,
        }, {
            preserveState: true,
            preserveScroll: true,
            only: ['pages', 'filters'],
        });
    };

    // Debounced search
    useEffect(() => {
        const timer = setTimeout(() => {
            if (searchTerm !== (initialFilters.search || '')) {
                applyFilters(searchTerm, filterStatus, filterType);
            }
        }, 500);
        return () => clearTimeout(timer);
    }, [searchTerm]);

    // Immediate filter application for status and type (skip initial mount)
    const isInitialMount = useRef(true);
    useEffect(() => {
        if (isInitialMount.current) {
            isInitialMount.current = false;
            return;
        }
        if (filterStatus !== (initialFilters.status || 'all') || filterType !== (initialFilters.page_type || 'all')) {
            applyFilters(searchTerm, filterStatus, filterType);
        }
    }, [filterStatus, filterType]);

    // Use pages from server (already filtered)
    const filteredPages = pages.data || [];

    const getStatusColor = (status) => {
        switch (status) {
            case 'published': 
                return 'bg-emerald-100 text-emerald-800 dark:bg-emerald-900/20 dark:text-emerald-400 border-emerald-200 dark:border-emerald-800';
            case 'draft': 
                return 'bg-amber-100 text-amber-800 dark:bg-amber-900/20 dark:text-amber-400 border-amber-200 dark:border-amber-800';
            case 'archived': 
                return 'bg-slate-100 text-slate-800 dark:bg-slate-800 dark:text-slate-300 border-slate-200 dark:border-slate-700';
            default: 
                return 'bg-slate-100 text-slate-800 dark:bg-slate-800 dark:text-slate-300 border-slate-200 dark:border-slate-700';
        }
    };

    const getTypeIcon = (type) => {
        switch (type) {
            case 'homepage': return '🏠';
            case 'service': return '⚙️';
            case 'blog': return '📝';
            case 'contact': return '📞';
            case 'about': return 'ℹ️';
            case 'landing': return '🎯';
            case 'pricing': return '💰';
            case 'guarantees': return '🛡️';
            case 'reviews': return '⭐';
            case 'writers': return '✍️';
            default: return '📄';
        }
    };

    const getTypeColor = (type) => {
        switch (type) {
            case 'homepage': return 'bg-blue-500';
            case 'service': return 'bg-purple-500';
            case 'blog': return 'bg-green-500';
            case 'contact': return 'bg-orange-500';
            case 'about': return 'bg-indigo-500';
            case 'landing': return 'bg-pink-500';
            case 'pricing': return 'bg-emerald-500';
            case 'guarantees': return 'bg-cyan-500';
            case 'reviews': return 'bg-yellow-500';
            case 'writers': return 'bg-rose-500';
            default: return 'bg-gray-500';
        }
    };

    return (
        <AdminLayout>
            <Head title="CMS Dashboard" />
            
            <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-50 dark:from-slate-900 dark:via-slate-800 dark:to-slate-900">
                <div className="space-y-8 p-6">
                    {/* Enhanced Header */}
                    <div className="relative">
                        <div className="absolute inset-0 bg-gradient-to-r from-blue-600/10 via-purple-600/10 to-indigo-600/10 dark:from-blue-400/5 dark:via-purple-400/5 dark:to-indigo-400/5 rounded-2xl blur-3xl"></div>
                        <div className="relative bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border border-white/20 dark:border-slate-700/50 rounded-2xl p-8 shadow-xl">
                            <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-6">
                                <div className="space-y-2">
                                    <div className="flex items-center gap-3">
                                        <div className="p-2 bg-gradient-to-r from-blue-600 to-purple-600 rounded-xl">
                                            <Sparkles className="h-6 w-6 text-white" />
                                        </div>
                                        <h1 className="text-4xl font-bold bg-gradient-to-r from-slate-900 via-blue-900 to-purple-900 dark:from-slate-100 dark:via-blue-100 dark:to-purple-100 bg-clip-text text-transparent">
                                            Content Management
                                        </h1>
                                    </div>
                                    <p className="text-slate-600 dark:text-slate-400 text-lg">
                                        Manage your website pages, content, and SEO with precision
                                    </p>
                                </div>
                                <div className="flex gap-3">
                                    <Button 
                                        asChild 
                                        className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white dark:text-white shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-105 border-0 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 dark:focus:ring-offset-slate-800"
                                    >
                                        <a href="/admin/cms/pages/create" className="flex items-center justify-center">
                                            <Plus className="h-5 w-5 mr-2" />
                                            Create New Page
                                        </a>
                                    </Button>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Enhanced Stats Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                        <Card className="group hover:shadow-xl transition-all duration-300 transform hover:scale-105 bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border-white/20 dark:border-slate-700/50">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-slate-600 dark:text-slate-400">Total Pages</CardTitle>
                                <div className="p-2 bg-blue-100 dark:bg-blue-900/30 rounded-lg group-hover:bg-blue-200 dark:group-hover:bg-blue-900/50 transition-colors">
                                    <FileText className="h-4 w-4 text-blue-600 dark:text-blue-400" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-3xl font-bold text-slate-900 dark:text-slate-100">{stats.totalPages || 0}</div>
                                <p className="text-xs text-slate-500 dark:text-slate-400 mt-1">
                                    <span className="text-emerald-600 dark:text-emerald-400 font-medium">{stats.publishedPages || 0}</span> published
                                </p>
                            </CardContent>
                        </Card>

                        <Card className="group hover:shadow-xl transition-all duration-300 transform hover:scale-105 bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border-white/20 dark:border-slate-700/50">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-slate-600 dark:text-slate-400">Content Blocks</CardTitle>
                                <div className="p-2 bg-purple-100 dark:bg-purple-900/30 rounded-lg group-hover:bg-purple-200 dark:group-hover:bg-purple-900/50 transition-colors">
                                    <BarChart3 className="h-4 w-4 text-purple-600 dark:text-purple-400" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-3xl font-bold text-slate-900 dark:text-slate-100">{stats.totalContentBlocks || 0}</div>
                                <p className="text-xs text-slate-500 dark:text-slate-400 mt-1">
                                    <span className="text-purple-600 dark:text-purple-400 font-medium">{stats.activeContentBlocks || 0}</span> active
                                </p>
                            </CardContent>
                        </Card>

                        <Card className="group hover:shadow-xl transition-all duration-300 transform hover:scale-105 bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border-white/20 dark:border-slate-700/50">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-slate-600 dark:text-slate-400">Media Files</CardTitle>
                                <div className="p-2 bg-emerald-100 dark:bg-emerald-900/30 rounded-lg group-hover:bg-emerald-200 dark:group-hover:bg-emerald-900/50 transition-colors">
                                    <Image className="h-4 w-4 text-emerald-600 dark:text-emerald-400" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-3xl font-bold text-slate-900 dark:text-slate-100">{stats.totalMedia || 0}</div>
                                <p className="text-xs text-slate-500 dark:text-slate-400 mt-1">
                                    <span className="text-emerald-600 dark:text-emerald-400 font-medium">{stats.optimizedMedia || 0}</span> optimized
                                </p>
                            </CardContent>
                        </Card>

                        <Card className="group hover:shadow-xl transition-all duration-300 transform hover:scale-105 bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border-white/20 dark:border-slate-700/50">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-slate-600 dark:text-slate-400">SEO Score</CardTitle>
                                <div className="p-2 bg-orange-100 dark:bg-orange-900/30 rounded-lg group-hover:bg-orange-200 dark:group-hover:bg-orange-900/50 transition-colors">
                                    <TrendingUp className="h-4 w-4 text-orange-600 dark:text-orange-400" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-3xl font-bold text-slate-900 dark:text-slate-100">{stats.seoScore || 0}%</div>
                                <p className="text-xs text-slate-500 dark:text-slate-400 mt-1">
                                    Average SEO optimization
                                </p>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Enhanced Filters and Search */}
                    <Card className="bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border-white/20 dark:border-slate-700/50 shadow-xl">
                        <CardHeader className="pb-4">
                            <div className="flex items-center justify-between">
                                <div className="flex items-center gap-3">
                                    <div className="p-2 bg-gradient-to-r from-indigo-500 to-purple-500 rounded-lg">
                                        <FileText className="h-5 w-5 text-white" />
                                    </div>
                                    <CardTitle className="text-xl font-semibold text-slate-900 dark:text-slate-100">Pages</CardTitle>
                                </div>
                                <Button
                                    variant="outline"
                                    size="sm"
                                    onClick={() => setShowFilters(!showFilters)}
                                    className="border-slate-200 dark:border-slate-700 hover:bg-slate-50 dark:hover:bg-slate-700"
                                >
                                    <Filter className="h-4 w-4 mr-2" />
                                    Filters
                                    <ChevronDown className={`h-4 w-4 ml-2 transition-transform ${showFilters ? 'rotate-180' : ''}`} />
                                </Button>
                            </div>
                            
                            <div className="flex flex-col sm:flex-row gap-4 mt-6">
                                <div className="relative flex-1">
                                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-slate-400 h-4 w-4" />
                                    <Input
                                        type="text"
                                        placeholder="Search pages by title or slug..."
                                        value={searchTerm}
                                        onChange={(e) => setSearchTerm(e.target.value)}
                                        className="pl-10 bg-white/50 dark:bg-slate-700/50 border-slate-200 dark:border-slate-600 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                    />
                                </div>
                            </div>

                            {showFilters && (
                                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 mt-4 p-4 bg-slate-50 dark:bg-slate-700/50 rounded-lg border border-slate-200 dark:border-slate-600">
                                    <div>
                                        <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-2">Status</label>
                                        <select
                                            value={filterStatus}
                                            onChange={(e) => setFilterStatus(e.target.value)}
                                            className="w-full px-3 py-2 bg-white dark:bg-slate-700 border border-slate-200 dark:border-slate-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-slate-900 dark:text-slate-100"
                                        >
                                            <option value="all">All Status</option>
                                            <option value="published">Published</option>
                                            <option value="draft">Draft</option>
                                            <option value="archived">Archived</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-2">Type</label>
                                        <select
                                            value={filterType}
                                            onChange={(e) => setFilterType(e.target.value)}
                                            className="w-full px-3 py-2 bg-white dark:bg-slate-700 border border-slate-200 dark:border-slate-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-slate-900 dark:text-slate-100"
                                        >
                                            <option value="all">All Types</option>
                                            <option value="homepage">Homepage</option>
                                            <option value="service">Service</option>
                                            <option value="blog">Blog</option>
                                            <option value="contact">Contact</option>
                                            <option value="about">About</option>
                                            <option value="pricing">Pricing</option>
                                            <option value="guarantees">Guarantees</option>
                                            <option value="reviews">Reviews</option>
                                            <option value="writers">Writers</option>
                                            <option value="landing">Landing</option>
                                        </select>
                                    </div>
                                </div>
                            )}
                        </CardHeader>
                        
                        <CardContent>
                            {filteredPages.length === 0 ? (
                                <div className="text-center py-16">
                                    <div className="p-4 bg-slate-100 dark:bg-slate-700 rounded-full w-20 h-20 mx-auto mb-6 flex items-center justify-center">
                                        <FileText className="h-10 w-10 text-slate-400" />
                                    </div>
                                    <h3 className="text-xl font-semibold text-slate-900 dark:text-slate-100 mb-2">No pages found</h3>
                                    <p className="text-slate-500 dark:text-slate-400 mb-6 max-w-md mx-auto">
                                        {searchTerm || filterStatus !== 'all' || filterType !== 'all'
                                            ? 'Try adjusting your search or filters to find what you\'re looking for'
                                            : 'Get started by creating your first page to build your website'
                                        }
                                    </p>
                                    {(!searchTerm && filterStatus === 'all' && filterType === 'all') && (
                                        <Button 
                                            asChild
                                            className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white dark:text-white shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-105 border-0 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 dark:focus:ring-offset-slate-800"
                                        >
                                            <a href="/admin/cms/pages/create" className="flex items-center justify-center">
                                                <Plus className="h-5 w-5 mr-2" />
                                                Create Your First Page
                                            </a>
                                        </Button>
                                    )}
                                </div>
                            ) : (
                                <div className="space-y-3">
                                    {filteredPages.map((page, index) => (
                                        <div
                                            key={page.id}
                                            className="group flex items-center justify-between p-6 bg-white/50 dark:bg-slate-700/50 border border-slate-200 dark:border-slate-600 rounded-xl hover:bg-white dark:hover:bg-slate-700 hover:shadow-lg transition-all duration-300 transform hover:scale-[1.02] relative z-10"
                                            style={{ animationDelay: `${index * 50}ms` }}
                                        >
                                            <div className="flex items-center space-x-4">
                                                <div className="relative">
                                                    <div className={`w-12 h-12 ${getTypeColor(page.page_type)} rounded-xl flex items-center justify-center text-white text-xl shadow-lg`}>
                                                        {getTypeIcon(page.page_type)}
                                                    </div>
                                                    <div className="absolute -top-1 -right-1 w-4 h-4 bg-white dark:bg-slate-800 rounded-full border-2 border-slate-200 dark:border-slate-600"></div>
                                                </div>
                                                <div className="space-y-1">
                                                    <h3 className="font-semibold text-slate-900 dark:text-slate-100 text-lg group-hover:text-blue-600 dark:group-hover:text-blue-400 transition-colors">
                                                        {page.title}
                                                    </h3>
                                                    <div className="flex items-center space-x-3">
                                                        <Badge className={`${getStatusColor(page.status)} border font-medium`}>
                                                            {page.status}
                                                        </Badge>
                                                        <span className="text-sm text-slate-500 dark:text-slate-400 font-mono">
                                                            /{page.slug}
                                                        </span>
                                                        <span className="text-sm text-slate-500 dark:text-slate-400">
                                                            • {page.page_type}
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div className="flex items-center space-x-4">
                                                <div className="text-right">
                                                    <div className="flex items-center text-sm text-slate-500 dark:text-slate-400 mb-1">
                                                        <Calendar className="h-4 w-4 mr-1" />
                                                        Updated
                                                    </div>
                                                    <div className="text-sm font-medium text-slate-900 dark:text-slate-100">
                                                        {new Date(page.updated_at).toLocaleDateString()}
                                                    </div>
                                                </div>
                                                
                                                <DropdownMenu>
                                                    <DropdownMenuTrigger asChild>
                                                        <Button 
                                                            variant="ghost" 
                                                            size="sm"
                                                            className="hover:bg-slate-100 dark:hover:bg-slate-600 transition-colors"
                                                        >
                                                            <MoreVertical className="h-4 w-4" />
                                                        </Button>
                                                    </DropdownMenuTrigger>
                                                    <DropdownMenuContent align="end" className="w-48 z-50 relative">
                                                        <a href={`${page.page_type === 'service' ? '/services' : ''}/${page.slug}`} target="_blank" rel="noopener noreferrer">
                                                            <DropdownMenuItem className="hover:bg-blue-50 dark:hover:bg-blue-900/20">
                                                                <Eye className="h-4 w-4 mr-2 text-blue-600 dark:text-blue-400" />
                                                                <span className="text-blue-600 dark:text-blue-400">View Live</span>
                                                                <ArrowUpRight className="h-3 w-3 ml-auto" />
                                                            </DropdownMenuItem>
                                                        </a>
                                                        <a href={`/admin/cms/pages/${page.id}/edit`}>
                                                            <DropdownMenuItem className="hover:bg-purple-50 dark:hover:bg-purple-900/20">
                                                                <Edit className="h-4 w-4 mr-2 text-purple-600 dark:text-purple-400" />
                                                                <span className="text-purple-600 dark:text-purple-400">Edit Content</span>
                                                            </DropdownMenuItem>
                                                        </a>
                                                        <a href={`/admin/cms/pages/${page.id}`}>
                                                            <DropdownMenuItem className="hover:bg-indigo-50 dark:hover:bg-indigo-900/20">
                                                                <BarChart3 className="h-4 w-4 mr-2 text-indigo-600 dark:text-indigo-400" />
                                                                <span className="text-indigo-600 dark:text-indigo-400">View Details</span>
                                                            </DropdownMenuItem>
                                                        </a>
                                                        <DropdownMenuSeparator />
                                                        <DropdownMenuItem 
                                                            onClick={() => {
                                                                const newTitle = prompt('Enter new title for duplicated page:', `${page.title} (Copy)`);
                                                                if (newTitle) {
                                                                    fetch(`/admin/cms/pages/${page.id}/duplicate`, {
                                                                        method: 'POST',
                                                                        headers: {
                                                                            'Content-Type': 'application/json',
                                                                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                                                                        },
                                                                        body: JSON.stringify({ new_title: newTitle })
                                                                    })
                                                                    .then(response => response.json())
                                                                    .then(data => {
                                                                        if (data.success) {
                                                                            window.location.reload();
                                                                        } else {
                                                                            alert('Failed to duplicate page: ' + data.message);
                                                                        }
                                                                    });
                                                                }
                                                            }}
                                                            className="hover:bg-green-50 dark:hover:bg-green-900/20"
                                                        >
                                                            <Copy className="h-4 w-4 mr-2 text-green-600 dark:text-green-400" />
                                                            <span className="text-green-600 dark:text-green-400">Duplicate</span>
                                                        </DropdownMenuItem>
                                                        <DropdownMenuSeparator />
                                                        <DropdownMenuItem 
                                                            className="text-red-600 dark:text-red-400 hover:bg-red-50 dark:hover:bg-red-900/20"
                                                            onClick={() => {
                                                                if (confirm('Are you sure you want to delete this page? This action cannot be undone.')) {
                                                                    fetch(`/admin/cms/pages/${page.id}`, {
                                                                        method: 'DELETE',
                                                                        headers: {
                                                                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                                                                        }
                                                                    })
                                                                    .then(response => response.json())
                                                                    .then(data => {
                                                                        if (data.success) {
                                                                            window.location.reload();
                                                                        } else {
                                                                            alert('Failed to delete page: ' + data.message);
                                                                        }
                                                                    });
                                                                }
                                                            }}
                                                        >
                                                            <Trash2 className="h-4 w-4 mr-2" />
                                                            Delete Page
                                                        </DropdownMenuItem>
                                                    </DropdownMenuContent>
                                                </DropdownMenu>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            )}

                            {/* Pagination Controls */}
                            {pages.links && pages.links.length > 3 && (
                                <div className="flex items-center justify-between mt-6 pt-6 border-t border-slate-200 dark:border-slate-700">
                                    <div className="text-sm text-slate-600 dark:text-slate-400">
                                        Showing {pages.from || 0} to {pages.to || 0} of {pages.total || 0} pages
                                    </div>
                                    <div className="flex items-center space-x-2">
                                        {pages.links.map((link, index) => {
                                            if (index === 0) {
                                                // Previous button
                                                return (
                                                    <button
                                                        key={index}
                                                        onClick={() => link.url && router.get(link.url, {}, { preserveState: true, preserveScroll: true })}
                                                        disabled={!link.url}
                                                        className={`px-3 py-2 text-sm font-medium rounded-lg border transition-colors ${
                                                            link.url
                                                                ? 'bg-white dark:bg-slate-700 border-slate-300 dark:border-slate-600 text-slate-700 dark:text-slate-300 hover:bg-slate-50 dark:hover:bg-slate-600'
                                                                : 'bg-slate-100 dark:bg-slate-800 border-slate-200 dark:border-slate-700 text-slate-400 dark:text-slate-500 cursor-not-allowed'
                                                        }`}
                                                    >
                                                        Previous
                                                    </button>
                                                );
                                            } else if (index === pages.links.length - 1) {
                                                // Next button
                                                return (
                                                    <button
                                                        key={index}
                                                        onClick={() => link.url && router.get(link.url, {}, { preserveState: true, preserveScroll: true })}
                                                        disabled={!link.url}
                                                        className={`px-3 py-2 text-sm font-medium rounded-lg border transition-colors ${
                                                            link.url
                                                                ? 'bg-white dark:bg-slate-700 border-slate-300 dark:border-slate-600 text-slate-700 dark:text-slate-300 hover:bg-slate-50 dark:hover:bg-slate-600'
                                                                : 'bg-slate-100 dark:bg-slate-800 border-slate-200 dark:border-slate-700 text-slate-400 dark:text-slate-500 cursor-not-allowed'
                                                        }`}
                                                    >
                                                        Next
                                                    </button>
                                                );
                                            } else if (link.label === '...') {
                                                // Ellipsis
                                                return (
                                                    <span key={index} className="px-3 py-2 text-slate-400 dark:text-slate-500">
                                                        ...
                                                    </span>
                                                );
                                            } else {
                                                // Page number
                                                return (
                                                    <button
                                                        key={index}
                                                        onClick={() => link.url && router.get(link.url, {}, { preserveState: true, preserveScroll: true })}
                                                        className={`px-3 py-2 text-sm font-medium rounded-lg border transition-colors ${
                                                            link.active
                                                                ? 'bg-blue-600 text-white border-blue-600'
                                                                : 'bg-white dark:bg-slate-700 border-slate-300 dark:border-slate-600 text-slate-700 dark:text-slate-300 hover:bg-slate-50 dark:hover:bg-slate-600'
                                                        }`}
                                                    >
                                                        {link.label}
                                                    </button>
                                                );
                                            }
                                        })}
                                    </div>
                                </div>
                            )}
                        </CardContent>
                    </Card>

                    {/* Enhanced Quick Actions */}
                    <Card className="bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border-white/20 dark:border-slate-700/50 shadow-xl">
                        <CardHeader>
                            <div className="flex items-center gap-3">
                                <div className="p-2 bg-gradient-to-r from-emerald-500 to-teal-500 rounded-lg">
                                    <Zap className="h-5 w-5 text-white" />
                                </div>
                                <CardTitle className="text-xl font-semibold text-slate-900 dark:text-slate-100">Quick Actions</CardTitle>
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                                <Button variant="outline" asChild className="h-auto p-6 group hover:shadow-lg transition-all duration-300 transform hover:scale-105 bg-white/50 dark:bg-slate-700/50 border-slate-200 dark:border-slate-600 hover:border-emerald-300 dark:hover:border-emerald-600">
                                    <a href="/admin/cms/media">
                                        <div className="text-left space-y-3">
                                            <div className="p-3 bg-emerald-100 dark:bg-emerald-900/30 rounded-xl w-fit group-hover:bg-emerald-200 dark:group-hover:bg-emerald-900/50 transition-colors">
                                                <Image className="h-6 w-6 text-emerald-600 dark:text-emerald-400" />
                                            </div>
                                            <div>
                                                <div className="font-semibold text-slate-900 dark:text-slate-100">Media Library</div>
                                                <div className="text-sm text-slate-500 dark:text-slate-400">Manage images, videos, and files</div>
                                            </div>
                                        </div>
                                    </a>
                                </Button>
                                
                                <Button variant="outline" asChild className="h-auto p-6 group hover:shadow-lg transition-all duration-300 transform hover:scale-105 bg-white/50 dark:bg-slate-700/50 border-slate-200 dark:border-slate-600 hover:border-orange-300 dark:hover:border-orange-600">
                                    <a href="/admin/cms/seo">
                                        <div className="text-left space-y-3">
                                            <div className="p-3 bg-orange-100 dark:bg-orange-900/30 rounded-xl w-fit group-hover:bg-orange-200 dark:group-hover:bg-orange-900/50 transition-colors">
                                                <TrendingUp className="h-6 w-6 text-orange-600 dark:text-orange-400" />
                                            </div>
                                            <div>
                                                <div className="font-semibold text-slate-900 dark:text-slate-100">SEO Overview</div>
                                                <div className="text-sm text-slate-500 dark:text-slate-400">Review SEO performance and metrics</div>
                                            </div>
                                        </div>
                                    </a>
                                </Button>
                                
                                <Button variant="outline" asChild className="h-auto p-6 group hover:shadow-lg transition-all duration-300 transform hover:scale-105 bg-white/50 dark:bg-slate-700/50 border-slate-200 dark:border-slate-600 hover:border-purple-300 dark:hover:border-purple-600">
                                    <a href="/admin/cms/content-blocks">
                                        <div className="text-left space-y-3">
                                            <div className="p-3 bg-purple-100 dark:bg-purple-900/30 rounded-xl w-fit group-hover:bg-purple-200 dark:group-hover:bg-purple-900/50 transition-colors">
                                                <BarChart3 className="h-6 w-6 text-purple-600 dark:text-purple-400" />
                                            </div>
                                            <div>
                                                <div className="font-semibold text-slate-900 dark:text-slate-100">Content Blocks</div>
                                                <div className="text-sm text-slate-500 dark:text-slate-400">Manage reusable content components</div>
                                            </div>
                                        </div>
                                    </a>
                                </Button>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </AdminLayout>
    );
}
