import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Badge } from '@/Components/ui/Badge';
import { 
    Upload, 
    Image, 
    Video, 
    File, 
    Search, 
    Filter,
    Download,
    Trash2,
    Edit,
    Eye,
    RefreshCw,
    CheckCircle,
    AlertCircle,
    MoreVertical,
    Plus
} from 'lucide-react';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
    DropdownMenuTrigger,
} from '@/Components/ui/dropdown-menu';

export default function MediaLibrary({ pageId, mediaFiles, onUpdate }) {
    const [media, setMedia] = useState(mediaFiles || []);
    const [selectedFiles, setSelectedFiles] = useState([]);
    const [searchTerm, setSearchTerm] = useState('');
    const [filterType, setFilterType] = useState('all');
    const [filterContext, setFilterContext] = useState('all');
    const [uploading, setUploading] = useState(false);
    const [uploadProgress, setUploadProgress] = useState(0);

    // Filter media based on search and filters
    const filteredMedia = media.filter(file => {
        const matchesSearch = file.title?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                             file.alt_text?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                             file.caption?.toLowerCase().includes(searchTerm.toLowerCase());
        const matchesType = filterType === 'all' || file.media_type === filterType;
        const matchesContext = filterContext === 'all' || file.usage_context === filterContext;
        
        return matchesSearch && matchesType && matchesContext;
    });

    const handleFileUpload = async (event) => {
        const files = Array.from(event.target.files);
        setUploading(true);
        setUploadProgress(0);

        try {
            for (let i = 0; i < files.length; i++) {
                const file = files[i];
                const formData = new FormData();
                formData.append('file', file);
                formData.append('page_id', pageId);
                formData.append('usage_context', 'general');

                const response = await fetch('/admin/cms/media', {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                    },
                    body: formData
                });

                const data = await response.json();
                
                if (data.success) {
                    setMedia(prev => [data.media, ...prev]);
                    onUpdate([data.media, ...media]);
                } else {
                    alert(`Failed to upload ${file.name}: ${data.message}`);
                }

                setUploadProgress(((i + 1) / files.length) * 100);
            }
            
            alert('Files uploaded successfully!');
        } catch (error) {
            alert('Upload failed: ' + error.message);
        } finally {
            setUploading(false);
            setUploadProgress(0);
            event.target.value = '';
        }
    };

    const handleDeleteMedia = async (mediaId) => {
        if (!confirm('Are you sure you want to delete this media file?')) {
            return;
        }

        try {
            const response = await fetch(`/admin/cms/media/${mediaId}`, {
                method: 'DELETE',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                }
            });

            const data = await response.json();
            
            if (data.success) {
                setMedia(prev => prev.filter(file => file.id !== mediaId));
                onUpdate(media.filter(file => file.id !== mediaId));
                alert('Media deleted successfully!');
            } else {
                alert('Failed to delete media: ' + data.message);
            }
        } catch (error) {
            alert('Delete failed: ' + error.message);
        }
    };

    const handleBulkDelete = async () => {
        if (selectedFiles.length === 0) {
            alert('Please select files to delete');
            return;
        }

        if (!confirm(`Are you sure you want to delete ${selectedFiles.length} files?`)) {
            return;
        }

        try {
            const response = await fetch('/admin/cms/media/bulk-delete', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                },
                body: JSON.stringify({
                    media_ids: selectedFiles
                })
            });

            const data = await response.json();
            
            if (data.success) {
                setMedia(prev => prev.filter(file => !selectedFiles.includes(file.id)));
                onUpdate(media.filter(file => !selectedFiles.includes(file.id)));
                setSelectedFiles([]);
                alert(`Successfully deleted ${data.deleted_count} files`);
            } else {
                alert('Failed to delete files: ' + data.message);
            }
        } catch (error) {
            alert('Bulk delete failed: ' + error.message);
        }
    };

    const handleOptimizeImages = async () => {
        try {
            const response = await fetch('/admin/cms/media/optimize', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                },
                body: JSON.stringify({
                    page_id: pageId
                })
            });

            const data = await response.json();
            
            if (data.success) {
                alert(`Optimization completed: ${data.results.success} successful, ${data.results.failed} failed`);
                // Refresh media list
                window.location.reload();
            } else {
                alert('Failed to optimize images: ' + data.message);
            }
        } catch (error) {
            alert('Optimization failed: ' + error.message);
        }
    };

    const updateMediaMetadata = async (mediaId, metadata) => {
        try {
            const response = await fetch(`/admin/cms/media/${mediaId}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                },
                body: JSON.stringify(metadata)
            });

            const data = await response.json();
            
            if (data.success) {
                setMedia(prev => prev.map(file => 
                    file.id === mediaId ? { ...file, ...metadata } : file
                ));
                onUpdate(media.map(file => 
                    file.id === mediaId ? { ...file, ...metadata } : file
                ));
                alert('Media updated successfully!');
            } else {
                alert('Failed to update media: ' + data.message);
            }
        } catch (error) {
            alert('Update failed: ' + error.message);
        }
    };

    const getMediaIcon = (mediaType) => {
        switch (mediaType) {
            case 'image': return <Image className="h-8 w-8 text-blue-500" />;
            case 'video': return <Video className="h-8 w-8 text-purple-500" />;
            case 'document': return <File className="h-8 w-8 text-gray-500" />;
            default: return <File className="h-8 w-8 text-gray-500" />;
        }
    };

    const getFileSize = (bytes) => {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    };

    const toggleFileSelection = (fileId) => {
        setSelectedFiles(prev => 
            prev.includes(fileId) 
                ? prev.filter(id => id !== fileId)
                : [...prev, fileId]
        );
    };

    const selectAllFiles = () => {
        setSelectedFiles(filteredMedia.map(file => file.id));
    };

    const clearSelection = () => {
        setSelectedFiles([]);
    };

    return (
        <div className="space-y-6">
            {/* Upload Section */}
            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center space-x-2">
                        <Upload className="h-5 w-5" />
                        <span>Upload Media</span>
                    </CardTitle>
                </CardHeader>
                <CardContent>
                    <div className="border-2 border-dashed border-gray-300 rounded-lg p-6">
                        <div className="text-center">
                            <Upload className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                            <div className="space-y-2">
                                <h3 className="text-lg font-medium text-gray-900">
                                    Upload files
                                </h3>
                                <p className="text-gray-500">
                                    Drag and drop files here, or click to select
                                </p>
                                <input
                                    type="file"
                                    multiple
                                    onChange={handleFileUpload}
                                    className="hidden"
                                    id="file-upload"
                                    accept="image/*,video/*,.pdf,.doc,.docx"
                                />
                                <Button asChild>
                                    <label htmlFor="file-upload" className="cursor-pointer">
                                        <Plus className="h-4 w-4 mr-2" />
                                        Select Files
                                    </label>
                                </Button>
                            </div>
                        </div>
                        
                        {uploading && (
                            <div className="mt-4">
                                <div className="flex justify-between items-center mb-2">
                                    <span className="text-sm text-gray-600">Uploading...</span>
                                    <span className="text-sm text-gray-600">{Math.round(uploadProgress)}%</span>
                                </div>
                                <div className="w-full bg-gray-200 rounded-full h-2">
                                    <div 
                                        className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                        style={{ width: `${uploadProgress}%` }}
                                    ></div>
                                </div>
                            </div>
                        )}
                    </div>
                </CardContent>
            </Card>

            {/* Filters and Actions */}
            <Card>
                <CardContent className="pt-6">
                    <div className="flex flex-col sm:flex-row gap-4">
                        <div className="relative flex-1">
                            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                            <Input
                                placeholder="Search media..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                                className="pl-10"
                            />
                        </div>
                        
                        <select
                            value={filterType}
                            onChange={(e) => setFilterType(e.target.value)}
                            className="px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        >
                            <option value="all">All Types</option>
                            <option value="image">Images</option>
                            <option value="video">Videos</option>
                            <option value="document">Documents</option>
                        </select>
                        
                        <select
                            value={filterContext}
                            onChange={(e) => setFilterContext(e.target.value)}
                            className="px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        >
                            <option value="all">All Contexts</option>
                            <option value="hero">Hero</option>
                            <option value="benefits">Benefits</option>
                            <option value="process">Process</option>
                            <option value="general">General</option>
                        </select>
                    </div>
                    
                    {selectedFiles.length > 0 && (
                        <div className="flex items-center justify-between mt-4 p-3 bg-blue-50 rounded-lg">
                            <span className="text-sm text-blue-700">
                                {selectedFiles.length} files selected
                            </span>
                            <div className="flex space-x-2">
                                <Button variant="outline" size="sm" onClick={clearSelection}>
                                    Clear Selection
                                </Button>
                                <Button 
                                    variant="destructive" 
                                    size="sm" 
                                    onClick={handleBulkDelete}
                                >
                                    <Trash2 className="h-4 w-4 mr-2" />
                                    Delete Selected
                                </Button>
                            </div>
                        </div>
                    )}
                </CardContent>
            </Card>

            {/* Media Grid */}
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {filteredMedia.map((file) => (
                    <Card key={file.id} className={`cursor-pointer transition-all duration-200 ${
                        selectedFiles.includes(file.id) ? 'ring-2 ring-blue-500' : ''
                    }`}>
                        <CardContent className="p-4">
                            <div className="space-y-3">
                                {/* Media Preview */}
                                <div className="relative">
                                    <div className="aspect-square bg-gray-100 rounded-lg flex items-center justify-center">
                                        {file.media_type === 'image' ? (
                                            <img
                                                src={file.file_path}
                                                alt={file.alt_text || file.title}
                                                className="w-full h-full object-cover rounded-lg"
                                            />
                                        ) : (
                                            getMediaIcon(file.media_type)
                                        )}
                                    </div>
                                    
                                    {/* Selection Checkbox */}
                                    <div className="absolute top-2 left-2">
                                        <input
                                            type="checkbox"
                                            checked={selectedFiles.includes(file.id)}
                                            onChange={() => toggleFileSelection(file.id)}
                                            className="w-4 h-4 text-blue-600 bg-white border-gray-300 rounded focus:ring-blue-500"
                                        />
                                    </div>
                                    
                                    {/* Status Badge */}
                                    <div className="absolute top-2 right-2">
                                        {file.is_optimized ? (
                                            <Badge variant="default" className="text-xs">
                                                <CheckCircle className="h-3 w-3 mr-1" />
                                                Optimized
                                            </Badge>
                                        ) : (
                                            <Badge variant="secondary" className="text-xs">
                                                <AlertCircle className="h-3 w-3 mr-1" />
                                                Raw
                                            </Badge>
                                        )}
                                    </div>
                                </div>

                                {/* File Info */}
                                <div className="space-y-1">
                                    <h4 className="font-medium text-sm truncate">
                                        {file.title || 'Untitled'}
                                    </h4>
                                    <p className="text-xs text-gray-500">
                                        {getFileSize(file.file_size)}
                                    </p>
                                    <div className="flex items-center justify-between">
                                        <Badge variant="outline" className="text-xs">
                                            {file.media_type}
                                        </Badge>
                                        <span className="text-xs text-gray-500">
                                            {new Date(file.created_at).toLocaleDateString()}
                                        </span>
                                    </div>
                                </div>

                                {/* Actions */}
                                <DropdownMenu>
                                    <DropdownMenuTrigger asChild>
                                        <Button variant="ghost" size="sm" className="w-full">
                                            <MoreVertical className="h-4 w-4 mr-2" />
                                            Actions
                                        </Button>
                                    </DropdownMenuTrigger>
                                    <DropdownMenuContent align="end">
                                        <DropdownMenuItem asChild>
                                            <a href={file.file_path} target="_blank" rel="noopener noreferrer">
                                                <Eye className="h-4 w-4 mr-2" />
                                                View
                                            </a>
                                        </DropdownMenuItem>
                                        <DropdownMenuItem asChild>
                                            <a href={file.file_path} download>
                                                <Download className="h-4 w-4 mr-2" />
                                                Download
                                            </a>
                                        </DropdownMenuItem>
                                        <DropdownMenuItem
                                            onClick={() => {
                                                const altText = prompt('Alt Text:', file.alt_text || '');
                                                const title = prompt('Title:', file.title || '');
                                                const caption = prompt('Caption:', file.caption || '');
                                                const context = prompt('Usage Context:', file.usage_context || '');
                                                
                                                if (altText !== null || title !== null || caption !== null || context !== null) {
                                                    updateMediaMetadata(file.id, {
                                                        alt_text: altText,
                                                        title: title,
                                                        caption: caption,
                                                        usage_context: context
                                                    });
                                                }
                                            }}
                                        >
                                            <Edit className="h-4 w-4 mr-2" />
                                            Edit Metadata
                                        </DropdownMenuItem>
                                        <DropdownMenuSeparator />
                                        <DropdownMenuItem 
                                            className="text-red-600"
                                            onClick={() => handleDeleteMedia(file.id)}
                                        >
                                            <Trash2 className="h-4 w-4 mr-2" />
                                            Delete
                                        </DropdownMenuItem>
                                    </DropdownMenuContent>
                                </DropdownMenu>
                            </div>
                        </CardContent>
                    </Card>
                ))}
            </div>

            {/* Empty State */}
            {filteredMedia.length === 0 && (
                <Card>
                    <CardContent className="text-center py-12">
                        <Image className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                        <h3 className="text-lg font-medium text-gray-900 mb-2">No media found</h3>
                        <p className="text-gray-500 mb-4">
                            {searchTerm || filterType !== 'all' || filterContext !== 'all'
                                ? 'Try adjusting your search or filters'
                                : 'Upload some media files to get started'
                            }
                        </p>
                        {(!searchTerm && filterType === 'all' && filterContext === 'all') && (
                            <Button asChild>
                                <label htmlFor="file-upload" className="cursor-pointer">
                                    <Upload className="h-4 w-4 mr-2" />
                                    Upload Files
                                </label>
                            </Button>
                        )}
                    </CardContent>
                </Card>
            )}

            {/* Quick Actions */}
            <Card>
                <CardHeader>
                    <CardTitle>Quick Actions</CardTitle>
                </CardHeader>
                <CardContent>
                    <div className="flex flex-wrap gap-2">
                        <Button variant="outline" size="sm" onClick={selectAllFiles}>
                            Select All Visible
                        </Button>
                        <Button variant="outline" size="sm" onClick={clearSelection}>
                            Clear Selection
                        </Button>
                        <Button variant="outline" size="sm" onClick={handleOptimizeImages}>
                            <RefreshCw className="h-4 w-4 mr-2" />
                            Optimize Images
                        </Button>
                    </div>
                </CardContent>
            </Card>
        </div>
    );
}
