import React, { useState, useEffect, useRef, useCallback } from 'react';
import { Head, useForm } from '@inertiajs/react';
import { toast } from 'react-hot-toast';
import AdminLayout from '@/Layouts/AdminLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Textarea } from '@/Components/ui/Textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/Components/ui/Select';
import { Badge } from '@/Components/ui/Badge';
import { 
    Save, 
    Eye, 
    ArrowLeft,
    Plus,
    Trash2,
    Move,
    Settings,
    Image,
    Type,
    Layout,
    Code,
    Star
} from 'lucide-react';
import ContentBlockEditor from '../Components/ContentBlockEditor';
import SEOEditor from '../SEO/SEOEditor';
import MediaLibrary from '../Media/MediaLibrary';

export default function EditPage({ page, pageTypes, templates, parentPages }) {
    const [activeTab, setActiveTab] = useState('content');
    const [contentBlocks, setContentBlocks] = useState(page.content_blocks || []);
    
    // Initialize SEO data - extract only the fields we need, excluding id, timestamps, page_id
    const initializeSeoData = () => {
        if (!page.seo) return {};
        const { id, page_id, created_at, updated_at, ...seoFields } = page.seo;
        return seoFields;
    };
    
    const [seoData, setSeoData] = useState(initializeSeoData());
    const [mediaFiles, setMediaFiles] = useState(page.media || []);
    const [hasUnsavedChanges, setHasUnsavedChanges] = useState(false);
    const [lastSavedBlock, setLastSavedBlock] = useState(null);
    const [auditLog, setAuditLog] = useState([]);
    
    // Track if component just mounted to avoid triggering unsaved changes on initial load
    const isInitialMount = useRef(true);

    const { data, setData, put, processing, errors } = useForm({
        title: page.title || '',
        slug: page.slug || '',
        page_type: page.page_type || 'service',
        status: page.status || 'draft',
        parent_id: page.parent_id || '',
        template: page.template || 'default',
        seo_priority: page.seo_priority || 0,
        published_at: page.published_at || '',
        content_blocks: contentBlocks,
        seo_data: seoData,
        media_files: mediaFiles
    });

    // Update form data when content blocks change
    useEffect(() => {
        setData('content_blocks', contentBlocks);
    }, [contentBlocks]);

    // Update form data when SEO data changes
    useEffect(() => {
        // Skip on initial mount
        if (isInitialMount.current) {
            isInitialMount.current = false;
            return;
        }
        
        console.log('SEO data changed:', seoData);
        setData('seo_data', seoData);
        setHasUnsavedChanges(true);
    }, [seoData]);

    // Update form data when media files change
    useEffect(() => {
        setData('media_files', mediaFiles);
    }, [mediaFiles]);

    const appendAuditEntry = useCallback((entry) => {
        setAuditLog((current) => {
            const next = [{ id: Date.now(), ...entry }, ...current];
            return next.slice(0, 25);
        });
    }, []);

    const describeBlock = useCallback((block, fallbackId) => {
        if (!block) {
            return `Block ${fallbackId}`;
        }

        const titleCandidates = [
            block.content?.title,
            block.content?.heading,
            block.block_key,
        ].filter((value) => typeof value === 'string' && value.trim().length > 0);

        const primary = titleCandidates.length > 0 ? titleCandidates[0].trim() : null;
        const typeLabel = block.block_type || `Block ${fallbackId}`;

        return primary ? `${primary} (${typeLabel})` : typeLabel;
    }, []);

    const handleSubmit = (e) => {
        e.preventDefault();
        
        console.log('Submitting form data:', {
            ...data,
            seo_data_fields: Object.keys(data.seo_data || {})
        });
        
        put(`/admin/cms/pages/${page.id}`, {
            preserveScroll: true,
            onSuccess: () => {
                setHasUnsavedChanges(false);
                setLastSavedBlock(null);
                isInitialMount.current = true;
                toast.success('Page updated successfully! Cache cleared.', {
                    duration: 3000,
                    position: 'top-right',
                });
                console.log('Page updated successfully');
            },
            onError: (errors) => {
                console.error('Update failed:', errors);
                toast.error('Failed to update page. Check console for errors.', {
                    duration: 5000,
                    position: 'top-right',
                });
            },
            onFinish: () => {
                console.log('Request finished');
            }
        });
    };

    const addContentBlock = (blockType) => {
        // Generate a unique block key
        let blockKey = `${blockType}_${Date.now()}`;
        
        // For scrollable sections, use a more descriptive key
        if (blockType === 'service_scrollable') {
            const scrollableCount = contentBlocks.filter(b => b.block_type === 'service_scrollable').length;
            blockKey = `${page.slug || 'service'}_deep_dive_${scrollableCount + 1}`;
        } else if (blockType.startsWith('service_')) {
            // For other service blocks, use the page slug
            blockKey = `${page.slug || 'service'}_${blockType.replace('service_', '')}`;
        }
        
        const newBlock = {
            id: Date.now(), // Temporary ID
            block_type: blockType,
            block_key: blockKey,
            content: getDefaultContent(blockType),
            order: contentBlocks.length,
            is_active: true
        };
        setContentBlocks([...contentBlocks, newBlock]);
    };

    const getDefaultContent = (blockType) => {
        const defaults = {
            hero: {
                title: '',
                subtitle: '',
                description: '',
                cta_text: '',
                cta_link: '',
                background_image: '',
                background_color: ''
            },
            overview: {
                title: '',
                description: '',
                features: [],
                image: ''
            },
            benefits: {
                title: '',
                subtitle: '',
                benefits: []
            },
            process: {
                title: '',
                subtitle: '',
                steps: []
            },
            testimonials: {
                title: '',
                subtitle: '',
                testimonials: []
            },
            faqs: {
                title: '',
                subtitle: '',
                faqs: []
            },
            service_hero: {
                title: '',
                description: '',
                description_html: '',
                keyPoints: [],
                ctaText: 'Get started',
                ctaLink: '/place-order',
                trustSignals: []
            },
            service_overview: {
                title: 'Service Overview',
                description: '',
                description_html: '',
                features: []
            },
            service_process: {
                title: 'How It Works',
                description: '',
                description_html: '',
                steps: []
            },
            service_benefits: {
                title: 'Key Benefits',
                description: '',
                description_html: '',
                benefits: [],
                ctaText: 'See pricing',
                ctaLink: '/pricing'
            },
            service_testimonials: {
                title: 'What Students Say',
                description: '',
                description_html: '',
                testimonials: [],
                trustMetrics: {}
            },
            service_faqs: {
                title: 'Frequently Asked Questions',
                description: '',
                description_html: '',
                faqs: []
            },
            service_cta: {
                title: 'Ready to Get Started?',
                description: '',
                description_html: '',
                benefits: [],
                primaryButtonText: 'Place Order',
                primaryButtonLink: '/place-order',
                secondaryButtonText: 'Contact Us',
                secondaryButtonLink: '/contact',
                bgColor: 'blue',
                urgencyText: '',
                guarantees: [],
                trustBadges: []
            },
            service_links: {
                title: 'Related Services',
                description: '',
                description_html: '',
                linkCategory: 'related',
                links: []
            },
            service_scrollable: {
                eyebrow: 'Deep Dive',
                title: '',
                description: '',
                description_html: '',
                preferredHeight: 480,
                sections: []
            },
            price_calculator: {
                eyebrow: '',
                title: 'Get Instant Pricing',
                subtitle: 'Calculate the cost of your assignment in seconds. No commitment required.',
                description: '',
                buttonText: 'Calculate Your Price',
                buttonVariant: 'primary',
                showEyebrow: true,
                showDescription: false,
                backgroundColor: 'gradient',
                alignment: 'center'
            }
        };
        return defaults[blockType] || {};
    };

    const updateContentBlock = (blockId, updatedContent) => {
        console.log('Edit.jsx: updateContentBlock called', { blockId, updatedContent });
        const targetBlock = contentBlocks.find((block) => block.id === blockId);
        const descriptor = describeBlock(targetBlock, blockId);
        setContentBlocks(blocks => {
            const updated = blocks.map(block => 
                block.id === blockId ? { ...block, content: updatedContent } : block
            );
            console.log('Edit.jsx: Updated blocks', updated);
            return updated;
        });
        setHasUnsavedChanges(true);
        setLastSavedBlock(blockId);
        appendAuditEntry({
            type: 'content',
            message: `Updated content for ${descriptor}`,
            timestamp: new Date().toISOString(),
        });
        
        // Clear the last saved block indicator after 3 seconds
        setTimeout(() => {
            setLastSavedBlock(null);
        }, 3000);
    };

    const deleteContentBlock = (blockId) => {
        const targetBlock = contentBlocks.find((block) => block.id === blockId);
        const descriptor = describeBlock(targetBlock, blockId);
        setContentBlocks(blocks => blocks.filter(block => block.id !== blockId));
        setHasUnsavedChanges(true);
        appendAuditEntry({
            type: 'delete',
            message: `Removed ${descriptor}`,
            timestamp: new Date().toISOString(),
        });
    };

    const reorderContentBlocks = (startIndex, endIndex) => {
        const result = Array.from(contentBlocks);
        const [removed] = result.splice(startIndex, 1);
        result.splice(endIndex, 0, removed);
        
        // Update order property
        const reorderedBlocks = result.map((block, index) => ({
            ...block,
            order: index
        }));
        
        setContentBlocks(reorderedBlocks);
        const movedBlock = reorderedBlocks[endIndex];
        setHasUnsavedChanges(true);
        appendAuditEntry({
            type: 'reorder',
            message: `Reordered ${describeBlock(movedBlock, movedBlock?.id)} to position ${endIndex + 1}`,
            timestamp: new Date().toISOString(),
        });
    };

    const toggleBlockActive = (blockId, nextIsActive) => {
        const targetBlock = contentBlocks.find((block) => block.id === blockId);
        const descriptor = describeBlock(targetBlock, blockId);

        setContentBlocks((blocks) =>
            blocks.map((block) =>
                block.id === blockId ? { ...block, is_active: nextIsActive } : block
            )
        );

        setHasUnsavedChanges(true);
        setLastSavedBlock(blockId);
        appendAuditEntry({
            type: 'visibility',
            message: `${nextIsActive ? 'Activated' : 'Deactivated'} ${descriptor}`,
            timestamp: new Date().toISOString(),
        });

        setTimeout(() => {
            setLastSavedBlock(null);
        }, 3000);
    };

    const tabs = [
        { id: 'content', label: 'Content', icon: Type },
        { id: 'seo', label: 'SEO', icon: Star },
        { id: 'media', label: 'Media', icon: Image },
        { id: 'settings', label: 'Settings', icon: Settings }
    ];

    return (
        <AdminLayout>
            <Head title={`Edit ${page.title}`} />
            
            <div className="space-y-6">
                {/* Header */}
                <div className="flex justify-between items-center">
                    <div className="flex items-center space-x-4">
                        <a href="/admin/cms/pages">
                            <Button variant="ghost" size="sm">
                                <ArrowLeft className="h-4 w-4 mr-2" />
                                Back to Pages
                            </Button>
                        </a>
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900">Edit Page</h1>
                            <p className="text-gray-600 mt-1">{page.title}</p>
                        </div>
                    </div>
                    <div className="flex items-center space-x-2">
                        <a href={`${page.page_type === 'service' ? '/services' : ''}/${page.slug}`} target="_blank" rel="noopener noreferrer">
                            <Button variant="outline">
                                <Eye className="h-4 w-4 mr-2" />
                                Preview
                            </Button>
                        </a>
                        <Button 
                            onClick={handleSubmit} 
                            disabled={processing}
                            className={`relative ${hasUnsavedChanges ? 'ring-2 ring-orange-400 ring-offset-2 animate-pulse' : ''}`}
                        >
                            <Save className="h-4 w-4 mr-2" />
                            {processing ? 'Saving...' : hasUnsavedChanges ? 'Save to Database' : 'Save Changes'}
                            {hasUnsavedChanges && (
                                <span className="absolute -top-1 -right-1 flex h-3 w-3">
                                    <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-orange-400 opacity-75"></span>
                                    <span className="relative inline-flex rounded-full h-3 w-3 bg-orange-500"></span>
                                </span>
                            )}
                        </Button>
                    </div>
                </div>

                {/* Status Badge */}
                <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-2">
                        <Badge variant={page.status === 'published' ? 'default' : 'secondary'}>
                            {page.status}
                        </Badge>
                        <span className="text-sm text-gray-500">
                            Last updated: {new Date(page.updated_at).toLocaleDateString()}
                        </span>
                    </div>
                    {hasUnsavedChanges && (
                        <div className="flex items-center space-x-2 px-4 py-2 bg-orange-50 border border-orange-200 rounded-md animate-pulse">
                            <span className="relative flex h-2 w-2">
                                <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-orange-400 opacity-75"></span>
                                <span className="relative inline-flex rounded-full h-2 w-2 bg-orange-500"></span>
                            </span>
                            <span className="text-sm font-medium text-orange-700">
                                Unsaved changes in memory
                            </span>
                            <span className="text-xs text-orange-600">
                                Click "Save to Database" to persist
                            </span>
                        </div>
                    )}
                </div>

                {/* Tabs */}
                <div className="border-b border-gray-200">
                    <nav className="-mb-px flex space-x-8">
                        {tabs.map((tab) => {
                            const Icon = tab.icon;
                            return (
                                <button
                                    key={tab.id}
                                    onClick={() => setActiveTab(tab.id)}
                                    className={`flex items-center space-x-2 py-2 px-1 border-b-2 font-medium text-sm ${
                                        activeTab === tab.id
                                            ? 'border-blue-500 text-blue-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    <Icon className="h-4 w-4" />
                                    <span>{tab.label}</span>
                                </button>
                            );
                        })}
                    </nav>
                </div>

                {/* Tab Content */}
                <form onSubmit={handleSubmit}>
                    {activeTab === 'content' && (
                        <div className="space-y-6">
                            {/* Basic Page Info */}
                            <Card>
                                <CardHeader>
                                    <CardTitle>Basic Information</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <Label htmlFor="title">Title</Label>
                                            <Input
                                                id="title"
                                                value={data.title}
                                                onChange={(e) => setData('title', e.target.value)}
                                                className={errors.title ? 'border-red-500' : ''}
                                            />
                                            {errors.title && (
                                                <p className="text-red-500 text-sm mt-1">{errors.title}</p>
                                            )}
                                        </div>
                                        <div>
                                            <Label htmlFor="slug">Slug</Label>
                                            <Input
                                                id="slug"
                                                value={data.slug}
                                                onChange={(e) => setData('slug', e.target.value)}
                                                className={errors.slug ? 'border-red-500' : ''}
                                            />
                                            {errors.slug && (
                                                <p className="text-red-500 text-sm mt-1">{errors.slug}</p>
                                            )}
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            {/* Content Blocks */}
                            <Card>
                                <CardHeader>
                                    <div className="flex justify-between items-center">
                                        <CardTitle>Content Blocks</CardTitle>
                                        <div className="flex space-x-2">
                                            <Select onValueChange={addContentBlock}>
                                                <SelectTrigger className="w-48">
                                                    <SelectValue placeholder="Add Content Block" />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="service_hero">Service Hero</SelectItem>
                                                    <SelectItem value="service_overview">Service Overview</SelectItem>
                                                    <SelectItem value="service_process">Service Process</SelectItem>
                                                    <SelectItem value="service_benefits">Service Benefits</SelectItem>
                                                    <SelectItem value="service_testimonials">Service Testimonials</SelectItem>
                                                    <SelectItem value="service_faqs">Service FAQs</SelectItem>
                                                    <SelectItem value="service_cta">Service CTA</SelectItem>
                                                    <SelectItem value="service_links">Service Links</SelectItem>
                                                    <SelectItem value="service_scrollable">Deep Dive Section</SelectItem>
                                                    <SelectItem value="price_calculator">💰 Price Calculator</SelectItem>
                                                </SelectContent>
                                            </Select>
                                        </div>
                                    </div>
                                </CardHeader>
                                <CardContent>
                                    <ContentBlockEditor
                                        blocks={contentBlocks}
                                        onUpdate={updateContentBlock}
                                        onDelete={deleteContentBlock}
                                        onReorder={reorderContentBlocks}
                                        lastSavedBlockId={lastSavedBlock}
                                        onToggleActive={toggleBlockActive}
                                    />
                                </CardContent>
                            </Card>
                            <Card>
                                <CardHeader>
                                    <CardTitle>Editor Activity</CardTitle>
                                </CardHeader>
                                <CardContent>
                                    {auditLog.length === 0 ? (
                                        <p className="text-sm text-gray-500 dark:text-gray-400">
                                            No changes recorded yet. Adjust block content, order, or visibility to see the audit trail.
                                        </p>
                                    ) : (
                                        <ul className="space-y-2 text-sm text-gray-600 dark:text-gray-300">
                                            {auditLog.map((entry) => (
                                                <li key={entry.id} className="flex items-start justify-between gap-4 border-b border-gray-100 pb-2 last:border-0 last:pb-0 dark:border-gray-700">
                                                    <span>{entry.message}</span>
                                                    <span className="text-xs text-gray-400 dark:text-gray-500 whitespace-nowrap">
                                                        {new Date(entry.timestamp).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                                                    </span>
                                                </li>
                                            ))}
                                        </ul>
                                    )}
                                </CardContent>
                            </Card>
                            
                            {/* Save Button for Content */}
                            <div className="flex justify-end mt-4">
                                <Button 
                                    onClick={handleSubmit} 
                                    disabled={processing}
                                    className={hasUnsavedChanges ? 'ring-2 ring-orange-400 ring-offset-2' : ''}
                                >
                                    <Save className="h-4 w-4 mr-2" />
                                    {processing ? 'Saving...' : 'Save Content Changes'}
                                </Button>
                            </div>
                        </div>
                    )}

                    {activeTab === 'seo' && (
                        <div className="space-y-4">
                            <SEOEditor
                                seoData={seoData}
                                onUpdate={setSeoData}
                                pageData={{
                                    title: data.title,
                                    slug: data.slug,
                                    page_type: data.page_type
                                }}
                            />
                            
                            {/* Save Button for SEO */}
                            <div className="flex justify-end">
                                <Button 
                                    onClick={handleSubmit} 
                                    disabled={processing}
                                    className={hasUnsavedChanges ? 'ring-2 ring-orange-400 ring-offset-2' : ''}
                                >
                                    <Save className="h-4 w-4 mr-2" />
                                    {processing ? 'Saving...' : 'Save SEO Changes'}
                                </Button>
                            </div>
                        </div>
                    )}

                    {activeTab === 'media' && (
                        <div className="space-y-4">
                            <MediaLibrary
                                pageId={page.id}
                                mediaFiles={mediaFiles}
                                onUpdate={setMediaFiles}
                            />
                            
                            {/* Save Button for Media */}
                            <div className="flex justify-end">
                                <Button 
                                    onClick={handleSubmit} 
                                    disabled={processing}
                                    className={hasUnsavedChanges ? 'ring-2 ring-orange-400 ring-offset-2' : ''}
                                >
                                    <Save className="h-4 w-4 mr-2" />
                                    {processing ? 'Saving...' : 'Save Media Changes'}
                                </Button>
                            </div>
                        </div>
                    )}

                    {activeTab === 'settings' && (
                        <div className="space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Page Settings</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <Label htmlFor="page_type">Page Type</Label>
                                            <Select value={data.page_type} onValueChange={(value) => setData('page_type', value)}>
                                                <SelectTrigger>
                                                    <SelectValue />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    {Object.entries(pageTypes).map(([key, label]) => (
                                                        <SelectItem key={key} value={key}>
                                                            {label}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                        </div>
                                        <div>
                                            <Label htmlFor="status">Status</Label>
                                            <Select value={data.status} onValueChange={(value) => setData('status', value)}>
                                                <SelectTrigger>
                                                    <SelectValue />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="draft">Draft</SelectItem>
                                                    <SelectItem value="published">Published</SelectItem>
                                                    <SelectItem value="archived">Archived</SelectItem>
                                                </SelectContent>
                                            </Select>
                                        </div>
                                    </div>
                                    
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <Label htmlFor="template">Template</Label>
                                            <Select value={data.template} onValueChange={(value) => setData('template', value)}>
                                                <SelectTrigger>
                                                    <SelectValue />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    {Object.entries(templates).map(([key, label]) => (
                                                        <SelectItem key={key} value={key}>
                                                            {label}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                        </div>
                                        <div>
                                            <Label htmlFor="parent_id">Parent Page</Label>
                                            <Select value={data.parent_id} onValueChange={(value) => setData('parent_id', value)}>
                                                <SelectTrigger>
                                                    <SelectValue placeholder="No parent page" />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="">No parent page</SelectItem>
                                                    {parentPages.map((parent) => (
                                                        <SelectItem key={parent.id} value={parent.id}>
                                                            {parent.title}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                        </div>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <Label htmlFor="seo_priority">SEO Priority (0-100)</Label>
                                            <Input
                                                id="seo_priority"
                                                type="number"
                                                min="0"
                                                max="100"
                                                value={data.seo_priority}
                                                onChange={(e) => setData('seo_priority', parseInt(e.target.value))}
                                            />
                                        </div>
                                        <div>
                                            <Label htmlFor="published_at">Published Date</Label>
                                            <Input
                                                id="published_at"
                                                type="datetime-local"
                                                value={data.published_at}
                                                onChange={(e) => setData('published_at', e.target.value)}
                                            />
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>
                            
                            {/* Save Button for Settings */}
                            <div className="flex justify-end mt-4">
                                <Button 
                                    onClick={handleSubmit} 
                                    disabled={processing}
                                    className={hasUnsavedChanges ? 'ring-2 ring-orange-400 ring-offset-2' : ''}
                                >
                                    <Save className="h-4 w-4 mr-2" />
                                    {processing ? 'Saving...' : 'Save Settings'}
                                </Button>
                            </div>
                        </div>
                    )}
                </form>
            </div>
        </AdminLayout>
    );
}
