import React, { useState, useEffect } from 'react';
import { Head, useForm } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Button } from '@/Components/ui/Button';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Label } from '@/Components/ui/Label';
import { Input } from '@/Components/ui/Input';
import { Save, Eye } from 'lucide-react';
import RichTextEditor from '@/Components/Admin/RichTextEditor';

function findBlock(blocks, key) {
  return blocks.find(b => b.block_key === key) || null;
}

function upsertBlock(blocks, key, type, content) {
  const idx = blocks.findIndex(b => b.block_key === key);
  const base = { block_key: key, block_type: type, content, order: blocks.length, is_active: true };
  if (idx === -1) return [...blocks, base];
  const updated = [...blocks];
  updated[idx] = { ...updated[idx], content };
  return updated;
}

const DEFAULT_ICON_OPTIONS = [
  'CheckCircle',
  'Award',
  'Clock',
  'Shield',
  'Users',
  'FileText',
  'BookOpen',
  'Zap',
];

function RichListEditor({
  label,
  items = [],
  onChange,
  addLabel = 'Add Item',
  itemLabel = 'Item',
  height = 160,
  richKey = null,
  createItem,
  emptyHint = 'No items yet. Click the button to add one.',
}) {
  const list = Array.isArray(items) ? items : [];

  const makeItem = (index) => {
    if (createItem) {
      return createItem(index);
    }

    if (richKey) {
      return { [richKey]: '' };
    }

    return '';
  };

  const getValue = (item) => {
    if (richKey) {
      if (item && typeof item === 'object') {
        return item[richKey] || '';
      }
      if (typeof item === 'string') {
        return item;
      }
      return '';
    }

    if (typeof item === 'string') {
      return item;
    }

    if (item && typeof item === 'object') {
      return item.text || '';
    }

    return '';
  };

  const handleValueChange = (index, value) => {
    const updated = [...list];

    if (richKey) {
      const current = updated[index];
      if (current && typeof current === 'object') {
        updated[index] = { ...current, [richKey]: value };
      } else {
        updated[index] = { [richKey]: value };
      }
    } else {
      updated[index] = value;
    }

    onChange(updated);
  };

  const handleAdd = () => {
    onChange([...list, makeItem(list.length)]);
  };

  const handleRemove = (index) => {
    onChange(list.filter((_, i) => i !== index));
  };

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <Label>{label}</Label>
        <Button type="button" onClick={handleAdd}>
          {addLabel}
        </Button>
      </div>
      {list.length === 0 && (
        <p className="text-sm text-gray-500 dark:text-gray-400">{emptyHint}</p>
      )}
      {list.map((item, index) => (
        <Card key={index}>
          <CardContent className="space-y-3 pt-4">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-600 dark:text-gray-300">
                {itemLabel} {index + 1}
              </span>
              <Button
                variant="ghost"
                size="sm"
                type="button"
                onClick={() => handleRemove(index)}
              >
                Remove
              </Button>
            </div>
            <RichTextEditor
              value={getValue(item)}
              onChange={(val) => handleValueChange(index, val)}
              height={height}
              placeholder="Add rich text content..."
            />
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

function FeatureListEditor({ value = [], onChange }) {
  const list = Array.isArray(value) ? value : [];

  const normalise = (item = {}) => ({
    title: item.title || '',
    description: item.description || '',
    icon: item.icon || 'CheckCircle',
  });

  const add = () => onChange([...list, normalise()]);

  const update = (index, patch) => {
    const updated = [...list];
    updated[index] = normalise({ ...updated[index], ...patch });
    onChange(updated);
  };

  const remove = (index) => onChange(list.filter((_, i) => i !== index));

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <Label>Features</Label>
        <Button type="button" onClick={add}>
          Add Feature
        </Button>
      </div>
      {list.length === 0 && (
        <p className="text-sm text-gray-500 dark:text-gray-400">
          No features yet. Add feature cards to highlight selling points.
        </p>
      )}
      {list.map((feature, index) => (
        <Card key={index}>
          <CardContent className="space-y-4 pt-4">
            <div className="grid md:grid-cols-2 gap-4">
              <div>
                <Label>Feature title</Label>
                <Input
                  value={feature.title}
                  onChange={(e) => update(index, { title: e.target.value })}
                  placeholder="e.g., Transparent Pricing"
                />
              </div>
              <div>
                <Label>Icon</Label>
                <select
                  value={feature.icon}
                  onChange={(e) => update(index, { icon: e.target.value })}
                  className="block w-full rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-800 py-2 px-3 text-sm text-gray-700 dark:text-gray-100 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                  {DEFAULT_ICON_OPTIONS.map((icon) => (
                    <option key={icon} value={icon}>
                      {icon}
                    </option>
                  ))}
                </select>
              </div>
            </div>
            <div>
              <Label>Feature description</Label>
              <RichTextEditor
                value={feature.description}
                onChange={(val) => update(index, { description: val })}
                height={180}
                placeholder="Describe the benefit and consider linking to relevant resources."
              />
            </div>
            <div className="flex justify-end">
              <Button variant="ghost" type="button" onClick={() => remove(index)}>
                Remove
              </Button>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

function BenefitCardsEditor({ value = [], onChange }) {
  const list = Array.isArray(value) ? value : [];

  const normalise = (item = {}) => ({
    title: item.title || '',
    description: item.description || '',
    icon: item.icon || 'CheckCircle',
  });

  const add = () => onChange([...list, normalise()]);

  const update = (index, patch) => {
    const updated = [...list];
    updated[index] = normalise({ ...updated[index], ...patch });
    onChange(updated);
  };

  const remove = (index) => onChange(list.filter((_, i) => i !== index));

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <Label>Benefit cards</Label>
        <Button type="button" onClick={add}>
          Add Benefit
        </Button>
      </div>
      {list.length === 0 && (
        <p className="text-sm text-gray-500 dark:text-gray-400">
          No benefit cards configured. Add benefit cards to showcase reasons to choose this service.
        </p>
      )}
      {list.map((benefit, index) => (
        <Card key={index}>
          <CardContent className="space-y-4 pt-4">
            <div className="grid md:grid-cols-2 gap-4">
              <div>
                <Label>Benefit title</Label>
                <Input
                  value={benefit.title}
                  onChange={(e) => update(index, { title: e.target.value })}
                  placeholder="e.g., Dedicated account manager"
                />
              </div>
              <div>
                <Label>Icon</Label>
                <select
                  value={benefit.icon}
                  onChange={(e) => update(index, { icon: e.target.value })}
                  className="block w-full rounded-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-800 py-2 px-3 text-sm text-gray-700 dark:text-gray-100 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                  {DEFAULT_ICON_OPTIONS.map((icon) => (
                    <option key={icon} value={icon}>
                      {icon}
                    </option>
                  ))}
                </select>
              </div>
            </div>
            <div>
              <Label>Benefit description</Label>
              <RichTextEditor
                value={benefit.description}
                onChange={(val) => update(index, { description: val })}
                height={180}
                placeholder="Explain why this benefit matters. Feel free to add internal links."
              />
            </div>
            <div className="flex justify-end">
              <Button variant="ghost" type="button" onClick={() => remove(index)}>
                Remove
              </Button>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

function StepsEditor({ value = [], onChange }) {
  const add = () => onChange([...(value || []), { title: '', description: '' }]);
  const update = (i, patch) => onChange(value.map((s, idx) => idx === i ? { ...s, ...patch } : s));
  const remove = (i) => onChange(value.filter((_, idx) => idx !== i));
  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <Label>Steps</Label>
        <Button type="button" onClick={add}>Add Step</Button>
      </div>
      {(value || []).map((s, i) => (
        <Card key={i}>
          <CardContent className="space-y-3 pt-4">
            <div>
              <Label>Step Title</Label>
              <Input value={s.title} onChange={(e) => update(i, { title: e.target.value })} />
            </div>
            <div>
              <Label>Step Description</Label>
              <RichTextEditor value={s.description} onChange={(v) => update(i, { description: v })} height={220} />
            </div>
            <div className="flex justify-end"><Button variant="ghost" type="button" onClick={() => remove(i)}>Remove</Button></div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

function FAQsEditor({ value = [], onChange }) {
  const add = () => onChange([...(value || []), { question: '', answer: '' }]);
  const update = (i, patch) => onChange(value.map((s, idx) => idx === i ? { ...s, ...patch } : s));
  const remove = (i) => onChange(value.filter((_, idx) => idx !== i));
  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <Label>FAQs</Label>
        <Button type="button" onClick={add}>Add FAQ</Button>
      </div>
      {(value || []).map((f, i) => (
        <Card key={i}>
          <CardContent className="space-y-3 pt-4">
            <div>
              <Label>Question</Label>
              <Input value={f.question} onChange={(e) => update(i, { question: e.target.value })} />
            </div>
            <div>
              <Label>Answer</Label>
              <RichTextEditor value={f.answer} onChange={(v) => update(i, { answer: v })} height={240} />
            </div>
            <div className="flex justify-end"><Button variant="ghost" type="button" onClick={() => remove(i)}>Remove</Button></div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

function LinkListEditor({ label = 'Links', value = [], onChange }) {
  const add = () => onChange([...(value || []), { title: '', url: '' }]);
  const update = (i, patch) => onChange(value.map((link, idx) => (idx === i ? { ...link, ...patch } : link)));
  const remove = (i) => onChange(value.filter((_, idx) => idx !== i));

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <Label>{label}</Label>
        <Button type="button" onClick={add}>
          Add Link
        </Button>
      </div>
      {(value || []).map((link, i) => (
        <Card key={`${link.title}-${i}`}>
          <CardContent className="grid gap-4 pt-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label>Link title</Label>
              <Input
                value={link.title || ''}
                onChange={(e) => update(i, { title: e.target.value })}
                placeholder="e.g., How to get quality essays on a budget"
              />
            </div>
            <div className="space-y-2">
              <Label>URL</Label>
              <Input
                value={link.url || ''}
                onChange={(e) => update(i, { url: e.target.value })}
                placeholder="e.g., /blog/affordable-essay-guide-placeholder"
              />
            </div>
            <div className="md:col-span-2 flex justify-end">
              <Button variant="ghost" type="button" onClick={() => remove(i)}>
                Remove
              </Button>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

function TrustMatrixEditor({ value = [], onChange }) {
  const list = Array.isArray(value) ? value : [];
  const add = () => onChange([...list, { challenge: '', help: '', outcome: '' }]);
  const update = (i, patch) =>
    onChange(
      list.map((row, idx) =>
        idx === i ? { challenge: '', help: '', outcome: '', ...(row || {}), ...patch } : row
      )
    );
  const remove = (i) => onChange(value.filter((_, idx) => idx !== i));
  return (
    <div className="space-y-3">
      <div className="flex justify-between items-center">
        <Label>Trust Matrix Rows</Label>
        <Button type="button" onClick={add}>Add Row</Button>
      </div>
      {(value || []).map((r, i) => (
        <div key={i} className="grid md:grid-cols-3 gap-3">
          <div>
            <Label>Customer challenge</Label>
            <RichTextEditor
              value={r.challenge}
              onChange={(val) => update(i, { challenge: val })}
              height={160}
              placeholder="Describe the challenge and consider linking to supporting content."
            />
          </div>
          <div>
            <Label>How we help</Label>
            <RichTextEditor
              value={r.help}
              onChange={(val) => update(i, { help: val })}
              height={160}
              placeholder="Explain how your team solves the challenge."
            />
          </div>
          <div>
            <Label>What you get</Label>
            <RichTextEditor
              value={r.outcome}
              onChange={(val) => update(i, { outcome: val })}
              height={160}
              placeholder="Share the measurable outcome or transformation."
            />
          </div>
          <div className="md:col-span-3 flex justify-end">
            <Button variant="ghost" size="sm" type="button" onClick={() => remove(i)}>Remove Row</Button>
          </div>
        </div>
      ))}
    </div>
  );
}

export default function AffordableEditor({ page }) {
  const [blocks, setBlocks] = useState(page.content_blocks || []);
  const { put, processing } = useForm({});

  // Helpers for each block
  const heroBlock = findBlock(blocks, 'affordable_hero');
  const defaultHero = {
    title: 'Affordable Essay Writing Service',
    description:
      `<div class="space-y-4 text-left md:text-center text-white"><p class="text-lg text-white leading-relaxed">Ready to <strong class="text-white">buy essays online</strong> without gambling on quality? Academic Scribe matches you with vetted writers who follow your brief, meet your deadline, and deliver polished drafts you can submit confidently.</p><p class="text-lg text-white leading-relaxed">Upload your requirements, choose your turnaround, and we handle the rest. Secure checkout, transparent pricing, and responsive support keep every <strong class="text-white">buy essay writing service</strong> order on track.</p><ul class="list-disc list-inside space-y-2 text-blue-100"><li><span class="text-white">Subject specialists ready for <strong class="text-white">buy custom essay online</strong> requests.</span></li><li><span class="text-white">Encrypted payments and private messaging protect your identity.</span></li><li><span class="text-white">Free revisions ensure you get exactly what you asked for.</span></li></ul></div>`,
    ctaText: 'Get a price in 30 seconds',
    ctaLink: '/place-order',
    keyPoints: [
      'Expert writers in every subject',
      '100% original content with reports',
      'Rush delivery options when you need them',
    ],
    trustSignals: [
      '4.8/5 Rating',
      '150,000+ Students',
      '100% Plagiarism-Free',
    ],
  };

  const hero = {
    ...defaultHero,
    ...(heroBlock?.content || {}),
    keyPoints:
      heroBlock?.content?.keyPoints && heroBlock.content.keyPoints.length > 0
        ? heroBlock.content.keyPoints
        : defaultHero.keyPoints,
    trustSignals:
      heroBlock?.content?.trustSignals && heroBlock.content.trustSignals.length > 0
        ? heroBlock.content.trustSignals
        : defaultHero.trustSignals,
    ctaLink:
      heroBlock?.content?.ctaLink && heroBlock.content.ctaLink.trim().length > 0
        ? heroBlock.content.ctaLink
        : '/place-order',
  };
  const overview = findBlock(blocks, 'affordable_overview')?.content || { title: '', description: '', features: [] };
  const process = findBlock(blocks, 'affordable_process')?.content || { title: '', description: '', steps: [] };
  const benefits = findBlock(blocks, 'affordable_benefits')?.content || { title: '', description: '', benefits: [], guarantees: [], ctaText: '', ctaLink: '' };
  const faqs = findBlock(blocks, 'affordable_faqs')?.content || { title: '', description: '', faqs: [] };
  const cta = findBlock(blocks, 'affordable_cta')?.content || { title: '', description: '', benefits: [], primaryButtonText: '', primaryButtonLink: '', secondaryButtonText: '', secondaryButtonLink: '' };
  const trust = findBlock(blocks, 'affordable_trust')?.content || { title: 'Why students trust our service', description: '', rows: [] };
  const defaultResourceLinks = {
    title: 'Resources',
    description: 'Hand-picked reads coming soon for affordable essay help.',
    links: [
      {
        title: 'Upcoming guide: How to get quality essays on a budget',
        url: '/blog/affordable-essay-guide-placeholder',
      },
      {
        title: 'Upcoming breakdown: What affects essay costs in 2025',
        url: '/blog/essay-costs-2025-placeholder',
      },
    ],
  };

  const resourceLinksBlock = findBlock(blocks, 'affordable_resource_links');
  const resourceLinks = {
    title: resourceLinksBlock?.content?.title || defaultResourceLinks.title,
    description: resourceLinksBlock?.content?.description || defaultResourceLinks.description,
    links: resourceLinksBlock?.content?.links || defaultResourceLinks.links,
  };

  useEffect(() => {
    if (!resourceLinksBlock) {
      setBlocks((prev) =>
        upsertBlock(prev, 'affordable_resource_links', 'links', defaultResourceLinks)
      );
    }
  }, [resourceLinksBlock]);

  const saveBlocks = (e) => {
    e.preventDefault();
    const payload = { 
      title: page.title, slug: page.slug, page_type: 'service', status: 'published',
      content_blocks: blocks
    };
    put(`/admin/cms/pages/${page.id}`, {
      data: payload,
      preserveScroll: true,
    });
  };

  const update = (key, type, content) => setBlocks(prev => upsertBlock(prev, key, type, content));

  return (
    <AdminLayout>
      <Head title={`Edit: ${page.title}`} />
      <div className="flex items-center justify-between mb-6">
        <h1 className="text-2xl font-bold">Affordable Essay Writing – Editor</h1>
        <div className="flex gap-2">
          <a href={`/services/affordable-essay-writing`} target="_blank" rel="noopener noreferrer">
            <Button variant="outline"><Eye className="h-4 w-4 mr-2" />Preview</Button>
          </a>
          <Button onClick={saveBlocks} disabled={processing}><Save className="h-4 w-4 mr-2" />Save</Button>
        </div>
      </div>

      {/* Hero */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Hero</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={hero.title} onChange={(e) => update('affordable_hero', 'hero', { ...hero, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={hero.description} onChange={(v) => update('affordable_hero', 'hero', { ...hero, description: v })} height={260} />
          </div>
          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <Label>Primary CTA Text</Label>
              <Input value={hero.ctaText || ''} onChange={(e) => update('affordable_hero', 'hero', { ...hero, ctaText: e.target.value })} />
            </div>
            <div>
              <Label>Primary CTA Link</Label>
              <Input value={hero.ctaLink || ''} onChange={(e) => update('affordable_hero', 'hero', { ...hero, ctaLink: e.target.value })} placeholder="/place-order" />
            </div>
          </div>
          <RichListEditor
            label="Key points"
            addLabel="Add Key Point"
            itemLabel="Key point"
            items={hero.keyPoints || []}
            onChange={(arr) => update('affordable_hero', 'hero', { ...hero, keyPoints: arr })}
            height={140}
            emptyHint="No key points yet. Add concise proof points and link to relevant internal resources."
          />
          <RichListEditor
            label="Trust signals"
            addLabel="Add Trust Signal"
            itemLabel="Trust signal"
            items={hero.trustSignals || []}
            onChange={(arr) => update('affordable_hero', 'hero', { ...hero, trustSignals: arr })}
            height={140}
            emptyHint="No trust signals yet. Add social proof, ratings, or guarantees."
          />
        </CardContent>
      </Card>

      {/* Overview */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Overview</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={overview.title} onChange={(e) => update('affordable_overview', 'overview', { ...overview, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={overview.description} onChange={(v) => update('affordable_overview', 'overview', { ...overview, description: v })} height={340} />
          </div>
          <FeatureListEditor
            value={overview.features}
            onChange={(arr) => update('affordable_overview', 'overview', { ...overview, features: arr })}
          />
        </CardContent>
      </Card>

      {/* Process */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Process</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={process.title} onChange={(e) => update('affordable_process', 'process', { ...process, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={process.description} onChange={(v) => update('affordable_process', 'process', { ...process, description: v })} height={240} />
          </div>
          <StepsEditor value={process.steps} onChange={(val) => update('affordable_process', 'process', { ...process, steps: val })} />
        </CardContent>
      </Card>

      {/* Benefits */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Benefits</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={benefits.title} onChange={(e) => update('affordable_benefits', 'benefits', { ...benefits, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={benefits.description} onChange={(v) => update('affordable_benefits', 'benefits', { ...benefits, description: v })} height={240} />
          </div>
          <BenefitCardsEditor
            value={benefits.benefits}
            onChange={(arr) => update('affordable_benefits', 'benefits', { ...benefits, benefits: arr })}
          />
          <RichListEditor
            label="Guarantees"
            addLabel="Add Guarantee"
            itemLabel="Guarantee"
            items={benefits.guarantees || []}
            onChange={(arr) => update('affordable_benefits', 'benefits', { ...benefits, guarantees: arr })}
            height={150}
            emptyHint="Highlight promises such as plagiarism reports, refund assurances, or support policies."
          />
        </CardContent>
      </Card>

      {/* FAQs */}
      <Card className="mb-6">
        <CardHeader><CardTitle>FAQs</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={faqs.title} onChange={(e) => update('affordable_faqs', 'faqs', { ...faqs, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={faqs.description} onChange={(v) => update('affordable_faqs', 'faqs', { ...faqs, description: v })} height={200} />
          </div>
          <FAQsEditor value={faqs.faqs} onChange={(val) => update('affordable_faqs', 'faqs', { ...faqs, faqs: val })} />
        </CardContent>
      </Card>

      {/* Trust Matrix */}
      <Card className="mb-10">
        <CardHeader><CardTitle>Trust Matrix</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Section Title</Label>
            <Input value={trust.title} onChange={(e) => update('affordable_trust', 'content', { ...trust, title: e.target.value })} />
          </div>
          <div>
            <Label>Intro / Subtitle</Label>
            <RichTextEditor value={trust.description} onChange={(v) => update('affordable_trust', 'content', { ...trust, description: v })} height={200} />
          </div>
          <TrustMatrixEditor value={trust.rows} onChange={(val) => update('affordable_trust', 'content', { ...trust, rows: val })} />
        </CardContent>
      </Card>

      <Card className="mb-10">
        <CardHeader>
          <CardTitle>Resource Links (footer section)</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid gap-4 md:grid-cols-2">
            <div>
              <Label>Section title</Label>
              <Input
                value={resourceLinks.title}
                onChange={(e) =>
                  update('affordable_resource_links', 'links', {
                    ...resourceLinks,
                    title: e.target.value,
                  })
                }
              />
            </div>
            <div className="md:col-span-2">
              <Label>Section description (optional)</Label>
              <RichTextEditor
                value={resourceLinks.description}
                onChange={(val) =>
                  update('affordable_resource_links', 'links', {
                    ...resourceLinks,
                    description: val,
                  })
                }
                height={180}
                placeholder="Explain how these resources support the page. Use internal links generously."
              />
            </div>
          </div>
          <LinkListEditor
            value={resourceLinks.links}
            onChange={(links) =>
              update('affordable_resource_links', 'links', {
                ...resourceLinks,
                links,
              })
            }
          />
        </CardContent>
      </Card>

      <div className="flex justify-end pb-10">
        <Button onClick={saveBlocks} disabled={processing}><Save className="h-4 w-4 mr-2" />Save All Changes</Button>
      </div>
    </AdminLayout>
  );
}


