import React, { useState, useEffect } from 'react';
import { Head, useForm } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Button } from '@/Components/ui/Button';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Label } from '@/Components/ui/Label';
import { Input } from '@/Components/ui/Input';
import { Save, Eye } from 'lucide-react';
import RichTextEditor from '@/Components/Admin/RichTextEditor';

function findBlock(blocks, key) {
  return blocks.find(b => b.block_key === key) || null;
}

function upsertBlock(blocks, key, type, content) {
  const idx = blocks.findIndex(b => b.block_key === key);
  const base = { block_key: key, block_type: type, content, order: blocks.length, is_active: true };
  if (idx === -1) return [...blocks, base];
  const updated = [...blocks];
  updated[idx] = { ...updated[idx], content };
  return updated;
}

function ArrayField({ label, items, onChange, placeholder = 'Item text' }) {
  const [value, setValue] = useState('');
  return (
    <div className="space-y-2">
      <Label>{label}</Label>
      <div className="flex gap-2">
        <Input value={value} onChange={(e) => setValue(e.target.value)} placeholder={placeholder} />
        <Button type="button" onClick={() => { if (!value.trim()) return; onChange([...(items || []), value.trim()]); setValue(''); }}>Add</Button>
      </div>
      <ul className="list-disc pl-6 text-sm text-gray-700">
        {(items || []).map((it, i) => (
          <li key={i} className="flex items-center justify-between">
            <span>{it}</span>
            <Button variant="ghost" size="sm" type="button" onClick={() => onChange(items.filter((_, idx) => idx !== i))}>Remove</Button>
          </li>
        ))}
      </ul>
    </div>
  );
}

function StepsEditor({ value = [], onChange }) {
  const add = () => onChange([...(value || []), { title: '', description: '' }]);
  const update = (i, patch) => onChange(value.map((s, idx) => idx === i ? { ...s, ...patch } : s));
  const remove = (i) => onChange(value.filter((_, idx) => idx !== i));
  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <Label>Steps</Label>
        <Button type="button" onClick={add}>Add Step</Button>
      </div>
      {(value || []).map((s, i) => (
        <Card key={i}>
          <CardContent className="space-y-3 pt-4">
            <div>
              <Label>Step Title</Label>
              <Input value={s.title} onChange={(e) => update(i, { title: e.target.value })} />
            </div>
            <div>
              <Label>Step Description</Label>
              <RichTextEditor value={s.description} onChange={(v) => update(i, { description: v })} height={220} />
            </div>
            <div className="flex justify-end"><Button variant="ghost" type="button" onClick={() => remove(i)}>Remove</Button></div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

function FAQsEditor({ value = [], onChange }) {
  const add = () => onChange([...(value || []), { question: '', answer: '' }]);
  const update = (i, patch) => onChange(value.map((s, idx) => idx === i ? { ...s, ...patch } : s));
  const remove = (i) => onChange(value.filter((_, idx) => idx !== i));
  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <Label>FAQs</Label>
        <Button type="button" onClick={add}>Add FAQ</Button>
      </div>
      {(value || []).map((f, i) => (
        <Card key={i}>
          <CardContent className="space-y-3 pt-4">
            <div>
              <Label>Question</Label>
              <Input value={f.question} onChange={(e) => update(i, { question: e.target.value })} />
            </div>
            <div>
              <Label>Answer</Label>
              <RichTextEditor value={f.answer} onChange={(v) => update(i, { answer: v })} height={240} />
            </div>
            <div className="flex justify-end"><Button variant="ghost" type="button" onClick={() => remove(i)}>Remove</Button></div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

export default function CheapEditor({ page }) {
  const [blocks, setBlocks] = useState(page.content_blocks || []);
  const { put, processing } = useForm({});

  // Helpers for each block
  const heroBlock = findBlock(blocks, 'cheap_hero');
  const defaultHero = {
    title: 'Cheap Essay Writing Service You Can Trust',
    description:
      `<div class="space-y-4 text-left md:text-center text-white"><p class="text-lg text-white leading-relaxed">Looking for <strong class="text-white">cheap essays online</strong> that still meet your professor's standards? Academic Scribe balances savings and quality with honest pricing, expert writers, and transparent safeguards.</p><p class="text-lg text-white leading-relaxed">Choose your deadline, upload your brief, and our editors keep the guardrails in place so every <strong class="text-white">cheap essay writing service</strong> order stays safe and on-brand.</p><ul class="list-disc list-inside space-y-2 text-blue-100"><li><span class="text-white"><strong class="text-white">Affordable bundles</strong> designed for <strong class="text-white">low-cost essay writing</strong> needs.</span></li><li><span class="text-white"><strong class="text-white">Quality checks</strong>, plagiarism scans, and structured drafts on every project.</span></li><li><span class="text-white"><strong class="text-white">Supportive team</strong> ready to help you <strong class="text-white">buy cheap essays online</strong> without stress.</span></li></ul></div>`,
    ctaText: 'Get a price in 30 seconds',
    ctaLink: '/place-order',
    keyPoints: [
      'Affordable packages for every budget',
      'Human writers with subject expertise',
      'On-time delivery with revision support',
    ],
    trustSignals: [
      '4.8/5 Rating',
      '150,000+ Students',
      '100% Plagiarism-Free',
    ],
  };

  const hero = {
    ...defaultHero,
    ...(heroBlock?.content || {}),
    keyPoints:
      heroBlock?.content?.keyPoints && heroBlock.content.keyPoints.length > 0
        ? heroBlock.content.keyPoints
        : defaultHero.keyPoints,
    trustSignals:
      heroBlock?.content?.trustSignals && heroBlock.content.trustSignals.length > 0
        ? heroBlock.content.trustSignals
        : defaultHero.trustSignals,
    ctaLink:
      heroBlock?.content?.ctaLink && heroBlock.content.ctaLink.trim().length > 0
        ? heroBlock.content.ctaLink
        : '/place-order',
  };
  const overview = findBlock(blocks, 'cheap_overview')?.content || { title: '', description: '', features: [] };
  const process = findBlock(blocks, 'cheap_process')?.content || { title: '', description: '', steps: [] };
  const benefits = findBlock(blocks, 'cheap_benefits')?.content || { title: '', description: '', benefits: [], guarantees: [], ctaText: '', ctaLink: '' };
  const faqs = findBlock(blocks, 'cheap_faqs')?.content || { title: '', description: '', faqs: [] };
  const cta = findBlock(blocks, 'cheap_cta')?.content || { title: '', description: '', benefits: [], primaryButtonText: '', primaryButtonLink: '', secondaryButtonText: '', secondaryButtonLink: '' };
  const defaultResourceLinks = {
    title: 'Resources',
    description: 'Bookmark these upcoming reads for smart budget essay decisions.',
    links: [
      {
        title: 'Upcoming comparison: Cheap vs affordable essay services (placeholder)',
        url: '/blog/cheap-vs-affordable-placeholder',
      },
      {
        title: 'Upcoming article: Hidden costs of too-cheap essays (placeholder)',
        url: '/blog/hidden-costs-cheap-essays-placeholder',
      },
    ],
  };

  const resourceLinksBlock = findBlock(blocks, 'cheap_resource_links');
  const resourceLinks = {
    title: resourceLinksBlock?.content?.title || defaultResourceLinks.title,
    description: resourceLinksBlock?.content?.description || defaultResourceLinks.description,
    links: resourceLinksBlock?.content?.links || defaultResourceLinks.links,
  };

  useEffect(() => {
    if (!resourceLinksBlock) {
      setBlocks((prev) =>
        upsertBlock(prev, 'cheap_resource_links', 'links', defaultResourceLinks)
      );
    }
  }, [resourceLinksBlock]);

  const saveBlocks = (e) => {
    e.preventDefault();
    const payload = { 
      title: page.title, slug: page.slug, page_type: 'service', status: 'published',
      content_blocks: blocks
    };
    put(`/admin/cms/pages/${page.id}`, {
      data: payload,
      preserveScroll: true,
    });
  };

  const update = (key, type, content) => setBlocks(prev => upsertBlock(prev, key, type, content));

  return (
    <AdminLayout>
      <Head title={`Edit: ${page.title}`} />
      <div className="flex items-center justify-between mb-6">
        <h1 className="text-2xl font-bold">Cheap Essays Online – Editor</h1>
        <div className="flex gap-2">
          <a href={`/services/cheap-essays-online`} target="_blank" rel="noopener noreferrer">
            <Button variant="outline"><Eye className="h-4 w-4 mr-2" />Preview</Button>
          </a>
          <Button onClick={saveBlocks} disabled={processing}><Save className="h-4 w-4 mr-2" />Save</Button>
        </div>
      </div>

      {/* Hero */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Hero</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={hero.title} onChange={(e) => update('cheap_hero', 'hero', { ...hero, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={hero.description} onChange={(v) => update('cheap_hero', 'hero', { ...hero, description: v })} height={260} />
          </div>
          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <Label>Primary CTA Text</Label>
              <Input value={hero.ctaText || ''} onChange={(e) => update('cheap_hero', 'hero', { ...hero, ctaText: e.target.value })} />
            </div>
            <div>
              <Label>Primary CTA Link</Label>
              <Input value={hero.ctaLink || ''} onChange={(e) => update('cheap_hero', 'hero', { ...hero, ctaLink: e.target.value })} placeholder="/place-order" />
            </div>
          </div>
          <ArrayField label="Key points" items={hero.keyPoints || []} onChange={(arr) => update('cheap_hero', 'hero', { ...hero, keyPoints: arr })} placeholder="e.g., Budget-friendly" />
          <ArrayField label="Trust signals" items={hero.trustSignals || []} onChange={(arr) => update('cheap_hero', 'hero', { ...hero, trustSignals: arr })} placeholder="e.g., Quality safeguards" />
        </CardContent>
      </Card>

      {/* Overview */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Overview</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={overview.title} onChange={(e) => update('cheap_overview', 'overview', { ...overview, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={overview.description} onChange={(v) => update('cheap_overview', 'overview', { ...overview, description: v })} height={340} />
          </div>
          <ArrayField label="Features" items={overview.features} onChange={(arr) => update('cheap_overview', 'overview', { ...overview, features: arr })} />
        </CardContent>
      </Card>

      {/* Process */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Process</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={process.title} onChange={(e) => update('cheap_process', 'process', { ...process, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={process.description} onChange={(v) => update('cheap_process', 'process', { ...process, description: v })} height={240} />
          </div>
          <StepsEditor value={process.steps} onChange={(val) => update('cheap_process', 'process', { ...process, steps: val })} />
        </CardContent>
      </Card>

      {/* Benefits */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Benefits</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={benefits.title} onChange={(e) => update('cheap_benefits', 'benefits', { ...benefits, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={benefits.description} onChange={(v) => update('cheap_benefits', 'benefits', { ...benefits, description: v })} height={240} />
          </div>
          <ArrayField label="Benefits" items={benefits.benefits} onChange={(arr) => update('cheap_benefits', 'benefits', { ...benefits, benefits: arr })} />
          <ArrayField label="Guarantees" items={benefits.guarantees} onChange={(arr) => update('cheap_benefits', 'benefits', { ...benefits, guarantees: arr })} />
        </CardContent>
      </Card>

      {/* FAQs */}
      <Card className="mb-6">
        <CardHeader><CardTitle>FAQs</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={faqs.title} onChange={(e) => update('cheap_faqs', 'faqs', { ...faqs, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={faqs.description} onChange={(v) => update('cheap_faqs', 'faqs', { ...faqs, description: v })} height={200} />
          </div>
          <FAQsEditor value={faqs.faqs} onChange={(val) => update('cheap_faqs', 'faqs', { ...faqs, faqs: val })} />
        </CardContent>
      </Card>

      {/* CTA */}
      <Card className="mb-6">
        <CardHeader><CardTitle>Call to Action</CardTitle></CardHeader>
        <CardContent className="space-y-4">
          <div>
            <Label>Title</Label>
            <Input value={cta.title} onChange={(e) => update('cheap_cta', 'cta', { ...cta, title: e.target.value })} />
          </div>
          <div>
            <Label>Description</Label>
            <RichTextEditor value={cta.description} onChange={(v) => update('cheap_cta', 'cta', { ...cta, description: v })} height={200} />
          </div>
          <ArrayField label="Benefits" items={cta.benefits || []} onChange={(arr) => update('cheap_cta', 'cta', { ...cta, benefits: arr })} placeholder="e.g., Budget-friendly" />
          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <Label>Primary Button Text</Label>
              <Input value={cta.primaryButtonText || ''} onChange={(e) => update('cheap_cta', 'cta', { ...cta, primaryButtonText: e.target.value })} />
            </div>
            <div>
              <Label>Primary Button Link</Label>
              <Input value={cta.primaryButtonLink || ''} onChange={(e) => update('cheap_cta', 'cta', { ...cta, primaryButtonLink: e.target.value })} />
            </div>
            <div>
              <Label>Secondary Button Text</Label>
              <Input value={cta.secondaryButtonText || ''} onChange={(e) => update('cheap_cta', 'cta', { ...cta, secondaryButtonText: e.target.value })} />
            </div>
            <div>
              <Label>Secondary Button Link</Label>
              <Input value={cta.secondaryButtonLink || ''} onChange={(e) => update('cheap_cta', 'cta', { ...cta, secondaryButtonLink: e.target.value })} />
            </div>
          </div>
        </CardContent>
      </Card>

      <Card className="mb-10">
        <CardHeader>
          <CardTitle>Resource Links (footer section)</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid gap-4 md:grid-cols-2">
            <div>
              <Label>Section title</Label>
              <Input
                value={resourceLinks.title}
                onChange={(e) =>
                  update('cheap_resource_links', 'links', {
                    ...resourceLinks,
                    title: e.target.value,
                  })
                }
              />
            </div>
            <div>
              <Label>Section description (optional)</Label>
              <Input
                value={resourceLinks.description}
                onChange={(e) =>
                  update('cheap_resource_links', 'links', {
                    ...resourceLinks,
                    description: e.target.value,
                  })
                }
              />
            </div>
          </div>
          <LinkListEditor
            value={resourceLinks.links}
            onChange={(links) =>
              update('cheap_resource_links', 'links', {
                ...resourceLinks,
                links,
              })
            }
          />
        </CardContent>
      </Card>

      <div className="flex justify-end pb-10">
        <Button onClick={saveBlocks} disabled={processing}><Save className="h-4 w-4 mr-2" />Save All Changes</Button>
      </div>
    </AdminLayout>
  );
}



