import React from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import {
    Tag,
    ArrowLeft,
    Calendar,
    Info,
    DollarSign,
    Percent,
    Gift,
    Cake,
    Clock,
    CalendarDays,
} from "lucide-react";
import { Separator } from "@/Components/ui/Separator";
import Alert from "@/Components/ui/Alert";
import { Badge } from "@/Components/ui/Badge";

const GenerateAnniversary = ({ discountTypes }) => {
    const { data, setData, post, processing, errors, reset } = useForm({
        discount_type: "percentage",
        discount_amount: "15",
        prefix: "ANNIV",
        years_active: "1",
        valid_days: "7",
        max_uses: "1",
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.coupons.generate.anniversary"), {
            onSuccess: () => {
                reset();
            },
        });
    };

    return (
        <AdminLayout>
            <Head title="Generate Anniversary Coupon" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Cake className="w-6 h-6 mr-2 text-pink-500" />
                                Generate Anniversary Coupon
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create special discount coupons for customer
                                account anniversaries
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Coupons
                                </Link>
                            </Button>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit}>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <Card className="md:col-span-2">
                                <CardHeader>
                                    <CardTitle>Anniversary Discount</CardTitle>
                                    <CardDescription>
                                        Configure the discount for customer
                                        account anniversaries
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-6">
                                        {/* Discount Type Selection */}
                                        <div className="mb-6">
                                            <Label htmlFor="discount_type">
                                                Discount Type
                                            </Label>
                                            <div className="mt-1 flex gap-4">
                                                {Object.entries(
                                                    discountTypes
                                                ).map(([key, label]) => (
                                                    <Label
                                                        key={key}
                                                        htmlFor={`discount_type_${key}`}
                                                        className="flex items-center space-x-2 cursor-pointer"
                                                    >
                                                        <input
                                                            type="radio"
                                                            id={`discount_type_${key}`}
                                                            name="discount_type"
                                                            value={key}
                                                            checked={
                                                                data.discount_type ===
                                                                key
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "discount_type",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                                                        />
                                                        <span className="flex items-center">
                                                            {key ===
                                                            "percentage" ? (
                                                                <Percent className="h-4 w-4 mr-1 text-green-600" />
                                                            ) : (
                                                                <DollarSign className="h-4 w-4 mr-1 text-green-600" />
                                                            )}
                                                            {label}
                                                        </span>
                                                    </Label>
                                                ))}
                                            </div>
                                        </div>

                                        {/* Discount Amount */}
                                        <div>
                                            <Label
                                                htmlFor="discount_amount"
                                                className="required"
                                            >
                                                Discount Amount
                                            </Label>
                                            <div className="mt-1 relative">
                                                <Input
                                                    id="discount_amount"
                                                    type="number"
                                                    value={data.discount_amount}
                                                    onChange={(e) =>
                                                        setData(
                                                            "discount_amount",
                                                            e.target.value
                                                        )
                                                    }
                                                    required
                                                    min="0.01"
                                                    step={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "1"
                                                            : "0.01"
                                                    }
                                                    className="pl-7"
                                                    placeholder={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "15"
                                                            : "10.00"
                                                    }
                                                />
                                                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                    {data.discount_type ===
                                                    "percentage" ? (
                                                        <Percent className="h-4 w-4 text-gray-400" />
                                                    ) : (
                                                        <DollarSign className="h-4 w-4 text-gray-400" />
                                                    )}
                                                </div>
                                            </div>
                                            {errors.discount_amount && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.discount_amount}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                {data.discount_type ===
                                                "percentage"
                                                    ? "Enter a percentage value (e.g., 15 for 15% off)"
                                                    : "Enter a fixed amount (e.g., 10.00 for $10 off)"}
                                            </p>
                                        </div>

                                        {/* Anniversary Year */}
                                        <div>
                                            <Label
                                                htmlFor="years_active"
                                                className="required"
                                            >
                                                Years of Membership Required
                                            </Label>
                                            <div className="mt-1 relative">
                                                <Input
                                                    id="years_active"
                                                    type="number"
                                                    min="1"
                                                    max="20"
                                                    value={data.years_active}
                                                    onChange={(e) =>
                                                        setData(
                                                            "years_active",
                                                            e.target.value
                                                        )
                                                    }
                                                    required
                                                    className="pl-7"
                                                />
                                                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                    <CalendarDays className="h-4 w-4 text-gray-400" />
                                                </div>
                                            </div>
                                            {errors.years_active && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.years_active}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Minimum years of membership
                                                needed to receive this
                                                anniversary coupon
                                            </p>
                                        </div>

                                        {/* Coupon Validity Duration */}
                                        <div>
                                            <Label
                                                htmlFor="valid_days"
                                                className="required"
                                            >
                                                Coupon Validity (Days)
                                            </Label>
                                            <div className="mt-1 relative">
                                                <Input
                                                    id="valid_days"
                                                    type="number"
                                                    min="1"
                                                    max="30"
                                                    value={data.valid_days}
                                                    onChange={(e) =>
                                                        setData(
                                                            "valid_days",
                                                            e.target.value
                                                        )
                                                    }
                                                    required
                                                    className="pl-7"
                                                    placeholder="7"
                                                />
                                                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                    <Clock className="h-4 w-4 text-gray-400" />
                                                </div>
                                            </div>
                                            {errors.valid_days && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.valid_days}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Number of days the coupon will
                                                be valid after the anniversary
                                                date
                                            </p>
                                        </div>

                                        {/* Information Panel */}
                                        <div className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                                            <div className="flex">
                                                <Gift className="h-5 w-5 text-blue-500 mt-0.5 mr-3 flex-shrink-0" />
                                                <div>
                                                    <h4 className="text-sm font-medium text-blue-800 dark:text-blue-300">
                                                        How Anniversary Coupons
                                                        Work
                                                    </h4>
                                                    <p className="mt-1 text-sm text-blue-700 dark:text-blue-400">
                                                        These coupons will be
                                                        automatically offered to
                                                        customers on the
                                                        anniversary of their
                                                        account creation,
                                                        provided they've been
                                                        members for at least{" "}
                                                        {data.years_active}{" "}
                                                        {parseInt(
                                                            data.years_active
                                                        ) === 1
                                                            ? "year"
                                                            : "years"}
                                                        .
                                                    </p>
                                                    <p className="mt-2 text-sm text-blue-700 dark:text-blue-400">
                                                        The coupon will be valid
                                                        for {data.valid_days}{" "}
                                                        {parseInt(
                                                            data.valid_days
                                                        ) === 1
                                                            ? "day"
                                                            : "days"}
                                                        and will provide{" "}
                                                        {data.discount_type ===
                                                        "percentage"
                                                            ? `${data.discount_amount}% off`
                                                            : `$${parseFloat(
                                                                  data.discount_amount
                                                              ).toFixed(
                                                                  2
                                                              )} off`}
                                                        .
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Coupon Settings</CardTitle>
                                    <CardDescription>
                                        Configure general coupon parameters
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        <div>
                                            <Label htmlFor="prefix">
                                                Code Prefix
                                            </Label>
                                            <Input
                                                id="prefix"
                                                className="mt-1"
                                                value={data.prefix}
                                                onChange={(e) =>
                                                    setData(
                                                        "prefix",
                                                        e.target.value
                                                    )
                                                }
                                                maxLength={10}
                                                placeholder="ANNIV"
                                            />
                                            {errors.prefix && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.prefix}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Optional prefix for generated
                                                code (max 10 chars)
                                            </p>
                                        </div>

                                        <div>
                                            <Label htmlFor="max_uses">
                                                Maximum Uses Per User
                                            </Label>
                                            <Input
                                                id="max_uses"
                                                type="number"
                                                className="mt-1"
                                                min="1"
                                                value={data.max_uses}
                                                onChange={(e) =>
                                                    setData(
                                                        "max_uses",
                                                        e.target.value
                                                    )
                                                }
                                                placeholder="1"
                                            />
                                            {errors.max_uses && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.max_uses}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                How many times each user can use
                                                their anniversary coupon
                                            </p>
                                        </div>

                                        <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg mt-6">
                                            <h4 className="text-sm font-medium text-gray-900 dark:text-gray-100 flex items-center">
                                                <Cake className="w-4 h-4 mr-2 text-pink-500" />
                                                Coupon Preview
                                            </h4>
                                            <div className="mt-3 space-y-2">
                                                <div className="flex items-center justify-between">
                                                    <span className="text-sm text-gray-500 dark:text-gray-400">
                                                        Type:
                                                    </span>
                                                    <Badge
                                                        variant="outline"
                                                        className="bg-pink-100 text-pink-800 dark:bg-pink-900/30 dark:text-pink-300"
                                                    >
                                                        Anniversary
                                                    </Badge>
                                                </div>
                                                <div className="flex items-center justify-between">
                                                    <span className="text-sm text-gray-500 dark:text-gray-400">
                                                        Code Format:
                                                    </span>
                                                    <span className="text-sm font-mono bg-gray-100 dark:bg-gray-700 px-2 py-1 rounded">
                                                        {data.prefix}
                                                        {data.years_active}_XXXX
                                                    </span>
                                                </div>
                                                <div className="flex items-center justify-between">
                                                    <span className="text-sm text-gray-500 dark:text-gray-400">
                                                        Discount:
                                                    </span>
                                                    <span className="text-sm font-medium text-green-600 dark:text-green-400">
                                                        {data.discount_type ===
                                                        "percentage"
                                                            ? `${data.discount_amount}% off`
                                                            : `$${parseFloat(
                                                                  data.discount_amount
                                                              ).toFixed(
                                                                  2
                                                              )} off`}
                                                    </span>
                                                </div>
                                                <div className="flex items-center justify-between">
                                                    <span className="text-sm text-gray-500 dark:text-gray-400">
                                                        Validity:
                                                    </span>
                                                    <span className="text-sm">
                                                        {data.valid_days} days
                                                        from anniversary
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </CardContent>
                                <CardFooter className="flex justify-end">
                                    <Button type="submit" disabled={processing}>
                                        {processing
                                            ? "Generating..."
                                            : "Generate Anniversary Coupon"}
                                    </Button>
                                </CardFooter>
                            </Card>
                        </div>

                        {/* Info alert */}
                        <Alert
                            variant="info"
                            title="Anniversary Coupon Benefits"
                            className="mt-6"
                        >
                            <div className="text-sm">
                                <p className="mb-2">
                                    Anniversary coupons are a powerful tool for
                                    customer retention:
                                </p>
                                <ul className="list-disc pl-5 space-y-1">
                                    <li>
                                        Build customer loyalty by recognizing
                                        their ongoing relationship with your
                                        business
                                    </li>
                                    <li>
                                        Increase retention by providing special
                                        discounts on membership milestones
                                    </li>
                                    <li>
                                        Encourage inactive customers to return
                                        by surprising them with an anniversary
                                        offer
                                    </li>
                                    <li>
                                        Create higher discount tiers for longer
                                        membership durations (e.g., 15% for 1
                                        year, 20% for 2+ years)
                                    </li>
                                </ul>
                            </div>
                        </Alert>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
};

export default GenerateAnniversary;
