import React from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import { DatePicker } from "@/Components/ui/DatePicker";
import {
    Tag,
    ArrowLeft,
    Calendar,
    Info,
    DollarSign,
    Percent,
    Users,
    UserPlus,
    ShoppingCart,
} from "lucide-react";
import { Separator } from "@/Components/ui/Separator";
import Alert from "@/Components/ui/Alert";

const GenerateFirstOrder = ({ discountTypes }) => {
    const { data, setData, post, processing, errors, reset } = useForm({
        discount_type: "percentage",
        discount_amount: "15",
        prefix: "FIRST",
        start_date: null,
        expiry_date: null,
        max_uses: "1000",
        min_order_amount: "",
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.coupons.generate.first-order"), {
            onSuccess: () => {
                reset();
            },
        });
    };

    return (
        <AdminLayout>
            <Head title="Generate First Order Coupon" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <UserPlus className="w-6 h-6 mr-2 text-blue-500" />
                                Generate First Order Coupon
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create a discount coupon for new customers
                                making their first purchase
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Coupons
                                </Link>
                            </Button>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit}>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <Card className="md:col-span-2">
                                <CardHeader>
                                    <CardTitle>First Order Discount</CardTitle>
                                    <CardDescription>
                                        Configure the discount for first-time
                                        customers
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-6">
                                        {/* Discount Type Selection */}
                                        <div className="mb-6">
                                            <Label htmlFor="discount_type">
                                                Discount Type
                                            </Label>
                                            <div className="mt-1 flex gap-4">
                                                {Object.entries(
                                                    discountTypes
                                                ).map(([key, label]) => (
                                                    <Label
                                                        key={key}
                                                        htmlFor={`discount_type_${key}`}
                                                        className="flex items-center space-x-2 cursor-pointer"
                                                    >
                                                        <input
                                                            type="radio"
                                                            id={`discount_type_${key}`}
                                                            name="discount_type"
                                                            value={key}
                                                            checked={
                                                                data.discount_type ===
                                                                key
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "discount_type",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                                                        />
                                                        <span className="flex items-center">
                                                            {key ===
                                                            "percentage" ? (
                                                                <Percent className="h-4 w-4 mr-1 text-green-600" />
                                                            ) : (
                                                                <DollarSign className="h-4 w-4 mr-1 text-green-600" />
                                                            )}
                                                            {label}
                                                        </span>
                                                    </Label>
                                                ))}
                                            </div>
                                        </div>

                                        {/* Discount Amount */}
                                        <div>
                                            <Label
                                                htmlFor="discount_amount"
                                                className="required"
                                            >
                                                Discount Amount
                                            </Label>
                                            <div className="mt-1 relative">
                                                <Input
                                                    id="discount_amount"
                                                    type="number"
                                                    value={data.discount_amount}
                                                    onChange={(e) =>
                                                        setData(
                                                            "discount_amount",
                                                            e.target.value
                                                        )
                                                    }
                                                    required
                                                    min="0.01"
                                                    step={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "1"
                                                            : "0.01"
                                                    }
                                                    className="pl-7"
                                                    placeholder={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "15"
                                                            : "10.00"
                                                    }
                                                />
                                                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                    {data.discount_type ===
                                                    "percentage" ? (
                                                        <Percent className="h-4 w-4 text-gray-400" />
                                                    ) : (
                                                        <DollarSign className="h-4 w-4 text-gray-400" />
                                                    )}
                                                </div>
                                            </div>
                                            {errors.discount_amount && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.discount_amount}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                {data.discount_type ===
                                                "percentage"
                                                    ? "Enter a percentage value (e.g., 15 for 15% off)"
                                                    : "Enter a fixed amount (e.g., 10.00 for $10 off)"}
                                            </p>
                                        </div>

                                        {/* Minimum Order Amount (Optional) */}
                                        <div>
                                            <Label htmlFor="min_order_amount">
                                                Minimum Order Amount (Optional)
                                            </Label>
                                            <div className="mt-1 relative">
                                                <Input
                                                    id="min_order_amount"
                                                    type="number"
                                                    value={
                                                        data.min_order_amount
                                                    }
                                                    onChange={(e) =>
                                                        setData(
                                                            "min_order_amount",
                                                            e.target.value
                                                        )
                                                    }
                                                    min="0"
                                                    step="0.01"
                                                    className="pl-7"
                                                    placeholder="0.00"
                                                />
                                                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                    <DollarSign className="h-4 w-4 text-gray-400" />
                                                </div>
                                            </div>
                                            {errors.min_order_amount && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.min_order_amount}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Minimum order amount required to
                                                use this coupon. Leave empty for
                                                no minimum.
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Coupon Settings</CardTitle>
                                    <CardDescription>
                                        Configure general coupon parameters
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        <div>
                                            <Label htmlFor="prefix">
                                                Code Prefix
                                            </Label>
                                            <Input
                                                id="prefix"
                                                className="mt-1"
                                                value={data.prefix}
                                                onChange={(e) =>
                                                    setData(
                                                        "prefix",
                                                        e.target.value
                                                    )
                                                }
                                                maxLength={10}
                                                placeholder="FIRST"
                                            />
                                            {errors.prefix && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.prefix}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Optional prefix for generated
                                                code (max 10 chars)
                                            </p>
                                        </div>

                                        <div>
                                            <Label htmlFor="start_date">
                                                Valid From
                                            </Label>
                                            <DatePicker
                                                id="start_date"
                                                className="mt-1"
                                                date={data.start_date}
                                                onSelect={(date) =>
                                                    setData("start_date", date)
                                                }
                                                placeholder="Select start date..."
                                            />
                                            {errors.start_date && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.start_date}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <Label htmlFor="expiry_date">
                                                Expiry Date
                                            </Label>
                                            <DatePicker
                                                id="expiry_date"
                                                className="mt-1"
                                                date={data.expiry_date}
                                                onSelect={(date) =>
                                                    setData("expiry_date", date)
                                                }
                                                placeholder="Select expiry date..."
                                            />
                                            {errors.expiry_date && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.expiry_date}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <Label htmlFor="max_uses">
                                                Maximum Uses
                                            </Label>
                                            <Input
                                                id="max_uses"
                                                type="number"
                                                className="mt-1"
                                                min="1"
                                                value={data.max_uses}
                                                onChange={(e) =>
                                                    setData(
                                                        "max_uses",
                                                        e.target.value
                                                    )
                                                }
                                                placeholder="1000"
                                            />
                                            {errors.max_uses && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.max_uses}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Leave empty for unlimited uses
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                                <CardFooter className="flex justify-end">
                                    <Button type="submit" disabled={processing}>
                                        {processing
                                            ? "Generating..."
                                            : "Generate Coupon"}
                                    </Button>
                                </CardFooter>
                            </Card>
                        </div>

                        {/* Info alert */}
                        <Alert
                            variant="info"
                            title="How First Order Coupons Work"
                            className="mt-6"
                        >
                            <div className="text-sm">
                                <p className="mb-2">
                                    First order coupons are automatically
                                    restricted to customers who haven't placed
                                    an order before:
                                </p>
                                <ul className="list-disc pl-5 space-y-1">
                                    <li>
                                        These coupons will only be usable by
                                        customers with no previous orders
                                    </li>
                                    <li>
                                        They're perfect for attracting new
                                        customers with a special welcome
                                        discount
                                    </li>
                                    <li>
                                        Setting a minimum order amount can help
                                        ensure profitability
                                    </li>
                                    <li>
                                        The system will automatically suggest
                                        these coupons to new customers during
                                        checkout
                                    </li>
                                </ul>
                            </div>
                        </Alert>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
};

export default GenerateFirstOrder;
