import React from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import { DatePicker } from "@/Components/ui/DatePicker";
import {
    Tag,
    ArrowLeft,
    Calendar,
    Info,
    DollarSign,
    Percent,
    Clock,
    Zap,
    Timer,
    Users,
    PieChart,
} from "lucide-react";
import { Separator } from "@/Components/ui/Separator";
import Alert from "@/Components/ui/Alert";

const GenerateFlash = ({ discountTypes }) => {
    const { data, setData, post, processing, errors, reset } = useForm({
        discount_type: "percentage",
        discount_amount: "20",
        prefix: "FLASH",
        start_date: new Date(),
        duration_hours: "24",
        max_uses: "100",
        per_user_limit: "1",
        min_order_amount: "",
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.coupons.generate.flash"), {
            onSuccess: () => {
                reset();
            },
        });
    };

    return (
        <AdminLayout>
            <Head title="Generate Flash Sale Coupon" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Zap className="w-6 h-6 mr-2 text-amber-500" />
                                Generate Flash Sale Coupon
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create a time-limited flash sale discount coupon
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Coupons
                                </Link>
                            </Button>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit}>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <Card className="md:col-span-2">
                                <CardHeader>
                                    <CardTitle>Flash Sale Settings</CardTitle>
                                    <CardDescription>
                                        Configure the discount and duration of
                                        the flash sale
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-6">
                                        {/* Discount Type Selection */}
                                        <div className="mb-6">
                                            <Label htmlFor="discount_type">
                                                Discount Type
                                            </Label>
                                            <div className="mt-1 flex gap-4">
                                                {Object.entries(
                                                    discountTypes
                                                ).map(([key, label]) => (
                                                    <Label
                                                        key={key}
                                                        htmlFor={`discount_type_${key}`}
                                                        className="flex items-center space-x-2 cursor-pointer"
                                                    >
                                                        <input
                                                            type="radio"
                                                            id={`discount_type_${key}`}
                                                            name="discount_type"
                                                            value={key}
                                                            checked={
                                                                data.discount_type ===
                                                                key
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "discount_type",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                                                        />
                                                        <span className="flex items-center">
                                                            {key ===
                                                            "percentage" ? (
                                                                <Percent className="h-4 w-4 mr-1 text-green-600" />
                                                            ) : (
                                                                <DollarSign className="h-4 w-4 mr-1 text-green-600" />
                                                            )}
                                                            {label}
                                                        </span>
                                                    </Label>
                                                ))}
                                            </div>
                                        </div>

                                        {/* Discount Amount */}
                                        <div>
                                            <Label
                                                htmlFor="discount_amount"
                                                className="required"
                                            >
                                                Discount Amount
                                            </Label>
                                            <div className="mt-1 relative">
                                                <Input
                                                    id="discount_amount"
                                                    type="number"
                                                    value={data.discount_amount}
                                                    onChange={(e) =>
                                                        setData(
                                                            "discount_amount",
                                                            e.target.value
                                                        )
                                                    }
                                                    required
                                                    min="0.01"
                                                    step={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "1"
                                                            : "0.01"
                                                    }
                                                    className="pl-7"
                                                    placeholder={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "20"
                                                            : "15.00"
                                                    }
                                                />
                                                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                    {data.discount_type ===
                                                    "percentage" ? (
                                                        <Percent className="h-4 w-4 text-gray-400" />
                                                    ) : (
                                                        <DollarSign className="h-4 w-4 text-gray-400" />
                                                    )}
                                                </div>
                                            </div>
                                            {errors.discount_amount && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.discount_amount}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                {data.discount_type ===
                                                "percentage"
                                                    ? "Enter a percentage value (e.g., 20 for 20% off)"
                                                    : "Enter a fixed amount (e.g., 15.00 for $15 off)"}
                                            </p>
                                        </div>

                                        {/* Timing Settings */}
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <Label
                                                    htmlFor="start_date"
                                                    className="required"
                                                >
                                                    Start Date
                                                </Label>
                                                <DatePicker
                                                    id="start_date"
                                                    className="mt-1"
                                                    date={data.start_date}
                                                    onSelect={(date) =>
                                                        setData(
                                                            "start_date",
                                                            date
                                                        )
                                                    }
                                                    placeholder="Select start date..."
                                                    required
                                                />
                                                {errors.start_date && (
                                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                        {errors.start_date}
                                                    </p>
                                                )}
                                            </div>

                                            <div>
                                                <Label
                                                    htmlFor="duration_hours"
                                                    className="required"
                                                >
                                                    Duration (Hours)
                                                </Label>
                                                <div className="mt-1 relative">
                                                    <Input
                                                        id="duration_hours"
                                                        type="number"
                                                        value={
                                                            data.duration_hours
                                                        }
                                                        onChange={(e) =>
                                                            setData(
                                                                "duration_hours",
                                                                e.target.value
                                                            )
                                                        }
                                                        required
                                                        min="1"
                                                        max="720"
                                                        className="pl-7"
                                                        placeholder="24"
                                                    />
                                                    <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                        <Clock className="h-4 w-4 text-gray-400" />
                                                    </div>
                                                </div>
                                                {errors.duration_hours && (
                                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                        {errors.duration_hours}
                                                    </p>
                                                )}
                                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                    How long the flash sale will
                                                    last (1-720 hours)
                                                </p>
                                            </div>
                                        </div>

                                        {/* Usage Limits */}
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <Label htmlFor="max_uses">
                                                    Maximum Total Uses
                                                </Label>
                                                <div className="mt-1 relative">
                                                    <Input
                                                        id="max_uses"
                                                        type="number"
                                                        value={data.max_uses}
                                                        onChange={(e) =>
                                                            setData(
                                                                "max_uses",
                                                                e.target.value
                                                            )
                                                        }
                                                        min="1"
                                                        className="pl-7"
                                                        placeholder="100"
                                                    />
                                                    <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                        <Users className="h-4 w-4 text-gray-400" />
                                                    </div>
                                                </div>
                                                {errors.max_uses && (
                                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                        {errors.max_uses}
                                                    </p>
                                                )}
                                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                    Maximum number of times this
                                                    coupon can be used
                                                </p>
                                            </div>

                                            <div>
                                                <Label htmlFor="per_user_limit">
                                                    Per-User Limit
                                                </Label>
                                                <Input
                                                    id="per_user_limit"
                                                    type="number"
                                                    className="mt-1"
                                                    min="1"
                                                    value={data.per_user_limit}
                                                    onChange={(e) =>
                                                        setData(
                                                            "per_user_limit",
                                                            e.target.value
                                                        )
                                                    }
                                                    placeholder="1"
                                                />
                                                {errors.per_user_limit && (
                                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                        {errors.per_user_limit}
                                                    </p>
                                                )}
                                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                    Number of times each user
                                                    can use this coupon
                                                </p>
                                            </div>
                                        </div>

                                        {/* Minimum Order Amount (Optional) */}
                                        <div>
                                            <Label htmlFor="min_order_amount">
                                                Minimum Order Amount (Optional)
                                            </Label>
                                            <div className="mt-1 relative">
                                                <Input
                                                    id="min_order_amount"
                                                    type="number"
                                                    value={
                                                        data.min_order_amount
                                                    }
                                                    onChange={(e) =>
                                                        setData(
                                                            "min_order_amount",
                                                            e.target.value
                                                        )
                                                    }
                                                    min="0"
                                                    step="0.01"
                                                    className="pl-7"
                                                    placeholder="0.00"
                                                />
                                                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                    <DollarSign className="h-4 w-4 text-gray-400" />
                                                </div>
                                            </div>
                                            {errors.min_order_amount && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.min_order_amount}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Minimum order amount required to
                                                use this coupon
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Coupon Settings</CardTitle>
                                    <CardDescription>
                                        Configure additional coupon parameters
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        <div>
                                            <Label htmlFor="prefix">
                                                Code Prefix
                                            </Label>
                                            <Input
                                                id="prefix"
                                                className="mt-1"
                                                value={data.prefix}
                                                onChange={(e) =>
                                                    setData(
                                                        "prefix",
                                                        e.target.value
                                                    )
                                                }
                                                maxLength={10}
                                                placeholder="FLASH"
                                            />
                                            {errors.prefix && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.prefix}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Optional prefix for generated
                                                code (max 10 chars)
                                            </p>
                                        </div>

                                        <div>
                                            <div className="flex items-center space-x-2 mt-4">
                                                <PieChart className="h-5 w-5 text-blue-500" />
                                                <span className="font-medium text-sm text-gray-700 dark:text-gray-300">
                                                    Flash Sale Duration Preview
                                                </span>
                                            </div>
                                            <div className="mt-2 p-3 bg-gray-50 dark:bg-gray-800 rounded-md">
                                                <div className="flex justify-between text-sm">
                                                    <span className="text-gray-600 dark:text-gray-400">
                                                        Starts:
                                                    </span>
                                                    <span className="font-medium text-gray-800 dark:text-gray-200">
                                                        {data.start_date
                                                            ? data.start_date.toLocaleDateString()
                                                            : "Today"}
                                                    </span>
                                                </div>
                                                <div className="flex justify-between text-sm mt-1">
                                                    <span className="text-gray-600 dark:text-gray-400">
                                                        Ends:
                                                    </span>
                                                    <span className="font-medium text-gray-800 dark:text-gray-200">
                                                        {data.start_date &&
                                                        data.duration_hours
                                                            ? new Date(
                                                                  data.start_date.getTime() +
                                                                      parseInt(
                                                                          data.duration_hours
                                                                      ) *
                                                                          60 *
                                                                          60 *
                                                                          1000
                                                              ).toLocaleDateString()
                                                            : "Tomorrow"}
                                                    </span>
                                                </div>
                                                <div className="flex justify-between text-sm mt-1">
                                                    <span className="text-gray-600 dark:text-gray-400">
                                                        Duration:
                                                    </span>
                                                    <span className="font-medium text-gray-800 dark:text-gray-200">
                                                        {data.duration_hours}{" "}
                                                        hours
                                                        {data.duration_hours >=
                                                        24
                                                            ? ` (${(
                                                                  data.duration_hours /
                                                                  24
                                                              ).toFixed(
                                                                  1
                                                              )} days)`
                                                            : ""}
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </CardContent>
                                <CardFooter className="flex justify-end">
                                    <Button type="submit" disabled={processing}>
                                        {processing
                                            ? "Generating..."
                                            : "Generate Flash Coupon"}
                                    </Button>
                                </CardFooter>
                            </Card>
                        </div>

                        {/* Info alert */}
                        <Alert
                            variant="warning"
                            title="Flash Sale Best Practices"
                            className="mt-6"
                        >
                            <div className="text-sm">
                                <p className="mb-2">
                                    Flash sales create urgency and can drive
                                    immediate sales. Here are some tips:
                                </p>
                                <ul className="list-disc pl-5 space-y-1">
                                    <li>
                                        Keep the duration short (24-48 hours) to
                                        create a sense of urgency
                                    </li>
                                    <li>
                                        Consider limiting total uses to create
                                        scarcity
                                    </li>
                                    <li>
                                        Use higher discount percentages than
                                        regular promotions (20-30%)
                                    </li>
                                    <li>
                                        Promote the flash sale through email and
                                        social media to maximize visibility
                                    </li>
                                    <li>
                                        Set a reasonable minimum order amount to
                                        maintain profitability
                                    </li>
                                </ul>
                            </div>
                        </Alert>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
};

export default GenerateFlash;
