import React, { useState } from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import { DatePicker } from "@/Components/ui/DatePicker";
import {
    Tag,
    ArrowLeft,
    Calendar,
    Info,
    DollarSign,
    Percent,
    Users,
    UserPlus,
    Share2,
    Search,
    CheckCircle,
} from "lucide-react";
import { Separator } from "@/Components/ui/Separator";
import { Checkbox } from "@/Components/ui/Checkbox";
import Alert from "@/Components/ui/Alert";

const GenerateReferral = ({ discountTypes, clients }) => {
    const { data, setData, post, processing, errors, reset } = useForm({
        discount_type: "percentage",
        discount_amount: "15",
        prefix: "REF",
        start_date: null,
        expiry_date: null,
        max_uses: "10",
        per_user_limit: "1",
        user_specific: false,
        referrer_user_id: "",
    });

    const [searchTerm, setSearchTerm] = useState("");
    const [showClientList, setShowClientList] = useState(false);

    const filteredClients = searchTerm
        ? clients.filter(
              (client) =>
                  client.name
                      .toLowerCase()
                      .includes(searchTerm.toLowerCase()) ||
                  client.email.toLowerCase().includes(searchTerm.toLowerCase())
          )
        : [];

    const selectClient = (client) => {
        setData("referrer_user_id", client.id);
        setSearchTerm(`${client.name} (${client.email})`);
        setShowClientList(false);
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.coupons.generate.referral"), {
            onSuccess: () => {
                reset();
                setSearchTerm("");
            },
        });
    };

    const handleUserSpecificChange = (checked) => {
        setData((data) => ({
            ...data,
            user_specific: checked,
            referrer_user_id: checked ? data.referrer_user_id : "",
        }));

        if (!checked) {
            setSearchTerm("");
        }
    };

    return (
        <AdminLayout>
            <Head title="Generate Referral Coupon" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Share2 className="w-6 h-6 mr-2 text-indigo-500" />
                                Generate Referral Coupon
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create referral coupons to incentivize your
                                customers to refer others
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Coupons
                                </Link>
                            </Button>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit}>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <Card className="md:col-span-2">
                                <CardHeader>
                                    <CardTitle>Referral Discount</CardTitle>
                                    <CardDescription>
                                        Configure the discount for referred
                                        customers
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-6">
                                        {/* Discount Type Selection */}
                                        <div className="mb-6">
                                            <Label htmlFor="discount_type">
                                                Discount Type
                                            </Label>
                                            <div className="mt-1 flex gap-4">
                                                {Object.entries(
                                                    discountTypes
                                                ).map(([key, label]) => (
                                                    <Label
                                                        key={key}
                                                        htmlFor={`discount_type_${key}`}
                                                        className="flex items-center space-x-2 cursor-pointer"
                                                    >
                                                        <input
                                                            type="radio"
                                                            id={`discount_type_${key}`}
                                                            name="discount_type"
                                                            value={key}
                                                            checked={
                                                                data.discount_type ===
                                                                key
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "discount_type",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                                                        />
                                                        <span className="flex items-center">
                                                            {key ===
                                                            "percentage" ? (
                                                                <Percent className="h-4 w-4 mr-1 text-green-600" />
                                                            ) : (
                                                                <DollarSign className="h-4 w-4 mr-1 text-green-600" />
                                                            )}
                                                            {label}
                                                        </span>
                                                    </Label>
                                                ))}
                                            </div>
                                        </div>

                                        {/* Discount Amount */}
                                        <div>
                                            <Label
                                                htmlFor="discount_amount"
                                                className="required"
                                            >
                                                Discount Amount
                                            </Label>
                                            <div className="mt-1 relative">
                                                <Input
                                                    id="discount_amount"
                                                    type="number"
                                                    value={data.discount_amount}
                                                    onChange={(e) =>
                                                        setData(
                                                            "discount_amount",
                                                            e.target.value
                                                        )
                                                    }
                                                    required
                                                    min="0.01"
                                                    step={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "1"
                                                            : "0.01"
                                                    }
                                                    className="pl-7"
                                                    placeholder={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "15"
                                                            : "10.00"
                                                    }
                                                />
                                                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                                    {data.discount_type ===
                                                    "percentage" ? (
                                                        <Percent className="h-4 w-4 text-gray-400" />
                                                    ) : (
                                                        <DollarSign className="h-4 w-4 text-gray-400" />
                                                    )}
                                                </div>
                                            </div>
                                            {errors.discount_amount && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.discount_amount}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                {data.discount_type ===
                                                "percentage"
                                                    ? "Enter a percentage value (e.g., 15 for 15% off)"
                                                    : "Enter a fixed amount (e.g., 10.00 for $10 off)"}
                                            </p>
                                        </div>

                                        {/* Usage Limits */}
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <Label htmlFor="max_uses">
                                                    Maximum Total Uses
                                                </Label>
                                                <Input
                                                    id="max_uses"
                                                    type="number"
                                                    className="mt-1"
                                                    min="1"
                                                    value={data.max_uses}
                                                    onChange={(e) =>
                                                        setData(
                                                            "max_uses",
                                                            e.target.value
                                                        )
                                                    }
                                                    placeholder="10"
                                                />
                                                {errors.max_uses && (
                                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                        {errors.max_uses}
                                                    </p>
                                                )}
                                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                    How many times this referral
                                                    code can be used in total
                                                </p>
                                            </div>

                                            <div>
                                                <Label htmlFor="per_user_limit">
                                                    Per-User Limit
                                                </Label>
                                                <Input
                                                    id="per_user_limit"
                                                    type="number"
                                                    className="mt-1"
                                                    min="1"
                                                    value={data.per_user_limit}
                                                    onChange={(e) =>
                                                        setData(
                                                            "per_user_limit",
                                                            e.target.value
                                                        )
                                                    }
                                                    placeholder="1"
                                                />
                                                {errors.per_user_limit && (
                                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                        {errors.per_user_limit}
                                                    </p>
                                                )}
                                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                    How many times each referred
                                                    customer can use this code
                                                </p>
                                            </div>
                                        </div>

                                        {/* Specific Referrer Options */}
                                        <div className="p-4 bg-gray-50 dark:bg-gray-800 rounded-lg">
                                            <div className="flex items-center space-x-2 mb-4">
                                                <Checkbox
                                                    id="user_specific"
                                                    checked={data.user_specific}
                                                    onCheckedChange={
                                                        handleUserSpecificChange
                                                    }
                                                />
                                                <Label
                                                    htmlFor="user_specific"
                                                    className="font-medium cursor-pointer"
                                                >
                                                    Assign to specific client
                                                </Label>
                                            </div>

                                            {data.user_specific && (
                                                <div className="mt-3">
                                                    <Label
                                                        htmlFor="client_search"
                                                        className="required"
                                                    >
                                                        Select Client
                                                    </Label>
                                                    <div className="relative mt-1">
                                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                            <Search className="h-4 w-4 text-gray-400" />
                                                        </div>
                                                        <Input
                                                            id="client_search"
                                                            type="text"
                                                            placeholder="Search by name or email..."
                                                            className="pl-10"
                                                            value={searchTerm}
                                                            onChange={(e) => {
                                                                setSearchTerm(
                                                                    e.target
                                                                        .value
                                                                );
                                                                setShowClientList(
                                                                    true
                                                                );
                                                                if (
                                                                    e.target
                                                                        .value ===
                                                                    ""
                                                                ) {
                                                                    setData(
                                                                        "referrer_user_id",
                                                                        ""
                                                                    );
                                                                }
                                                            }}
                                                            onFocus={() => {
                                                                if (searchTerm)
                                                                    setShowClientList(
                                                                        true
                                                                    );
                                                            }}
                                                            required={
                                                                data.user_specific
                                                            }
                                                        />

                                                        {/* Client search results dropdown */}
                                                        {showClientList &&
                                                            searchTerm && (
                                                                <div className="absolute z-10 w-full mt-1 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md shadow-lg max-h-60 overflow-auto">
                                                                    {filteredClients.length >
                                                                    0 ? (
                                                                        filteredClients.map(
                                                                            (
                                                                                client
                                                                            ) => (
                                                                                <div
                                                                                    key={
                                                                                        client.id
                                                                                    }
                                                                                    className="p-2 hover:bg-gray-100 dark:hover:bg-gray-700 cursor-pointer flex justify-between items-center"
                                                                                    onClick={() =>
                                                                                        selectClient(
                                                                                            client
                                                                                        )
                                                                                    }
                                                                                >
                                                                                    <div>
                                                                                        <div className="font-medium">
                                                                                            {
                                                                                                client.name
                                                                                            }
                                                                                        </div>
                                                                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                                            {
                                                                                                client.email
                                                                                            }
                                                                                        </div>
                                                                                    </div>
                                                                                    {data.referrer_user_id ===
                                                                                        client.id && (
                                                                                        <CheckCircle className="h-4 w-4 text-green-500" />
                                                                                    )}
                                                                                </div>
                                                                            )
                                                                        )
                                                                    ) : (
                                                                        <div className="p-2 text-gray-500 dark:text-gray-400">
                                                                            No
                                                                            clients
                                                                            found
                                                                        </div>
                                                                    )}
                                                                </div>
                                                            )}
                                                    </div>
                                                    {errors.referrer_user_id && (
                                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                            {
                                                                errors.referrer_user_id
                                                            }
                                                        </p>
                                                    )}
                                                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                        This referral code will
                                                        be exclusive to the
                                                        selected client
                                                    </p>
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Coupon Settings</CardTitle>
                                    <CardDescription>
                                        Configure general coupon parameters
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        <div>
                                            <Label htmlFor="prefix">
                                                Code Prefix
                                            </Label>
                                            <Input
                                                id="prefix"
                                                className="mt-1"
                                                value={data.prefix}
                                                onChange={(e) =>
                                                    setData(
                                                        "prefix",
                                                        e.target.value
                                                    )
                                                }
                                                maxLength={10}
                                                placeholder="REF"
                                            />
                                            {errors.prefix && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.prefix}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Optional prefix for generated
                                                code (max 10 chars)
                                            </p>
                                        </div>

                                        <div>
                                            <Label htmlFor="start_date">
                                                Valid From
                                            </Label>
                                            <DatePicker
                                                id="start_date"
                                                className="mt-1"
                                                date={data.start_date}
                                                onSelect={(date) =>
                                                    setData("start_date", date)
                                                }
                                                placeholder="Select start date..."
                                            />
                                            {errors.start_date && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.start_date}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <Label htmlFor="expiry_date">
                                                Expiry Date
                                            </Label>
                                            <DatePicker
                                                id="expiry_date"
                                                className="mt-1"
                                                date={data.expiry_date}
                                                onSelect={(date) =>
                                                    setData("expiry_date", date)
                                                }
                                                placeholder="Select expiry date..."
                                            />
                                            {errors.expiry_date && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.expiry_date}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Recommended: 3 months from now
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                                <CardFooter className="flex justify-end">
                                    <Button type="submit" disabled={processing}>
                                        {processing
                                            ? "Generating..."
                                            : "Generate Referral Coupon"}
                                    </Button>
                                </CardFooter>
                            </Card>
                        </div>

                        {/* Info alert */}
                        <Alert
                            variant="info"
                            title="How Referral Coupons Work"
                            className="mt-6"
                        >
                            <div className="text-sm">
                                <p className="mb-2">
                                    Referral coupons help grow your customer
                                    base through word of mouth:
                                </p>
                                <ul className="list-disc pl-5 space-y-1">
                                    <li>
                                        <strong>
                                            General referral coupons
                                        </strong>{" "}
                                        can be shared by any customer
                                    </li>
                                    <li>
                                        <strong>
                                            User-specific referral coupons
                                        </strong>{" "}
                                        are tied to a specific customer,
                                        allowing you to track which customers
                                        are bringing in new business
                                    </li>
                                    <li>
                                        Set appropriate usage limits to prevent
                                        abuse while still encouraging sharing
                                    </li>
                                    <li>
                                        Consider implementing a dual-sided
                                        referral program where both the referrer
                                        and referee get rewards
                                    </li>
                                </ul>
                            </div>
                        </Alert>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
};

export default GenerateReferral;
