import React, { useState } from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import { DatePicker } from "@/Components/ui/DatePicker";
import {
    Tag,
    ArrowLeft,
    Calendar,
    Info,
    DollarSign,
    Percent,
    CalendarRange,
    Plus,
    Minus,
    Check,
    AlertTriangle,
} from "lucide-react";
import { Separator } from "@/Components/ui/Separator";
import Alert from "@/Components/ui/Alert";
import { Badge } from "@/Components/ui/Badge";
import { Checkbox } from "@/Components/ui/Checkbox";

const GenerateSeasonal = ({ seasonalTypes, discountTypes }) => {
    const { data, setData, post, processing, errors, reset } = useForm({
        discount_type: "percentage",
        seasonal_discounts: {},
        prefix: "SEASON",
        start_date: null,
        expiry_date: null,
        max_uses: "",
    });

    const [showAllSeasons, setShowAllSeasons] = useState(false);
    const [selectedSeasons, setSelectedSeasons] = useState([]);

    // Initialize seasonal discounts
    const initializeSeasonalDiscounts = () => {
        const initialDiscounts = {};
        seasonalTypes.forEach((season) => {
            initialDiscounts[season.id] = "";
        });
        setData("seasonal_discounts", initialDiscounts);
    };

    // Initialize seasonal discounts on component mount
    React.useEffect(() => {
        initializeSeasonalDiscounts();
    }, [seasonalTypes]);

    const handleSeasonalDiscountChange = (seasonId, value) => {
        const newDiscounts = { ...data.seasonal_discounts };
        newDiscounts[seasonId] = value;
        setData("seasonal_discounts", newDiscounts);
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.coupons.generate.seasonal"), {
            onSuccess: () => {
                reset();
                setSelectedSeasons([]);
            },
        });
    };

    // Helper function to apply the same discount to all selected seasons
    const applyToAllSeasons = (value) => {
        const newDiscounts = { ...data.seasonal_discounts };
        selectedSeasons.forEach((seasonId) => {
            newDiscounts[seasonId] = value;
        });
        setData("seasonal_discounts", newDiscounts);
    };

    const toggleSeasonSelection = (seasonId) => {
        setSelectedSeasons((prev) =>
            prev.includes(seasonId)
                ? prev.filter((id) => id !== seasonId)
                : [...prev, seasonId]
        );
    };

    const toggleAllSeasons = () => {
        if (selectedSeasons.length === seasonalTypes.length) {
            setSelectedSeasons([]);
        } else {
            setSelectedSeasons(seasonalTypes.map((season) => season.id));
        }
    };

    // Filter active seasons for display
    const displaySeasons = showAllSeasons
        ? seasonalTypes
        : seasonalTypes.slice(0, 3);

    const formatDateRange = (startDate, endDate) => {
        if (!startDate || !endDate) return "No date range set";

        const formatDate = (date) => {
            const d = new Date(date);
            return d.toLocaleDateString(undefined, {
                month: "short",
                day: "numeric",
            });
        };

        return `${formatDate(startDate)} - ${formatDate(endDate)}`;
    };

    const isCurrentlyActive = (season) => {
        if (!season.default_start_date || !season.default_end_date)
            return false;

        // This is a simplified check - the actual logic would be in your backend
        const now = new Date();
        const currentMonth = now.getMonth() + 1; // JavaScript months are 0-indexed
        const currentDay = now.getDate();

        // Extract month and day from the dates
        const startDate = new Date(season.default_start_date);
        const endDate = new Date(season.default_end_date);

        const startMonth = startDate.getMonth() + 1;
        const startDay = startDate.getDate();

        const endMonth = endDate.getMonth() + 1;
        const endDay = endDate.getDate();

        // Simple check - would need to be enhanced for seasons that span across years
        if (startMonth <= endMonth) {
            // Season within same year
            return (
                (currentMonth > startMonth ||
                    (currentMonth === startMonth && currentDay >= startDay)) &&
                (currentMonth < endMonth ||
                    (currentMonth === endMonth && currentDay <= endDay))
            );
        } else {
            // Season spans across years (e.g., winter: Dec-Feb)
            return currentMonth >= startMonth || currentMonth <= endMonth;
        }
    };

    return (
        <AdminLayout>
            <Head title="Generate Seasonal Coupons" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <CalendarRange className="w-6 h-6 mr-2 text-amber-500" />
                                Generate Seasonal Coupons
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create discount coupons for seasonal promotions
                                and holidays
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Coupons
                                </Link>
                            </Button>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit}>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <Card className="md:col-span-2">
                                <CardHeader>
                                    <CardTitle>Seasonal Discounts</CardTitle>
                                    <CardDescription>
                                        Configure discount amount for each
                                        seasonal period
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        {/* Discount Type Selection */}
                                        <div className="mb-6">
                                            <Label htmlFor="discount_type">
                                                Discount Type
                                            </Label>
                                            <div className="mt-1 flex gap-4">
                                                {Object.entries(
                                                    discountTypes
                                                ).map(([key, label]) => (
                                                    <Label
                                                        key={key}
                                                        htmlFor={`discount_type_${key}`}
                                                        className="flex items-center space-x-2 cursor-pointer"
                                                    >
                                                        <input
                                                            type="radio"
                                                            id={`discount_type_${key}`}
                                                            name="discount_type"
                                                            value={key}
                                                            checked={
                                                                data.discount_type ===
                                                                key
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "discount_type",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                                                        />
                                                        <span className="flex items-center">
                                                            {key ===
                                                            "percentage" ? (
                                                                <Percent className="h-4 w-4 mr-1 text-green-600" />
                                                            ) : (
                                                                <DollarSign className="h-4 w-4 mr-1 text-green-600" />
                                                            )}
                                                            {label}
                                                        </span>
                                                    </Label>
                                                ))}
                                            </div>
                                        </div>

                                        {/* Season Selection and Bulk Actions */}
                                        <div className="mb-6 p-4 bg-gray-50 dark:bg-gray-800/50 rounded-md">
                                            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-4">
                                                <div className="mb-2 sm:mb-0">
                                                    <Label
                                                        htmlFor="select_all"
                                                        className="flex items-center space-x-2 cursor-pointer"
                                                    >
                                                        <Checkbox
                                                            id="select_all"
                                                            checked={
                                                                selectedSeasons.length ===
                                                                    seasonalTypes.length &&
                                                                seasonalTypes.length >
                                                                    0
                                                            }
                                                            onCheckedChange={
                                                                toggleAllSeasons
                                                            }
                                                        />
                                                        <span>
                                                            Select All Seasons
                                                        </span>
                                                    </Label>
                                                </div>
                                                <div className="flex items-center">
                                                    <Input
                                                        id="apply_all"
                                                        type="number"
                                                        className="w-24 mr-2"
                                                        placeholder="Value"
                                                        min="0"
                                                        step={
                                                            data.discount_type ===
                                                            "percentage"
                                                                ? "1"
                                                                : "0.01"
                                                        }
                                                        disabled={
                                                            selectedSeasons.length ===
                                                            0
                                                        }
                                                    />
                                                    <Button
                                                        type="button"
                                                        onClick={(e) =>
                                                            applyToAllSeasons(
                                                                e.target
                                                                    .previousSibling
                                                                    .value
                                                            )
                                                        }
                                                        size="sm"
                                                        disabled={
                                                            selectedSeasons.length ===
                                                            0
                                                        }
                                                    >
                                                        Apply to Selected
                                                    </Button>
                                                </div>
                                            </div>

                                            {selectedSeasons.length > 0 && (
                                                <div className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                                                    {selectedSeasons.length}{" "}
                                                    season
                                                    {selectedSeasons.length !==
                                                    1
                                                        ? "s"
                                                        : ""}{" "}
                                                    selected
                                                </div>
                                            )}
                                        </div>

                                        {/* Season Specific Discounts */}
                                        <div className="space-y-6">
                                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                                {displaySeasons.map(
                                                    (season) => (
                                                        <div
                                                            key={season.id}
                                                            className={`p-4 border rounded-md ${
                                                                selectedSeasons.includes(
                                                                    season.id
                                                                )
                                                                    ? "border-blue-300 dark:border-blue-600 bg-blue-50 dark:bg-blue-900/10"
                                                                    : "border-gray-200 dark:border-gray-700"
                                                            }`}
                                                        >
                                                            <div className="flex justify-between items-start mb-3">
                                                                <div className="flex items-center space-x-2">
                                                                    <Checkbox
                                                                        checked={selectedSeasons.includes(
                                                                            season.id
                                                                        )}
                                                                        onCheckedChange={() =>
                                                                            toggleSeasonSelection(
                                                                                season.id
                                                                            )
                                                                        }
                                                                        id={`season_${season.id}`}
                                                                    />
                                                                    <div>
                                                                        <Label
                                                                            htmlFor={`season_${season.id}`}
                                                                            className="font-medium text-gray-900 dark:text-white cursor-pointer"
                                                                        >
                                                                            {
                                                                                season.name
                                                                            }
                                                                        </Label>
                                                                        <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                            {formatDateRange(
                                                                                season.default_start_date,
                                                                                season.default_end_date
                                                                            )}
                                                                        </p>
                                                                    </div>
                                                                </div>
                                                                {season.is_active && (
                                                                    <Badge
                                                                        variant={
                                                                            isCurrentlyActive(
                                                                                season
                                                                            )
                                                                                ? "success"
                                                                                : "outline"
                                                                        }
                                                                    >
                                                                        {isCurrentlyActive(
                                                                            season
                                                                        )
                                                                            ? "Active Now"
                                                                            : "Active"}
                                                                    </Badge>
                                                                )}
                                                            </div>
                                                            <div className="mt-2">
                                                                <Label
                                                                    htmlFor={`season_discount_${season.id}`}
                                                                    className="text-sm"
                                                                >
                                                                    Discount
                                                                    Amount
                                                                </Label>
                                                                <div className="mt-1 relative">
                                                                    <Input
                                                                        id={`season_discount_${season.id}`}
                                                                        type="number"
                                                                        min="0"
                                                                        step={
                                                                            data.discount_type ===
                                                                            "percentage"
                                                                                ? "1"
                                                                                : "0.01"
                                                                        }
                                                                        value={
                                                                            data
                                                                                .seasonal_discounts[
                                                                                season
                                                                                    .id
                                                                            ] ||
                                                                            ""
                                                                        }
                                                                        onChange={(
                                                                            e
                                                                        ) =>
                                                                            handleSeasonalDiscountChange(
                                                                                season.id,
                                                                                e
                                                                                    .target
                                                                                    .value
                                                                            )
                                                                        }
                                                                        className="pr-8"
                                                                        placeholder={
                                                                            data.discount_type ===
                                                                            "percentage"
                                                                                ? "10"
                                                                                : "5.00"
                                                                        }
                                                                    />
                                                                    <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                                                        {data.discount_type ===
                                                                        "percentage" ? (
                                                                            <Percent className="h-4 w-4 text-gray-400" />
                                                                        ) : (
                                                                            <DollarSign className="h-4 w-4 text-gray-400" />
                                                                        )}
                                                                    </div>
                                                                </div>
                                                                {errors[
                                                                    `seasonal_discounts.${season.id}`
                                                                ] && (
                                                                    <p className="mt-1 text-sm text-red-600">
                                                                        {
                                                                            errors[
                                                                                `seasonal_discounts.${season.id}`
                                                                            ]
                                                                        }
                                                                    </p>
                                                                )}
                                                            </div>
                                                        </div>
                                                    )
                                                )}
                                            </div>

                                            {/* Show/Hide Seasons Toggle */}
                                            {seasonalTypes.length > 3 && (
                                                <div className="flex justify-center">
                                                    <Button
                                                        type="button"
                                                        variant="outline"
                                                        size="sm"
                                                        onClick={() =>
                                                            setShowAllSeasons(
                                                                !showAllSeasons
                                                            )
                                                        }
                                                    >
                                                        {showAllSeasons ? (
                                                            <>
                                                                <Minus className="w-4 h-4 mr-2" />
                                                                Show Less
                                                            </>
                                                        ) : (
                                                            <>
                                                                <Plus className="w-4 h-4 mr-2" />
                                                                Show All{" "}
                                                                {
                                                                    seasonalTypes.length
                                                                }{" "}
                                                                Seasons
                                                            </>
                                                        )}
                                                    </Button>
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Coupon Settings</CardTitle>
                                    <CardDescription>
                                        Configure general coupon parameters
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        <div>
                                            <Label htmlFor="prefix">
                                                Code Prefix
                                            </Label>
                                            <Input
                                                id="prefix"
                                                className="mt-1"
                                                value={data.prefix}
                                                onChange={(e) =>
                                                    setData(
                                                        "prefix",
                                                        e.target.value
                                                    )
                                                }
                                                maxLength={10}
                                                placeholder="SEASON"
                                            />
                                            {errors.prefix && (
                                                <p className="mt-1 text-sm text-red-600">
                                                    {errors.prefix}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Optional prefix for generated
                                                codes (max 10 chars)
                                            </p>
                                        </div>

                                        <div>
                                            <Label htmlFor="start_date">
                                                Override Start Date (Optional)
                                            </Label>
                                            <DatePicker
                                                id="start_date"
                                                className="mt-1"
                                                date={data.start_date}
                                                onSelect={(date) =>
                                                    setData("start_date", date)
                                                }
                                                placeholder="Select start date..."
                                            />
                                            {errors.start_date && (
                                                <p className="mt-1 text-sm text-red-600">
                                                    {errors.start_date}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                If not set, coupon will use the
                                                season's default dates
                                            </p>
                                        </div>

                                        <div>
                                            <Label htmlFor="expiry_date">
                                                Override Expiry Date (Optional)
                                            </Label>
                                            <DatePicker
                                                id="expiry_date"
                                                className="mt-1"
                                                date={data.expiry_date}
                                                onSelect={(date) =>
                                                    setData("expiry_date", date)
                                                }
                                                placeholder="Select expiry date..."
                                            />
                                            {errors.expiry_date && (
                                                <p className="mt-1 text-sm text-red-600">
                                                    {errors.expiry_date}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                If not set, coupon will use the
                                                season's default end date
                                            </p>
                                        </div>

                                        <div>
                                            <Label htmlFor="max_uses">
                                                Maximum Uses
                                            </Label>
                                            <Input
                                                id="max_uses"
                                                type="number"
                                                className="mt-1"
                                                min="1"
                                                value={data.max_uses}
                                                onChange={(e) =>
                                                    setData(
                                                        "max_uses",
                                                        e.target.value
                                                    )
                                                }
                                                placeholder="Unlimited"
                                            />
                                            {errors.max_uses && (
                                                <p className="mt-1 text-sm text-red-600">
                                                    {errors.max_uses}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Leave empty for unlimited uses
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                                <CardFooter className="flex justify-end">
                                    <Button
                                        type="submit"
                                        disabled={
                                            processing ||
                                            selectedSeasons.length === 0
                                        }
                                    >
                                        {processing
                                            ? "Generating..."
                                            : "Generate Coupons"}
                                    </Button>
                                </CardFooter>
                            </Card>
                        </div>

                        {/* Warning if no seasons selected */}
                        {selectedSeasons.length === 0 && (
                            <Alert
                                variant="warning"
                                title="No Seasons Selected"
                                className="mt-6"
                                icon={AlertTriangle}
                            >
                                <p className="text-sm">
                                    Please select at least one season to
                                    generate coupons for.
                                </p>
                            </Alert>
                        )}

                        {/* Info alert */}
                        <Alert
                            variant="info"
                            title="How Seasonal Coupons Work"
                            className="mt-6"
                        >
                            <div className="text-sm">
                                <p className="mb-2">
                                    Seasonal coupons help drive sales during
                                    specific time periods:
                                </p>
                                <ul className="list-disc pl-5 space-y-1">
                                    <li>
                                        Coupons will be automatically available
                                        during their respective seasonal periods
                                    </li>
                                    <li>
                                        Each coupon can have custom discount
                                        amounts based on the seasonal event
                                    </li>
                                    <li>
                                        You can override the default seasonal
                                        dates if needed for specific promotions
                                    </li>
                                    <li>
                                        Consider higher discounts for key
                                        shopping seasons like Black Friday or
                                        Back to School
                                    </li>
                                </ul>
                            </div>
                        </Alert>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
};

export default GenerateSeasonal;
