import React, { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Search,
    Filter,
    Plus,
    Calendar,
    ChevronDown,
    ChevronUp,
    MoreHorizontal,
    Edit,
    Trash2,
    Eye,
    Tag,
    Percent,
    DollarSign,
    ToggleLeft,
    ToggleRight,
    Download,
    RefreshCw,
    BarChart,
} from "lucide-react";
import { Transition } from "@headlessui/react";
import Pagination from "@/Components/Pagination";
import Dropdown, { DropdownLink } from "@/Components/Dropdown";
import { DropdownMenu } from "@/Components/DropdownMenu";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import { toast } from "react-hot-toast";

const Index = ({ coupons, filters, couponTypes }) => {
    const [showFilters, setShowFilters] = useState(false);
    const [selectedCoupons, setSelectedCoupons] = useState([]);
    const [sortConfig, setSortConfig] = useState({
        key: filters.sort || "created_at",
        direction: filters.direction || "desc",
    });

    // Form for filters
    const { data, setData, get, processing, errors } = useForm({
        search: filters.search || "",
        type: filters.type || "",
        status: filters.status || "",
        date_range: filters.date_range || "",
    });

    // Helper function to get the correct route based on coupon type
    const getCouponRoute = (action, coupon) => {
        // If it's a default coupon, use default-coupons routes
        if (coupon.coupon_type === 'default') {
            return route(`admin.default-coupons.${action}`, coupon.id);
        }
        // Otherwise use regular coupons routes
        return route(`admin.coupons.${action}`, coupon.id);
    };

    // Handle filter submission
    const handleFilterSubmit = (e) => {
        e.preventDefault();
        get(route("admin.coupons.index"), {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Reset filters
    const resetFilters = () => {
        DropdownLink;
        setData({
            search: "",
            type: "",
            status: "",
            date_range: "",
        });
        get(route("admin.coupons.index"), {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Handle sorting
    const requestSort = (key) => {
        const direction =
            sortConfig.key === key && sortConfig.direction === "asc"
                ? "desc"
                : "asc";
        setSortConfig({ key, direction });

        get(
            route("admin.coupons.index", {
                ...data,
                sort: key,
                direction: direction,
            }),
            {
                preserveState: true,
                preserveScroll: true,
            }
        );
    };

    // Format date for display
    const formatDate = (dateString) => {
        if (!dateString) return "N/A";
        return new Date(dateString).toLocaleDateString();
    };

    // Toggle coupon selection
    const toggleCouponSelection = (couponId) => {
        setSelectedCoupons((prev) =>
            prev.includes(couponId)
                ? prev.filter((id) => id !== couponId)
                : [...prev, couponId]
        );
    };

    // Toggle all coupons
    const toggleAllCoupons = () => {
        if (selectedCoupons.length === coupons.data.length) {
            setSelectedCoupons([]);
        } else {
            setSelectedCoupons(coupons.data.map((coupon) => coupon.id));
        }
    };

    // Bulk actions
    const handleBulkAction = (action) => {
        if (selectedCoupons.length === 0) {
            toast.error("Please select at least one coupon");
            return;
        }

        if (action === "activate") {
            if (confirm("Activate all selected coupons?")) {
                // Submit the bulk action form
                const form = document.getElementById("bulk-toggle-form");
                document.getElementById("bulk-status").value = "1";
                form.submit();
            }
        } else if (action === "deactivate") {
            if (confirm("Deactivate all selected coupons?")) {
                // Submit the bulk action form
                const form = document.getElementById("bulk-toggle-form");
                document.getElementById("bulk-status").value = "0";
                form.submit();
            }
        } else if (action === "delete") {
            if (
                confirm(
                    "Delete all selected coupons? This action cannot be undone."
                )
            ) {
                // Submit the bulk delete form
                document.getElementById("bulk-delete-form").submit();
            }
        }
    };

    // Get coupon status badge
    const getCouponStatusBadge = (coupon) => {
        if (!coupon.is_active) {
            return (
                <Badge
                    variant="secondary"
                    className="bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200"
                >
                    Inactive
                </Badge>
            );
        }

        if (coupon.expiry_date && new Date(coupon.expiry_date) < new Date()) {
            return (
                <Badge
                    variant="warning"
                    className="bg-amber-100 dark:bg-amber-900/30 text-amber-800 dark:text-amber-300"
                >
                    Expired
                </Badge>
            );
        }

        if (coupon.max_uses && coupon.uses_count >= coupon.max_uses) {
            return (
                <Badge
                    variant="warning"
                    className="bg-orange-100 dark:bg-orange-900/30 text-orange-800 dark:text-orange-300"
                >
                    Limit Reached
                </Badge>
            );
        }

        return (
            <Badge
                variant="success"
                className="bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-300"
            >
                Active
            </Badge>
        );
    };

    // Get coupon type badge
    const getCouponTypeBadge = (type) => {
        const typeColors = {
            first_order: "blue",
            flash: "purple",
            referral: "indigo",
            loyalty: "emerald",
            anniversary: "pink",
            seasonal: "amber",
            volume: "cyan",
        };

        const color = typeColors[type] || "gray";

        return (
            <Badge
                className={`bg-${color}-100 dark:bg-${color}-900/30 text-${color}-800 dark:text-${color}-300`}
            >
                {couponTypes[type] || type}
            </Badge>
        );
    };

    // Get discount display
    const getDiscountDisplay = (coupon) => {
        if (coupon.discount_type === "percentage") {
            return (
                <div className="flex items-center">
                    <Percent className="w-4 h-4 mr-1 text-green-600 dark:text-green-400" />
                    <span>{coupon.discount_amount}%</span>
                </div>
            );
        }
        return (
            <div className="flex items-center">
                <DollarSign className="w-4 h-4 mr-1 text-green-600 dark:text-green-400" />
                <span>${parseFloat(coupon.discount_amount).toFixed(2)}</span>
            </div>
        );
    };

    return (
        <AdminLayout>
            <Head title="Coupon Management" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-3 md:space-y-0 mb-6">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white">
                                Coupon Management
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create and manage discount coupons for your
                                customers
                            </p>
                        </div>
                        <div className="flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-2">
                            <Link href={route("admin.coupons.statistics")}>
                                <Button
                                    type="button"
                                    variant="outline"
                                    className="w-full sm:w-auto"
                                >
                                    <BarChart className="w-4 h-4 mr-2" />
                                    Statistics
                                </Button>
                            </Link>
                            <Dropdown>
                                <Dropdown.Trigger>
                                    <Button
                                        type="button"
                                        variant="outline"
                                        className="w-full sm:w-auto"
                                    >
                                        <Tag className="w-4 h-4 mr-2" />
                                        Generate Coupon
                                        <ChevronDown className="w-4 h-4 ml-2" />
                                    </Button>
                                </Dropdown.Trigger>
                                <Dropdown.Content align="right" width="48">
                                    <Dropdown.Link
                                        href={route(
                                            "admin.coupons.generate.flash.form"
                                        )}
                                    >
                                        Flash Sale
                                    </Dropdown.Link>
                                    <Dropdown.Link
                                        href={route(
                                            "admin.coupons.generate.first-order.form"
                                        )}
                                    >
                                        First Order
                                    </Dropdown.Link>
                                    <Dropdown.Link
                                        href={route(
                                            "admin.coupons.generate.loyalty.form"
                                        )}
                                    >
                                        Loyalty Rewards
                                    </Dropdown.Link>
                                    <Dropdown.Link
                                        href={route(
                                            "admin.coupons.generate.seasonal.form"
                                        )}
                                    >
                                        Seasonal
                                    </Dropdown.Link>
                                    <Dropdown.Link
                                        href={route(
                                            "admin.coupons.generate.volume.form"
                                        )}
                                    >
                                        Volume Discount
                                    </Dropdown.Link>
                                    <Dropdown.Link
                                        href={route(
                                            "admin.coupons.generate.referral.form"
                                        )}
                                    >
                                        Referral
                                    </Dropdown.Link>
                                    <Dropdown.Link
                                        href={route(
                                            "admin.coupons.generate.anniversary.form"
                                        )}
                                    >
                                        Anniversary
                                    </Dropdown.Link>
                                </Dropdown.Content>
                            </Dropdown>
                            <Link href={route("admin.coupons.create")}>
                                <Button className="w-full sm:w-auto">
                                    <Plus className="w-4 h-4 mr-2" />
                                    New Coupon
                                </Button>
                            </Link>
                        </div>
                    </div>

                    {/* Search and Filters */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg shadow mb-6">
                        <div className="p-4 sm:p-6 border-b border-gray-200 dark:border-gray-700">
                            <div className="flex flex-col sm:flex-row sm:items-center space-y-3 sm:space-y-0 justify-between">
                                <div className="w-full sm:max-w-xs flex-grow sm:flex-grow-0">
                                    <label htmlFor="search" className="sr-only">
                                        Search coupons
                                    </label>
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <Search
                                                className="h-5 w-5 text-gray-400"
                                                aria-hidden="true"
                                            />
                                        </div>
                                        <input
                                            type="text"
                                            name="search"
                                            id="search"
                                            className="block w-full pl-10 pr-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md leading-5 bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm transition duration-150 ease-in-out"
                                            placeholder="Search coupons"
                                            value={data.search}
                                            onChange={(e) =>
                                                setData(
                                                    "search",
                                                    e.target.value
                                                )
                                            }
                                            onKeyUp={(e) => {
                                                if (e.key === "Enter") {
                                                    handleFilterSubmit(e);
                                                }
                                            }}
                                        />
                                    </div>
                                </div>
                                <div className="flex items-center space-x-2">
                                    <Button
                                        type="button"
                                        variant="outline"
                                        onClick={() =>
                                            setShowFilters(!showFilters)
                                        }
                                        className="inline-flex items-center"
                                    >
                                        <Filter className="h-4 w-4 mr-2" />
                                        {showFilters
                                            ? "Hide Filters"
                                            : "Show Filters"}
                                    </Button>
                                </div>
                            </div>

                            {/* Expandable Filters */}
                            <Transition
                                show={showFilters}
                                enter="transition ease-out duration-200"
                                enterFrom="transform opacity-0 scale-95"
                                enterTo="transform opacity-100 scale-100"
                                leave="transition ease-in duration-150"
                                leaveFrom="transform opacity-100 scale-100"
                                leaveTo="transform opacity-0 scale-95"
                            >
                                <form
                                    onSubmit={handleFilterSubmit}
                                    className="mt-4"
                                >
                                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                                        <div>
                                            <label
                                                htmlFor="coupon-type"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Coupon Type
                                            </label>
                                            <select
                                                id="coupon-type"
                                                name="type"
                                                className="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                                value={data.type}
                                                onChange={(e) =>
                                                    setData(
                                                        "type",
                                                        e.target.value
                                                    )
                                                }
                                            >
                                                <option value="">
                                                    All Types
                                                </option>
                                                {Object.entries(
                                                    couponTypes
                                                ).map(([value, label]) => (
                                                    <option
                                                        key={value}
                                                        value={value}
                                                    >
                                                        {label}
                                                    </option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <label
                                                htmlFor="status"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Status
                                            </label>
                                            <select
                                                id="status"
                                                name="status"
                                                className="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                                value={data.status}
                                                onChange={(e) =>
                                                    setData(
                                                        "status",
                                                        e.target.value
                                                    )
                                                }
                                            >
                                                <option value="">
                                                    All Statuses
                                                </option>
                                                <option value="active">
                                                    Active
                                                </option>
                                                <option value="inactive">
                                                    Inactive
                                                </option>
                                                <option value="expired">
                                                    Expired
                                                </option>
                                                <option value="limit-reached">
                                                    Limit Reached
                                                </option>
                                            </select>
                                        </div>

                                        <div>
                                            <label
                                                htmlFor="date-range"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Date Range
                                            </label>
                                            <select
                                                id="date-range"
                                                name="date_range"
                                                className="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                                value={data.date_range}
                                                onChange={(e) =>
                                                    setData(
                                                        "date_range",
                                                        e.target.value
                                                    )
                                                }
                                            >
                                                <option value="">
                                                    All Time
                                                </option>
                                                <option value="today">
                                                    Today
                                                </option>
                                                <option value="this_week">
                                                    This Week
                                                </option>
                                                <option value="this_month">
                                                    This Month
                                                </option>
                                                <option value="last_30_days">
                                                    Last 30 Days
                                                </option>
                                                <option value="last_90_days">
                                                    Last 90 Days
                                                </option>
                                            </select>
                                        </div>

                                        <div className="flex items-end space-x-2">
                                            <Button
                                                type="submit"
                                                disabled={processing}
                                                className="px-4 py-2"
                                            >
                                                {processing ? (
                                                    <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                                ) : (
                                                    <Search className="w-4 h-4 mr-2" />
                                                )}
                                                Filter
                                            </Button>
                                            <Button
                                                type="button"
                                                variant="outline"
                                                onClick={resetFilters}
                                                className="px-4 py-2"
                                            >
                                                Reset
                                            </Button>
                                        </div>
                                    </div>
                                </form>
                            </Transition>
                        </div>

                        {/* Bulk Actions */}
                        {selectedCoupons.length > 0 && (
                            <div className="bg-blue-50 dark:bg-blue-900/20 border-t border-b border-blue-100 dark:border-blue-800 p-4">
                                <div className="flex items-center justify-between">
                                    <span className="text-sm font-medium text-blue-800 dark:text-blue-200">
                                        {selectedCoupons.length} coupons
                                        selected
                                    </span>
                                    <div className="flex space-x-2">
                                        <Button
                                            type="button"
                                            variant="outline"
                                            onClick={() =>
                                                handleBulkAction("activate")
                                            }
                                            className="text-xs px-2 py-1"
                                        >
                                            <ToggleRight className="w-4 h-4 mr-1" />
                                            Activate
                                        </Button>
                                        <Button
                                            type="button"
                                            variant="outline"
                                            onClick={() =>
                                                handleBulkAction("deactivate")
                                            }
                                            className="text-xs px-2 py-1"
                                        >
                                            <ToggleLeft className="w-4 h-4 mr-1" />
                                            Deactivate
                                        </Button>
                                        <Button
                                            type="button"
                                            variant="destructive"
                                            onClick={() =>
                                                handleBulkAction("delete")
                                            }
                                            className="text-xs px-2 py-1"
                                        >
                                            <Trash2 className="w-4 h-4 mr-1" />
                                            Delete
                                        </Button>
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Hidden Bulk Action Forms */}
                        <form
                            id="bulk-toggle-form"
                            method="POST"
                            action={route("admin.coupons.bulk-toggle-status")}
                        >
                            <input type="hidden" name="_method" value="PUT" />
                            <input
                                type="hidden"
                                name="_token"
                                value={window.csrfToken}
                            />
                            <input
                                type="hidden"
                                name="coupon_ids"
                                value={JSON.stringify(selectedCoupons)}
                            />
                            <input
                                type="hidden"
                                id="bulk-status"
                                name="status"
                                value="1"
                            />
                        </form>

                        <form
                            id="bulk-delete-form"
                            method="POST"
                            action={route("admin.coupons.bulk-delete")}
                        >
                            <input
                                type="hidden"
                                name="_method"
                                value="DELETE"
                            />
                            <input
                                type="hidden"
                                name="_token"
                                value={window.csrfToken}
                            />
                            <input
                                type="hidden"
                                name="coupon_ids"
                                value={JSON.stringify(selectedCoupons)}
                            />
                        </form>

                        {/* Coupons Table (CSS Grid) */}
                        <div className="overflow-x-auto">
                            {coupons.data.length > 0 ? (
                                <>
                                    {/* Table Header - Desktop */}
                                    <div className="hidden md:grid md:grid-cols-7 gap-4 px-6 py-3 border-b border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-800/80">
                                        <div className="flex items-center">
                                            <input
                                                type="checkbox"
                                                className="h-4 w-4 text-blue-600 rounded border-gray-300 dark:border-gray-600 dark:bg-gray-700 focus:ring-blue-500"
                                                checked={
                                                    selectedCoupons.length ===
                                                        coupons.data.length &&
                                                    coupons.data.length > 0
                                                }
                                                onChange={toggleAllCoupons}
                                            />
                                            <button
                                                className="ml-3 text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider flex items-center"
                                                onClick={() =>
                                                    requestSort("coupon_code")
                                                }
                                            >
                                                Code
                                                {sortConfig.key ===
                                                    "coupon_code" && (
                                                    <>
                                                        {sortConfig.direction ===
                                                        "asc" ? (
                                                            <ChevronUp className="h-4 w-4 ml-1" />
                                                        ) : (
                                                            <ChevronDown className="h-4 w-4 ml-1" />
                                                        )}
                                                    </>
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider flex items-center">
                                            <button
                                                className="flex items-center"
                                                onClick={() =>
                                                    requestSort(
                                                        "discount_amount"
                                                    )
                                                }
                                            >
                                                Discount
                                                {sortConfig.key ===
                                                    "discount_amount" && (
                                                    <>
                                                        {sortConfig.direction ===
                                                        "asc" ? (
                                                            <ChevronUp className="h-4 w-4 ml-1" />
                                                        ) : (
                                                            <ChevronDown className="h-4 w-4 ml-1" />
                                                        )}
                                                    </>
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider flex items-center">
                                            <button
                                                className="flex items-center"
                                                onClick={() =>
                                                    requestSort("coupon_type")
                                                }
                                            >
                                                Type
                                                {sortConfig.key ===
                                                    "coupon_type" && (
                                                    <>
                                                        {sortConfig.direction ===
                                                        "asc" ? (
                                                            <ChevronUp className="h-4 w-4 ml-1" />
                                                        ) : (
                                                            <ChevronDown className="h-4 w-4 ml-1" />
                                                        )}
                                                    </>
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider flex items-center">
                                            <button
                                                className="flex items-center"
                                                onClick={() =>
                                                    requestSort("uses_count")
                                                }
                                            >
                                                Usage
                                                {sortConfig.key ===
                                                    "uses_count" && (
                                                    <>
                                                        {sortConfig.direction ===
                                                        "asc" ? (
                                                            <ChevronUp className="h-4 w-4 ml-1" />
                                                        ) : (
                                                            <ChevronDown className="h-4 w-4 ml-1" />
                                                        )}
                                                    </>
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider flex items-center">
                                            <button
                                                className="flex items-center"
                                                onClick={() =>
                                                    requestSort("expiry_date")
                                                }
                                            >
                                                Expiry
                                                {sortConfig.key ===
                                                    "expiry_date" && (
                                                    <>
                                                        {sortConfig.direction ===
                                                        "asc" ? (
                                                            <ChevronUp className="h-4 w-4 ml-1" />
                                                        ) : (
                                                            <ChevronDown className="h-4 w-4 ml-1" />
                                                        )}
                                                    </>
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Status
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider text-right">
                                            Actions
                                        </div>
                                    </div>

                                    {/* Table Rows */}
                                    <div className="divide-y divide-gray-200 dark:divide-gray-700">
                                        {coupons.data.map((coupon) => (
                                            <React.Fragment key={coupon.id}>
                                                {/* Desktop Row */}
                                                <div className="hidden md:grid md:grid-cols-7 gap-4 px-6 py-4 hover:bg-gray-50 dark:hover:bg-gray-800/60">
                                                    <div className="flex items-center">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 text-blue-600 rounded border-gray-300 dark:border-gray-600 dark:bg-gray-700 focus:ring-blue-500"
                                                            checked={selectedCoupons.includes(
                                                                coupon.id
                                                            )}
                                                            onChange={() =>
                                                                toggleCouponSelection(
                                                                    coupon.id
                                                                )
                                                            }
                                                        />
                                                        <span className="ml-3 text-sm font-mono font-medium text-gray-900 dark:text-gray-100">
                                                            {coupon.coupon_code}
                                                        </span>
                                                    </div>
                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                        {getDiscountDisplay(
                                                            coupon
                                                        )}
                                                    </div>
                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                        {getCouponTypeBadge(
                                                            coupon.coupon_type
                                                        )}
                                                    </div>
                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                        {coupon.uses_count || 0}
                                                        {coupon.max_uses
                                                            ? ` / ${coupon.max_uses}`
                                                            : ""}
                                                    </div>
                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                        {coupon.expiry_date
                                                            ? formatDate(
                                                                  coupon.expiry_date
                                                              )
                                                            : "No Expiry"}
                                                    </div>
                                                    <div>
                                                        {getCouponStatusBadge(
                                                            coupon
                                                        )}
                                                    </div>
                                                    <div className="text-right">
                                                        <div className="flex items-center space-x-2 justify-end">
                                                            <Link
                                                                href={getCouponRoute("show", coupon)}
                                                                className="p-1.5 text-blue-600 hover:bg-blue-50 rounded-md"
                                                                title="View Details"
                                                            >
                                                                <Eye className="w-4 h-4" />
                                                            </Link>

                                                            <Link
                                                                href={getCouponRoute("edit", coupon)}
                                                                className="p-1.5 text-gray-600 hover:bg-gray-50 rounded-md"
                                                                title="Edit"
                                                            >
                                                                <Edit className="w-4 h-4" />
                                                            </Link>

                                                            <Link
                                                                href={route(
                                                                    "admin.coupons.toggle-status",
                                                                    coupon.id
                                                                )}
                                                                method="put"
                                                                as="button"
                                                                className="p-1.5 text-amber-600 hover:bg-amber-50 rounded-md"
                                                                title={
                                                                    coupon.is_active
                                                                        ? "Deactivate"
                                                                        : "Activate"
                                                                }
                                                            >
                                                                {coupon.is_active ? (
                                                                    <ToggleLeft className="w-4 h-4" />
                                                                ) : (
                                                                    <ToggleRight className="w-4 h-4" />
                                                                )}
                                                            </Link>

                                                            <Link
                                                                href={route(
                                                                    "admin.coupons.destroy",
                                                                    coupon.id
                                                                )}
                                                                method="delete"
                                                                as="button"
                                                                className="p-1.5 text-red-600 hover:bg-red-50 rounded-md"
                                                                title="Delete"
                                                                onClick={(
                                                                    e
                                                                ) => {
                                                                    if (
                                                                        !confirm(
                                                                            "Are you sure you want to delete this coupon?"
                                                                        )
                                                                    ) {
                                                                        e.preventDefault();
                                                                    }
                                                                }}
                                                            >
                                                                <Trash2 className="w-4 h-4" />
                                                            </Link>
                                                        </div>
                                                    </div>
                                                </div>

                                                {/* Mobile Card View */}
                                                <div className="block md:hidden p-4 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                                                    <div className="flex items-center justify-between mb-2">
                                                        <div className="flex items-center">
                                                            <input
                                                                type="checkbox"
                                                                className="h-4 w-4 text-blue-600 rounded border-gray-300 dark:border-gray-600 dark:bg-gray-700 focus:ring-blue-500"
                                                                checked={selectedCoupons.includes(
                                                                    coupon.id
                                                                )}
                                                                onChange={() =>
                                                                    toggleCouponSelection(
                                                                        coupon.id
                                                                    )
                                                                }
                                                            />
                                                            <span className="ml-3 text-sm font-mono font-medium text-gray-900 dark:text-gray-100">
                                                                {
                                                                    coupon.coupon_code
                                                                }
                                                            </span>
                                                        </div>
                                                        {getCouponStatusBadge(
                                                            coupon
                                                        )}
                                                    </div>

                                                    <div className="grid grid-cols-2 gap-2 mb-3">
                                                        <div>
                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                Discount
                                                            </p>
                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                {getDiscountDisplay(
                                                                    coupon
                                                                )}
                                                            </div>
                                                        </div>
                                                        <div>
                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                Type
                                                            </div>
                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                {getCouponTypeBadge(
                                                                    coupon.coupon_type
                                                                )}
                                                            </div>
                                                        </div>
                                                        <div>
                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                Usage
                                                            </p>
                                                            <p className="text-sm text-gray-900 dark:text-gray-100">
                                                                {coupon.uses_count ||
                                                                    0}
                                                                {coupon.max_uses
                                                                    ? ` / ${coupon.max_uses}`
                                                                    : ""}
                                                            </p>
                                                        </div>
                                                        <div>
                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                Expiry
                                                            </p>
                                                            <p className="text-sm text-gray-900 dark:text-gray-100">
                                                                {coupon.expiry_date
                                                                    ? formatDate(
                                                                          coupon.expiry_date
                                                                      )
                                                                    : "No Expiry"}
                                                            </p>
                                                        </div>
                                                    </div>

                                                    <div className="flex justify-end space-x-2 border-t border-gray-100 dark:border-gray-700 pt-2">
                                                        <Link
                                                            href={getCouponRoute("show", coupon)}
                                                            className="inline-flex items-center justify-center p-1.5 text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md"
                                                        >
                                                            <Eye className="w-4 h-4" />
                                                        </Link>
                                                        <Link
                                                            href={getCouponRoute("edit", coupon)}
                                                            className="inline-flex items-center justify-center p-1.5 text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md"
                                                        >
                                                            <Edit className="w-4 h-4" />
                                                        </Link>
                                                        <Link
                                                            href={route(
                                                                "admin.coupons.toggle-status",
                                                                coupon.id
                                                            )}
                                                            method="put"
                                                            as="button"
                                                            className="inline-flex items-center justify-center p-1.5 text-amber-600 dark:text-amber-400 hover:text-amber-800 dark:hover:text-amber-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md"
                                                        >
                                                            {coupon.is_active ? (
                                                                <ToggleLeft className="w-4 h-4" />
                                                            ) : (
                                                                <ToggleRight className="w-4 h-4" />
                                                            )}
                                                        </Link>
                                                        <Link
                                                            href={route(
                                                                "admin.coupons.destroy",
                                                                coupon.id
                                                            )}
                                                            method="delete"
                                                            as="button"
                                                            className="text-xs text-red-600 dark:text-red-400 hover:text-red-800 dark:hover:text-red-300"
                                                            onClick={(e) => {
                                                                if (
                                                                    !confirm(
                                                                        "Are you sure you want to delete this coupon?"
                                                                    )
                                                                ) {
                                                                    e.preventDefault();
                                                                }
                                                            }}
                                                        >
                                                            <Trash2 className="w-4 h-4" />
                                                        </Link>
                                                    </div>
                                                </div>
                                            </React.Fragment>
                                        ))}
                                    </div>
                                </>
                            ) : (
                                // Empty State
                                <div className="text-center py-12">
                                    <Tag className="mx-auto h-12 w-12 text-gray-400 dark:text-gray-500" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-gray-100">
                                        No coupons found
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        {data.search ||
                                        data.type ||
                                        data.status ||
                                        data.date_range
                                            ? "Try adjusting your search or filter criteria"
                                            : "Get started by creating a new coupon"}
                                    </p>
                                    <div className="mt-6">
                                        <Link
                                            href={route("admin.coupons.create")}
                                        >
                                            <Button type="button">
                                                <Plus className="h-4 w-4 mr-2" />
                                                New Coupon
                                            </Button>
                                        </Link>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Pagination */}
                    {coupons.data.length > 0 && (
                        <div className="mt-4 flex justify-between items-center">
                            <div className="text-sm text-gray-700 dark:text-gray-300">
                                Showing{" "}
                                <span className="font-medium">
                                    {coupons.from}
                                </span>{" "}
                                to{" "}
                                <span className="font-medium">
                                    {coupons.to}
                                </span>{" "}
                                of{" "}
                                <span className="font-medium">
                                    {coupons.total}
                                </span>{" "}
                                coupons
                            </div>
                            <Pagination links={coupons.links} />
                        </div>
                    )}

                    {/* Export Options */}
                    <div className="mt-8 border-t border-gray-200 dark:border-gray-700 pt-4">
                        <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Export Options
                        </h3>
                        <div className="flex space-x-2">
                            <Button variant="outline" size="sm">
                                <Download className="w-4 h-4 mr-2" />
                                Export CSV
                            </Button>
                            <Button variant="outline" size="sm">
                                <Download className="w-4 h-4 mr-2" />
                                Export PDF
                            </Button>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Index;
