import React, { useState } from "react";
import { Head, Link, router } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    ArrowLeft,
    Edit,
    Trash2,
    Tag,
    Calendar,
    Clock,
    User,
    DollarSign,
    Percent,
    Users,
    ShoppingBag,
    Award,
    Gift,
    Zap,
    Package,
    CalendarDays,
    CheckCircle,
    XCircle,
    AlertTriangle,
    ToggleLeft,
    ToggleRight,
    FileText,
    BarChart2,
    Mail,
    ExternalLink,
} from "lucide-react";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import { formatDate, formatCurrency } from "@/lib/utils";
import { confirmDialog } from "@/lib/confirmDialog";
import {
    Table,
    TableHeader,
    TableBody,
    TableRow,
    TableHead,
    TableCell,
} from "@/Components/ui/Table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/Components/ui/Tabs";
import UsageChart from "@/Components/Charts/UsageChart";
import Pagination from "@/Components/Pagination";

const Show = ({
    coupon,
    usageStats,
    userUsages,
    orders,
    couponTypes,
    discountTypes,
}) => {
    const [activeTab, setActiveTab] = useState("overview");

    // Format date for display
    const formatDateDisplay = (dateString) => {
        if (!dateString) return "No date set";
        return new Date(dateString).toLocaleString();
    };

    // Handle coupon deletion
    const handleDelete = () => {
        confirmDialog({
            title: "Delete Coupon",
            message: `Are you sure you want to delete the coupon "${coupon.coupon_code}"? This action cannot be undone.`,
            onConfirm: () => {
                router.delete(route("admin.coupons.destroy", coupon.id), {
                    preserveScroll: true,
                });
            },
        });
    };

    // Handle toggle coupon status
    const handleToggleStatus = () => {
        router.put(
            route("admin.coupons.toggle-status", coupon.id),
            { is_active: !coupon.is_active },
            { preserveScroll: true }
        );
    };

    // Get status badge
    const getStatusBadge = () => {
        const now = new Date();
        const startDate = coupon.start_date
            ? new Date(coupon.start_date)
            : null;
        const expiryDate = coupon.expiry_date
            ? new Date(coupon.expiry_date)
            : null;

        if (!coupon.is_active) {
            return (
                <Badge
                    variant="outline"
                    className="bg-gray-100 text-gray-800 border-gray-200"
                >
                    <XCircle className="w-3.5 h-3.5 mr-1" />
                    Inactive
                </Badge>
            );
        }

        if (startDate && now < startDate) {
            return (
                <Badge
                    variant="outline"
                    className="bg-yellow-100 text-yellow-800 border-yellow-200"
                >
                    <Clock className="w-3.5 h-3.5 mr-1" />
                    Upcoming
                </Badge>
            );
        }

        if (expiryDate && now > expiryDate) {
            return (
                <Badge
                    variant="outline"
                    className="bg-red-100 text-red-800 border-red-200"
                >
                    <AlertTriangle className="w-3.5 h-3.5 mr-1" />
                    Expired
                </Badge>
            );
        }

        if (coupon.max_uses && coupon.uses_count >= coupon.max_uses) {
            return (
                <Badge
                    variant="outline"
                    className="bg-purple-100 text-purple-800 border-purple-200"
                >
                    <AlertTriangle className="w-3.5 h-3.5 mr-1" />
                    Usage Limit Reached
                </Badge>
            );
        }

        return (
            <Badge
                variant="outline"
                className="bg-green-100 text-green-800 border-green-200"
            >
                <CheckCircle className="w-3.5 h-3.5 mr-1" />
                Active
            </Badge>
        );
    };

    // Get coupon type icon
    const getCouponTypeIcon = (type) => {
        const icons = {
            flash: <Zap className="w-5 h-5" />,
            first_order: <ShoppingBag className="w-5 h-5" />,
            referral: <Users className="w-5 h-5" />,
            loyalty: <Award className="w-5 h-5" />,
            anniversary: <Gift className="w-5 h-5" />,
            seasonal: <CalendarDays className="w-5 h-5" />,
            volume: <Package className="w-5 h-5" />,
        };
        return icons[type] || <Tag className="w-5 h-5" />;
    };

    // Get badge color for coupon type
    const getCouponTypeBadgeColor = (type) => {
        const colors = {
            flash: "bg-amber-100 text-amber-800 border-amber-200",
            first_order: "bg-green-100 text-green-800 border-green-200",
            referral: "bg-purple-100 text-purple-800 border-purple-200",
            loyalty: "bg-blue-100 text-blue-800 border-blue-200",
            anniversary: "bg-pink-100 text-pink-800 border-pink-200",
            seasonal: "bg-red-100 text-red-800 border-red-200",
            volume: "bg-indigo-100 text-indigo-800 border-indigo-200",
        };
        return colors[type] || "bg-gray-100 text-gray-800 border-gray-200";
    };

    return (
        <AdminLayout>
            <Head title={`Coupon: ${coupon.coupon_code}`} />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-3 md:space-y-0 mb-6">
                        <div>
                            <div className="flex items-center">
                                <h2 className="text-2xl font-semibold text-gray-900 dark:text-white flex items-center">
                                    <Tag className="h-6 w-6 mr-2 text-blue-500" />
                                    {coupon.coupon_code}
                                </h2>
                                {getStatusBadge()}
                            </div>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Created on{" "}
                                {formatDateDisplay(coupon.created_at)}
                            </p>
                        </div>
                        <div className="flex flex-wrap gap-3">
                            <Button
                                onClick={handleToggleStatus}
                                variant="outline"
                                className="inline-flex items-center space-x-2"
                            >
                                {coupon.is_active ? (
                                    <>
                                        <ToggleRight className="w-4 h-4" />
                                        <span>Deactivate</span>
                                    </>
                                ) : (
                                    <>
                                        <ToggleLeft className="w-4 h-4" />
                                        <span>Activate</span>
                                    </>
                                )}
                            </Button>
                            <Link
                                href={route("admin.coupons.edit", coupon.id)}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-blue-700 focus:bg-blue-700 active:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition"
                            >
                                <Edit className="w-4 h-4 mr-2" />
                                Edit
                            </Link>
                            <Button
                                onClick={handleDelete}
                                variant="destructive"
                                className="inline-flex items-center"
                            >
                                <Trash2 className="w-4 h-4 mr-2" />
                                Delete
                            </Button>
                            <Link
                                href={route("admin.coupons.index")}
                                className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md font-semibold text-xs text-gray-700 dark:text-gray-300 uppercase tracking-widest shadow-sm bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 dark:focus:ring-offset-gray-800 transition"
                            >
                                <ArrowLeft className="w-4 h-4 mr-2" />
                                Back
                            </Link>
                        </div>
                    </div>

                    <Tabs
                        defaultValue="overview"
                        onValueChange={setActiveTab}
                        className="w-full"
                    >
                        <TabsList className="mb-6 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg">
                            <TabsTrigger value="overview">
                                <FileText className="w-4 h-4 mr-2" />
                                Overview
                            </TabsTrigger>
                            <TabsTrigger value="usage">
                                <BarChart2 className="w-4 h-4 mr-2" />
                                Usage Analytics
                            </TabsTrigger>
                            <TabsTrigger value="orders">
                                <ShoppingBag className="w-4 h-4 mr-2" />
                                Orders ({orders.data.length})
                            </TabsTrigger>
                            <TabsTrigger value="users">
                                <Users className="w-4 h-4 mr-2" />
                                Users ({userUsages.data.length})
                            </TabsTrigger>
                        </TabsList>

                        <TabsContent value="overview">
                            <div className="grid grid-cols-1 gap-6 lg:grid-cols-3">
                                {/* Coupon Details Card */}
                                <div className="col-span-2 bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                                    <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                        <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                            Coupon Details
                                        </h3>
                                    </div>
                                    <div className="p-6">
                                        <dl className="grid grid-cols-1 gap-x-6 gap-y-6 sm:grid-cols-3">
                                            <div className="sm:col-span-1">
                                                <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Coupon Type
                                                </dt>
                                                <dd className="mt-1 flex items-center">
                                                    <Badge
                                                        variant="outline"
                                                        className={`flex items-center ${getCouponTypeBadgeColor(
                                                            coupon.coupon_type
                                                        )}`}
                                                    >
                                                        {getCouponTypeIcon(
                                                            coupon.coupon_type
                                                        )}
                                                        <span className="ml-1 capitalize">
                                                            {couponTypes[
                                                                coupon
                                                                    .coupon_type
                                                            ] ||
                                                                coupon.coupon_type}
                                                        </span>
                                                    </Badge>
                                                </dd>
                                            </div>
                                            <div className="sm:col-span-1">
                                                <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Discount Type
                                                </dt>
                                                <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                    {coupon.discount_type ===
                                                    "percentage" ? (
                                                        <span className="flex items-center">
                                                            <Percent className="w-4 h-4 mr-1 text-gray-400" />
                                                            {
                                                                coupon.discount_amount
                                                            }
                                                            % off
                                                        </span>
                                                    ) : (
                                                        <span className="flex items-center">
                                                            <DollarSign className="w-4 h-4 mr-1 text-gray-400" />
                                                            $
                                                            {parseFloat(
                                                                coupon.discount_amount ||
                                                                    0
                                                            ).toFixed(2)}{" "}
                                                            off
                                                        </span>
                                                    )}
                                                </dd>
                                            </div>
                                            {coupon.coupon_type === "loyalty" &&
                                                coupon.loyalty_tier && (
                                                    <div className="sm:col-span-1">
                                                        <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                            Loyalty Tier
                                                        </dt>
                                                        <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                            <span className="flex items-center">
                                                                <Award className="w-4 h-4 mr-1 text-blue-500" />
                                                                {
                                                                    coupon
                                                                        .loyalty_tier
                                                                        .name
                                                                }{" "}
                                                                (
                                                                {
                                                                    coupon
                                                                        .loyalty_tier
                                                                        .min_orders
                                                                }
                                                                + orders)
                                                            </span>
                                                        </dd>
                                                    </div>
                                                )}
                                            {coupon.coupon_type ===
                                                "seasonal" &&
                                                coupon.seasonal_type && (
                                                    <div className="sm:col-span-1">
                                                        <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                            Seasonal Event
                                                        </dt>
                                                        <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                            <span className="flex items-center">
                                                                <CalendarDays className="w-4 h-4 mr-1 text-red-500" />
                                                                {
                                                                    coupon
                                                                        .seasonal_type
                                                                        .name
                                                                }
                                                            </span>
                                                        </dd>
                                                    </div>
                                                )}
                                            <div className="sm:col-span-1">
                                                <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Start Date
                                                </dt>
                                                <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                    {coupon.start_date
                                                        ? formatDateDisplay(
                                                              coupon.start_date
                                                          )
                                                        : "Immediately"}
                                                </dd>
                                            </div>
                                            <div className="sm:col-span-1">
                                                <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Expiry Date
                                                </dt>
                                                <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                    {coupon.expiry_date
                                                        ? formatDateDisplay(
                                                              coupon.expiry_date
                                                          )
                                                        : "No expiration"}
                                                </dd>
                                            </div>
                                            <div className="sm:col-span-1">
                                                <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Max Uses
                                                </dt>
                                                <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                    {coupon.max_uses
                                                        ? `${coupon.uses_count} / ${coupon.max_uses}`
                                                        : "Unlimited"}
                                                </dd>
                                            </div>
                                            <div className="sm:col-span-1">
                                                <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Uses Per User
                                                </dt>
                                                <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                    {coupon.per_user_limit
                                                        ? coupon.per_user_limit
                                                        : "Unlimited"}
                                                </dd>
                                            </div>
                                            {coupon.min_order_amount && (
                                                <div className="sm:col-span-1">
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Min Order Amount
                                                    </dt>
                                                    <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                        $
                                                        {parseFloat(
                                                            coupon.min_order_amount ||
                                                                0
                                                        ).toFixed(2)}
                                                    </dd>
                                                </div>
                                            )}
                                            {coupon.min_pages > 0 && (
                                                <div className="sm:col-span-1">
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Min Pages
                                                    </dt>
                                                    <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                        {coupon.min_pages}
                                                    </dd>
                                                </div>
                                            )}
                                            <div className="sm:col-span-3">
                                                <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Restrictions
                                                </dt>
                                                <dd className="mt-1 text-sm text-gray-900 dark:text-white">
                                                    <ul className="list-disc pl-5 space-y-1">
                                                        {coupon.is_first_order_only && (
                                                            <li>
                                                                Only valid for
                                                                first orders
                                                            </li>
                                                        )}
                                                        {coupon.is_referral && (
                                                            <li>
                                                                This is a
                                                                referral coupon
                                                            </li>
                                                        )}
                                                        {coupon.user_specific && (
                                                            <li>
                                                                User-specific
                                                                coupon (ID:{" "}
                                                                {
                                                                    coupon.referrer_user_id
                                                                }
                                                                )
                                                            </li>
                                                        )}
                                                        {coupon.min_order_amount && (
                                                            <li>
                                                                Minimum order
                                                                amount: $
                                                                {parseFloat(
                                                                    coupon.min_order_amount ||
                                                                        0
                                                                ).toFixed(2)}
                                                            </li>
                                                        )}
                                                        {coupon.min_pages >
                                                            0 && (
                                                            <li>
                                                                Minimum page
                                                                count:{" "}
                                                                {
                                                                    coupon.min_pages
                                                                }
                                                            </li>
                                                        )}
                                                        {!coupon.is_first_order_only &&
                                                            !coupon.is_referral &&
                                                            !coupon.user_specific &&
                                                            !coupon.min_order_amount &&
                                                            !(
                                                                coupon.min_pages >
                                                                0
                                                            ) && (
                                                                <li>
                                                                    No special
                                                                    restrictions
                                                                </li>
                                                            )}
                                                    </ul>
                                                </dd>
                                            </div>
                                        </dl>
                                    </div>
                                </div>

                                {/* Usage Statistics Card */}
                                <div className="col-span-1 bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                                    <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                        <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                            Usage Statistics
                                        </h3>
                                    </div>
                                    <div className="p-6 space-y-6">
                                        <div className="flex items-center justify-between">
                                            <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                Total Uses
                                            </span>
                                            <span className="text-xl font-semibold text-gray-900 dark:text-white">
                                                {coupon.uses_count || 0}
                                            </span>
                                        </div>

                                        {coupon.max_uses && (
                                            <div>
                                                <div className="flex items-center justify-between mb-1">
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Usage Limit
                                                    </span>
                                                    <span className="text-sm text-gray-700 dark:text-gray-300">
                                                        {coupon.uses_count} /{" "}
                                                        {coupon.max_uses}
                                                    </span>
                                                </div>
                                                <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2.5">
                                                    <div
                                                        className="bg-blue-600 h-2.5 rounded-full"
                                                        style={{
                                                            width: `${Math.min(
                                                                100,
                                                                (coupon.uses_count /
                                                                    coupon.max_uses) *
                                                                    100
                                                            )}%`,
                                                        }}
                                                    ></div>
                                                </div>
                                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                    {Math.max(
                                                        0,
                                                        coupon.max_uses -
                                                            coupon.uses_count
                                                    )}{" "}
                                                    uses remaining
                                                </p>
                                            </div>
                                        )}

                                        <div className="pt-4 border-t border-gray-200 dark:border-gray-700">
                                            <dl className="grid grid-cols-2 gap-4">
                                                <div>
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Total Revenue
                                                    </dt>
                                                    <dd className="mt-1 text-lg font-semibold text-gray-900 dark:text-white">
                                                        $
                                                        {usageStats.total_revenue.toFixed(
                                                            2
                                                        )}
                                                    </dd>
                                                </div>
                                                <div>
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Total Discount
                                                    </dt>
                                                    <dd className="mt-1 text-lg font-semibold text-green-600 dark:text-green-400">
                                                        $
                                                        {parseFloat(
                                                            usageStats.total_discount ||
                                                                0
                                                        ).toFixed(2)}
                                                    </dd>
                                                </div>
                                                <div>
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Unique Users
                                                    </dt>
                                                    <dd className="mt-1 text-lg font-semibold text-gray-900 dark:text-white">
                                                        {
                                                            usageStats.unique_users
                                                        }
                                                    </dd>
                                                </div>
                                                <div>
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Avg. Discount
                                                    </dt>
                                                    <dd className="mt-1 text-lg font-semibold text-gray-900 dark:text-white">
                                                        $
                                                        {parseFloat(
                                                            usageStats.average_discount ||
                                                                0
                                                        ).toFixed(2)}
                                                    </dd>
                                                </div>
                                            </dl>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </TabsContent>

                        <TabsContent value="usage">
                            <div className="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                                <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                        Usage Analytics
                                    </h3>
                                </div>
                                <div className="p-6">
                                    {usageStats.monthly_usage &&
                                    usageStats.monthly_usage.length > 0 ? (
                                        <div className="h-72">
                                            <UsageChart
                                                data={usageStats.monthly_usage}
                                            />
                                        </div>
                                    ) : (
                                        <div className="text-center py-12">
                                            <BarChart2 className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                                No usage data available
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                This coupon hasn't been used
                                                yet.
                                            </p>
                                        </div>
                                    )}
                                </div>
                            </div>
                        </TabsContent>

                        <TabsContent value="orders">
                            <div className="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                                <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                        Orders Using This Coupon
                                    </h3>
                                </div>
                                <div className="overflow-x-auto">
                                    {orders.data.length > 0 ? (
                                        <Table>
                                            <Table.Header>
                                                <Table.Row>
                                                    <Table.Head>
                                                        Order ID
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Date
                                                    </Table.Head>
                                                    <Table.Head>
                                                        User
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Order Total
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Discount
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Final Amount
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Status
                                                    </Table.Head>
                                                    <Table.Head className="text-right">
                                                        Actions
                                                    </Table.Head>
                                                </Table.Row>
                                            </Table.Header>
                                            <Table.Body>
                                                {orders.data.map((order) => (
                                                    <Table.Row key={order.id}>
                                                        <Table.Cell className="font-medium">
                                                            #
                                                            {order.order_number}
                                                        </Table.Cell>
                                                        <Table.Cell>
                                                            {formatDate(
                                                                order.dateposted
                                                            )}
                                                        </Table.Cell>
                                                        <Table.Cell>
                                                            {order.user ? (
                                                                <div className="flex items-center">
                                                                    <span className="truncate max-w-[120px]">
                                                                        {
                                                                            order
                                                                                .user
                                                                                .name
                                                                        }
                                                                    </span>
                                                                    <a
                                                                        href={`mailto:${order.user.email}`}
                                                                        className="ml-2 text-gray-400 hover:text-gray-500"
                                                                        target="_blank"
                                                                        rel="noopener noreferrer"
                                                                    >
                                                                        <Mail className="h-4 w-4" />
                                                                    </a>
                                                                </div>
                                                            ) : (
                                                                "Unknown User"
                                                            )}
                                                        </Table.Cell>
                                                        <Table.Cell>
                                                            $
                                                            {parseFloat(
                                                                order.order_amount
                                                            ).toFixed(2)}
                                                        </Table.Cell>
                                                        <Table.Cell className="text-green-600 dark:text-green-400">
                                                            $
                                                            {parseFloat(
                                                                order.discount
                                                            ).toFixed(2)}
                                                        </Table.Cell>
                                                        <Table.Cell>
                                                            $
                                                            {parseFloat(
                                                                order.net_amount
                                                            ).toFixed(2)}
                                                        </Table.Cell>
                                                        <Table.Cell>
                                                            <Badge
                                                                variant="outline"
                                                                className={
                                                                    order.payment_status ===
                                                                    "paid"
                                                                        ? "bg-green-100 text-green-800 border-green-200"
                                                                        : order.payment_status ===
                                                                          "pending"
                                                                        ? "bg-yellow-100 text-yellow-800 border-yellow-200"
                                                                        : "bg-gray-100 text-gray-800 border-gray-200"
                                                                }
                                                            >
                                                                {
                                                                    order.payment_status
                                                                }
                                                            </Badge>
                                                        </Table.Cell>
                                                        <Table.Cell className="text-right">
                                                            <Link
                                                                href={route(
                                                                    "admin.orders.show",
                                                                    order.id
                                                                )}
                                                                className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300"
                                                            >
                                                                <ExternalLink className="h-4 w-4" />
                                                            </Link>
                                                        </Table.Cell>
                                                    </Table.Row>
                                                ))}
                                            </Table.Body>
                                        </Table>
                                    ) : (
                                        <div className="text-center py-12">
                                            <ShoppingBag className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                                No orders found
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                This coupon hasn't been used in
                                                any orders yet.
                                            </p>
                                        </div>
                                    )}
                                </div>
                                {orders.data.length > 0 && (
                                    <div className="px-6 py-4 border-t border-gray-200 dark:border-gray-700">
                                        <Pagination links={orders.links} />
                                    </div>
                                )}
                            </div>
                        </TabsContent>

                        <TabsContent value="users">
                            <div className="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                                <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                        Users Who Used This Coupon
                                    </h3>
                                </div>
                                <div className="overflow-x-auto">
                                    {userUsages.data.length > 0 ? (
                                        <Table>
                                            <Table.Header>
                                                <Table.Row>
                                                    <Table.Head>
                                                        User
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Email
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Used On
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Order ID
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Order Total
                                                    </Table.Head>
                                                    <Table.Head>
                                                        Discount
                                                    </Table.Head>
                                                    <Table.Head className="text-right">
                                                        Actions
                                                    </Table.Head>
                                                </Table.Row>
                                            </Table.Header>
                                            <Table.Body>
                                                {userUsages.data.map(
                                                    (usage) => (
                                                        <Table.Row
                                                            key={usage.id}
                                                        >
                                                            <Table.Cell className="font-medium">
                                                                {usage.user
                                                                    ? usage.user
                                                                          .name
                                                                    : "Unknown User"}
                                                            </Table.Cell>
                                                            <Table.Cell>
                                                                {usage.user ? (
                                                                    <div className="flex items-center">
                                                                        <span className="truncate max-w-[150px]">
                                                                            {
                                                                                usage
                                                                                    .user
                                                                                    .email
                                                                            }
                                                                        </span>

                                                                        <a
                                                                            href={`mailto:${usage.user.email}`}
                                                                            className="ml-2 text-gray-400 hover:text-gray-500"
                                                                            target="_blank"
                                                                            rel="noopener noreferrer"
                                                                        >
                                                                            <Mail className="h-4 w-4" />
                                                                        </a>
                                                                    </div>
                                                                ) : (
                                                                    "Unknown"
                                                                )}
                                                            </Table.Cell>
                                                            <Table.Cell>
                                                                {formatDate(
                                                                    usage.used_at
                                                                )}
                                                            </Table.Cell>
                                                            <Table.Cell>
                                                                {usage.order_id && usage.order ? (
                                                                    <Link
                                                                        href={route(
                                                                            "admin.orders.show",
                                                                            usage.order_id
                                                                        )}
                                                                        className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300"
                                                                    >
                                                                        #
                                                                        {
                                                                            usage.order.order_number
                                                                        }
                                                                    </Link>
                                                                ) : (
                                                                    <span className="text-gray-500 dark:text-gray-400">N/A</span>
                                                                )}
                                                            </Table.Cell>
                                                            <Table.Cell>
                                                                $
                                                                {parseFloat(
                                                                    usage.order_total ||
                                                                        0
                                                                ).toFixed(2)}
                                                            </Table.Cell>
                                                            <Table.Cell className="text-green-600 dark:text-green-400">
                                                                $
                                                                {parseFloat(
                                                                    usage.discount_amount ||
                                                                        0
                                                                ).toFixed(2)}
                                                            </Table.Cell>
                                                            <Table.Cell className="text-right">
                                                                {usage.user && (
                                                                    <Link
                                                                        href={route(
                                                                            "admin.users.show",
                                                                            usage
                                                                                .user
                                                                                .id
                                                                        )}
                                                                        className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300"
                                                                    >
                                                                        <ExternalLink className="h-4 w-4" />
                                                                    </Link>
                                                                )}
                                                            </Table.Cell>
                                                        </Table.Row>
                                                    )
                                                )}
                                            </Table.Body>
                                        </Table>
                                    ) : (
                                        <div className="text-center py-12">
                                            <Users className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                                No user data found
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                This coupon hasn't been used by
                                                any users yet.
                                            </p>
                                        </div>
                                    )}
                                </div>
                                {userUsages.data.length > 0 && (
                                    <div className="px-6 py-4 border-t border-gray-200 dark:border-gray-700">
                                        <Pagination links={userUsages.links} />
                                    </div>
                                )}
                            </div>
                        </TabsContent>
                    </Tabs>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Show;
