import React, { useState, useEffect } from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import { DatePicker } from "@/Components/ui/DatePicker";
import {
    ArrowLeft,
    BarChart,
    Download,
    PieChart,
    Users,
    Calendar,
    DollarSign,
    Tag,
    TrendingUp,
    Filter,
    RefreshCw,
    Tag as TagIcon,
    Percent,
    Search,
    Clock,
    CalendarRange,
    Award,
    Zap,
    User,
    FileText,
    ChevronUp,
    ChevronDown,
    Cake,
} from "lucide-react";
import { Separator } from "@/Components/ui/Separator";
import Alert from "@/Components/ui/Alert";
import { Badge } from "@/Components/ui/Badge";
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/Components/ui/Tabs";
import EmptyState from "@/Components/ui/EmptyState";

const Statistics = ({
    couponUsageData,
    couponTypeStats,
    dailyStats,
    topUsers,
    startDate,
    endDate,
}) => {
    const { data, setData, get, processing } = useForm({
        start_date: startDate,
        end_date: endDate,
    });

    const [activeTab, setActiveTab] = useState("overview");
    const [chartData, setChartData] = useState(null);
    const [sortConfig, setSortConfig] = useState({
        key: "usage_count",
        direction: "desc",
    });

    // Process chart data on load and when data changes
    useEffect(() => {
        if (dailyStats && dailyStats.length > 0) {
            prepareChartData();
        }
    }, [dailyStats]);

    const prepareChartData = () => {
        // This would be replaced with actual chart library implementation
        // Just simulating the data preparation here
        setChartData({
            labels: dailyStats.map((day) => day.date),
            datasets: [
                {
                    label: "Usage Count",
                    data: dailyStats.map((day) => day.usage_count),
                },
                {
                    label: "Discount Amount",
                    data: dailyStats.map((day) => day.total_discount),
                },
            ],
        });
    };

    const handleDateRangeChange = () => {
        get(route("admin.coupons.statistics"), {
            preserveState: true,
        });
    };

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: "USD",
        }).format(amount);
    };

    // Calculate summary statistics
    const calculateSummary = () => {
        if (!couponUsageData || couponUsageData.length === 0) {
            return {
                totalUsage: 0,
                totalDiscount: 0,
                avgDiscount: 0,
                uniqueCoupons: 0,
            };
        }

        const totalUsage = couponUsageData.reduce(
            (sum, coupon) => sum + coupon.usage_count,
            0
        );
        const totalDiscount = couponUsageData.reduce(
            (sum, coupon) => sum + parseFloat(coupon.total_discount),
            0
        );
        const avgDiscount = totalUsage > 0 ? totalDiscount / totalUsage : 0;
        const uniqueCoupons = couponUsageData.length;

        return {
            totalUsage,
            totalDiscount,
            avgDiscount,
            uniqueCoupons,
        };
    };

    const summary = calculateSummary();

    // Get coupon type icon
    const getCouponTypeIcon = (type) => {
        const icons = {
            first_order: <TagIcon className="h-4 w-4 mr-1 text-blue-500" />,
            flash: <Zap className="h-4 w-4 mr-1 text-purple-500" />,
            referral: <User className="h-4 w-4 mr-1 text-indigo-500" />,
            loyalty: <Award className="h-4 w-4 mr-1 text-emerald-500" />,
            anniversary: <Cake className="h-4 w-4 mr-1 text-pink-500" />,
            seasonal: <CalendarRange className="h-4 w-4 mr-1 text-amber-500" />,
            volume: <FileText className="h-4 w-4 mr-1 text-cyan-500" />,
        };

        return (
            icons[type] || <TagIcon className="h-4 w-4 mr-1 text-gray-500" />
        );
    };

    // Get coupon type label
    const getCouponTypeLabel = (type) => {
        const labels = {
            first_order: "First Order",
            flash: "Flash Sale",
            referral: "Referral",
            loyalty: "Loyalty",
            anniversary: "Anniversary",
            seasonal: "Seasonal",
            volume: "Volume",
        };

        return labels[type] || type;
    };

    // Sort function for tables
    const requestSort = (key) => {
        const direction =
            sortConfig.key === key && sortConfig.direction === "asc"
                ? "desc"
                : "asc";
        setSortConfig({ key, direction });
    };

    // Sort data for display
    const getSortedData = (data, config = sortConfig) => {
        if (!data || data.length === 0) return [];

        const sortableData = [...data];
        sortableData.sort((a, b) => {
            if (a[config.key] < b[config.key]) {
                return config.direction === "asc" ? -1 : 1;
            }
            if (a[config.key] > b[config.key]) {
                return config.direction === "asc" ? 1 : -1;
            }
            return 0;
        });
        return sortableData;
    };

    return (
        <AdminLayout>
            <Head title="Coupon Statistics & Analytics" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <BarChart className="w-6 h-6 mr-2 text-blue-500" />
                                Coupon Statistics & Analytics
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Track and analyze coupon performance and usage
                                patterns
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0 flex flex-wrap gap-2">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Coupons
                                </Link>
                            </Button>
                            <Button variant="outline" size="sm">
                                <Download className="w-4 h-4 mr-2" />
                                Export Data
                            </Button>
                        </div>
                    </div>

                    {/* Date Range Filter */}
                    <Card className="mb-6">
                        <CardContent className="py-4">
                            <div className="flex flex-col sm:flex-row sm:items-end gap-4">
                                <div className="flex-1">
                                    <Label htmlFor="start_date">
                                        Start Date
                                    </Label>
                                    <DatePicker
                                        id="start_date"
                                        date={
                                            data.start_date
                                                ? new Date(data.start_date)
                                                : null
                                        }
                                        onSelect={(date) =>
                                            setData("start_date", date)
                                        }
                                        className="mt-1 w-full"
                                    />
                                </div>
                                <div className="flex-1">
                                    <Label htmlFor="end_date">End Date</Label>
                                    <DatePicker
                                        id="end_date"
                                        date={
                                            data.end_date
                                                ? new Date(data.end_date)
                                                : null
                                        }
                                        onSelect={(date) =>
                                            setData("end_date", date)
                                        }
                                        className="mt-1 w-full"
                                    />
                                </div>
                                <div className="flex-none">
                                    <Button
                                        onClick={handleDateRangeChange}
                                        disabled={processing}
                                    >
                                        {processing ? (
                                            <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                        ) : (
                                            <Filter className="w-4 h-4 mr-2" />
                                        )}
                                        Filter
                                    </Button>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Summary Cards */}
                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                        <Card>
                            <CardContent className="p-6">
                                <div className="flex justify-between items-start">
                                    <div>
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                            Total Redemptions
                                        </p>
                                        <p className="text-3xl font-bold text-gray-900 dark:text-white mt-1">
                                            {summary.totalUsage}
                                        </p>
                                    </div>
                                    <div className="p-2 bg-blue-100 dark:bg-blue-900/30 rounded-full">
                                        <Tag className="h-6 w-6 text-blue-600 dark:text-blue-400" />
                                    </div>
                                </div>
                                <div className="mt-4 flex items-center text-sm">
                                    <TrendingUp className="h-4 w-4 text-green-500 mr-1" />
                                    <span className="text-green-500 font-medium">
                                        Active
                                    </span>
                                    <span className="text-gray-500 dark:text-gray-400 ml-1">
                                        in selected period
                                    </span>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent className="p-6">
                                <div className="flex justify-between items-start">
                                    <div>
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                            Total Discount Amount
                                        </p>
                                        <p className="text-3xl font-bold text-gray-900 dark:text-white mt-1">
                                            {formatCurrency(
                                                summary.totalDiscount
                                            )}
                                        </p>
                                    </div>
                                    <div className="p-2 bg-green-100 dark:bg-green-900/30 rounded-full">
                                        <DollarSign className="h-6 w-6 text-green-600 dark:text-green-400" />
                                    </div>
                                </div>
                                <div className="mt-4 text-sm text-gray-500 dark:text-gray-400">
                                    Avg. discount per redemption:{" "}
                                    {formatCurrency(summary.avgDiscount)}
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent className="p-6">
                                <div className="flex justify-between items-start">
                                    <div>
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                            Unique Coupons Used
                                        </p>
                                        <p className="text-3xl font-bold text-gray-900 dark:text-white mt-1">
                                            {summary.uniqueCoupons}
                                        </p>
                                    </div>
                                    <div className="p-2 bg-purple-100 dark:bg-purple-900/30 rounded-full">
                                        <TagIcon className="h-6 w-6 text-purple-600 dark:text-purple-400" />
                                    </div>
                                </div>
                                <div className="mt-4 text-sm text-gray-500 dark:text-gray-400">
                                    Out of {couponUsageData?.length || 0} total
                                    active coupons
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent className="p-6">
                                <div className="flex justify-between items-start">
                                    <div>
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                            Date Range
                                        </p>
                                        <p className="text-xl font-bold text-gray-900 dark:text-white mt-1">
                                            {data.start_date
                                                ? new Date(
                                                      data.start_date
                                                  ).toLocaleDateString()
                                                : "All time"}{" "}
                                            -{" "}
                                            {data.end_date
                                                ? new Date(
                                                      data.end_date
                                                  ).toLocaleDateString()
                                                : "Present"}
                                        </p>
                                    </div>
                                    <div className="p-2 bg-amber-100 dark:bg-amber-900/30 rounded-full">
                                        <Calendar className="h-6 w-6 text-amber-600 dark:text-amber-400" />
                                    </div>
                                </div>
                                <div className="mt-4 text-sm text-gray-500 dark:text-gray-400">
                                    {data.start_date && data.end_date
                                        ? `${Math.ceil(
                                              (new Date(data.end_date) -
                                                  new Date(data.start_date)) /
                                                  (1000 * 60 * 60 * 24)
                                          )} days`
                                        : "All historical data"}
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Tabs for different data views */}
                    <Tabs value={activeTab} onValueChange={setActiveTab}>
                        <TabsList className="mb-6">
                            <TabsTrigger value="overview">Overview</TabsTrigger>
                            <TabsTrigger value="by-type">
                                By Coupon Type
                            </TabsTrigger>
                            <TabsTrigger value="by-coupon">
                                By Coupon
                            </TabsTrigger>
                            <TabsTrigger value="by-user">By User</TabsTrigger>
                        </TabsList>

                        {/* Overview Tab */}
                        <TabsContent value="overview" className="space-y-6">
                            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                                {/* Daily Usage Chart */}
                                <Card className="lg:col-span-2">
                                    <CardHeader>
                                        <CardTitle>
                                            Daily Coupon Usage
                                        </CardTitle>
                                        <CardDescription>
                                            Tracking of coupon redemptions and
                                            discount amounts over time
                                        </CardDescription>
                                    </CardHeader>
                                    <CardContent>
                                        {dailyStats && dailyStats.length > 0 ? (
                                            <div className="h-80 w-full">
                                                {/* Chart would be implemented here with library like Chart.js or Recharts */}
                                                <div className="bg-gray-100 dark:bg-gray-800 h-full w-full flex items-center justify-center rounded-md">
                                                    <div className="text-center">
                                                        <BarChart className="h-16 w-16 mx-auto text-gray-400" />
                                                        <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                                                            Chart visualization
                                                            would be displayed
                                                            here using actual
                                                            chart library
                                                        </p>
                                                        <p className="mt-1 text-xs text-gray-500 dark:text-gray-500">
                                                            {dailyStats.length}{" "}
                                                            data points from{" "}
                                                            {
                                                                dailyStats[0]
                                                                    ?.date
                                                            }{" "}
                                                            to{" "}
                                                            {
                                                                dailyStats[
                                                                    dailyStats.length -
                                                                        1
                                                                ]?.date
                                                            }
                                                        </p>
                                                    </div>
                                                </div>
                                            </div>
                                        ) : (
                                            <EmptyState
                                                icon={BarChart}
                                                title="No daily usage data"
                                                description="There is no daily usage data available for the selected period."
                                                compact
                                            />
                                        )}
                                    </CardContent>
                                </Card>

                                {/* Coupon Type Distribution */}
                                <Card>
                                    <CardHeader>
                                        <CardTitle>
                                            Coupon Type Distribution
                                        </CardTitle>
                                        <CardDescription>
                                            Breakdown of coupon usage by type
                                        </CardDescription>
                                    </CardHeader>
                                    <CardContent>
                                        {couponTypeStats &&
                                        couponTypeStats.length > 0 ? (
                                            <div>
                                                <div className="h-48 mb-6">
                                                    {/* Pie chart would be implemented here */}
                                                    <div className="bg-gray-100 dark:bg-gray-800 h-full w-full flex items-center justify-center rounded-md">
                                                        <div className="text-center">
                                                            <PieChart className="h-12 w-12 mx-auto text-gray-400" />
                                                            <p className="mt-2 text-xs text-gray-600 dark:text-gray-400">
                                                                Pie chart
                                                                visualization
                                                                would be
                                                                displayed here
                                                            </p>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div className="space-y-3">
                                                    {couponTypeStats.map(
                                                        (stat, index) => (
                                                            <div
                                                                key={index}
                                                                className="flex items-center justify-between"
                                                            >
                                                                <div className="flex items-center">
                                                                    {getCouponTypeIcon(
                                                                        stat.coupon_type
                                                                    )}
                                                                    <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                                                        {getCouponTypeLabel(
                                                                            stat.coupon_type
                                                                        )}
                                                                    </span>
                                                                </div>
                                                                <div className="flex items-center space-x-4">
                                                                    <Badge variant="outline">
                                                                        {
                                                                            stat.usage_count
                                                                        }{" "}
                                                                        uses
                                                                    </Badge>
                                                                    <span className="text-sm font-medium text-green-600 dark:text-green-400">
                                                                        {formatCurrency(
                                                                            stat.total_discount
                                                                        )}
                                                                    </span>
                                                                </div>
                                                            </div>
                                                        )
                                                    )}
                                                </div>
                                            </div>
                                        ) : (
                                            <EmptyState
                                                icon={PieChart}
                                                title="No coupon type data"
                                                description="There is no coupon type data available for the selected period."
                                                compact
                                            />
                                        )}
                                    </CardContent>
                                </Card>

                                {/* Top Users */}
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Top Users</CardTitle>
                                        <CardDescription>
                                            Customers with the highest discount
                                            amounts
                                        </CardDescription>
                                    </CardHeader>
                                    <CardContent>
                                        {topUsers && topUsers.length > 0 ? (
                                            <div className="space-y-4">
                                                {topUsers
                                                    .slice(0, 5)
                                                    .map((user, index) => (
                                                        <div
                                                            key={index}
                                                            className="flex items-start justify-between"
                                                        >
                                                            <div className="flex items-center">
                                                                <div className="flex-shrink-0 h-8 w-8 bg-gray-200 dark:bg-gray-700 rounded-full flex items-center justify-center">
                                                                    <span className="text-xs font-medium">
                                                                        {user.user?.name?.charAt(
                                                                            0
                                                                        ) ||
                                                                            "U"}
                                                                    </span>
                                                                </div>
                                                                <div className="ml-3">
                                                                    <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                                        {user
                                                                            .user
                                                                            ?.name ||
                                                                            "Unknown User"}
                                                                    </p>
                                                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                        {
                                                                            user.usage_count
                                                                        }{" "}
                                                                        redemptions
                                                                    </p>
                                                                </div>
                                                            </div>
                                                            <div className="text-right">
                                                                <p className="text-sm font-medium text-green-600 dark:text-green-400">
                                                                    {formatCurrency(
                                                                        user.total_discount
                                                                    )}
                                                                </p>
                                                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                    total
                                                                    discount
                                                                </p>
                                                            </div>
                                                        </div>
                                                    ))}

                                                {topUsers.length > 5 && (
                                                    <Button
                                                        variant="outline"
                                                        size="sm"
                                                        className="w-full mt-2"
                                                        onClick={() =>
                                                            setActiveTab(
                                                                "by-user"
                                                            )
                                                        }
                                                    >
                                                        View All Users
                                                    </Button>
                                                )}
                                            </div>
                                        ) : (
                                            <EmptyState
                                                icon={Users}
                                                title="No user data"
                                                description="There is no user data available for the selected period."
                                                compact
                                            />
                                        )}
                                    </CardContent>
                                </Card>
                            </div>
                        </TabsContent>

                        {/* By Coupon Type Tab */}
                        <TabsContent value="by-type" className="space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>
                                        Coupon Performance by Type
                                    </CardTitle>
                                    <CardDescription>
                                        Detailed breakdown of coupon performance
                                        across different types
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    {couponTypeStats &&
                                    couponTypeStats.length > 0 ? (
                                        <div className="overflow-x-auto">
                                            <table className="w-full border-collapse">
                                                <thead>
                                                    <tr className="border-b border-gray-200 dark:border-gray-700">
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "coupon_type"
                                                                    )
                                                                }
                                                            >
                                                                Coupon Type
                                                                {sortConfig.key ===
                                                                    "coupon_type" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "usage_count"
                                                                    )
                                                                }
                                                            >
                                                                Usage Count
                                                                {sortConfig.key ===
                                                                    "usage_count" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "total_discount"
                                                                    )
                                                                }
                                                            >
                                                                Total Discount
                                                                {sortConfig.key ===
                                                                    "total_discount" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            Avg. Discount
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            % of Total
                                                        </th>
                                                    </tr>
                                                </thead>
                                                <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                                    {getSortedData(
                                                        couponTypeStats
                                                    ).map((stat, index) => {
                                                        const avgDiscount =
                                                            stat.usage_count > 0
                                                                ? parseFloat(
                                                                      stat.total_discount
                                                                  ) /
                                                                  stat.usage_count
                                                                : 0;
                                                        const percentOfTotal =
                                                            summary.totalUsage >
                                                            0
                                                                ? (stat.usage_count /
                                                                      summary.totalUsage) *
                                                                  100
                                                                : 0;

                                                        return (
                                                            <tr
                                                                key={index}
                                                                className="hover:bg-gray-50 dark:hover:bg-gray-800/50"
                                                            >
                                                                <td className="px-4 py-4 whitespace-nowrap">
                                                                    <div className="flex items-center">
                                                                        {getCouponTypeIcon(
                                                                            stat.coupon_type
                                                                        )}
                                                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                                            {getCouponTypeLabel(
                                                                                stat.coupon_type
                                                                            )}
                                                                        </span>
                                                                    </div>
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                                                    {
                                                                        stat.usage_count
                                                                    }
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-green-600 dark:text-green-400 font-medium">
                                                                    {formatCurrency(
                                                                        stat.total_discount
                                                                    )}
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                                                    {formatCurrency(
                                                                        avgDiscount
                                                                    )}
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap">
                                                                    <div className="flex items-center">
                                                                        <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2.5 mr-2 max-w-[100px]">
                                                                            <div
                                                                                className="bg-blue-600 dark:bg-blue-500 h-2.5 rounded-full"
                                                                                style={{
                                                                                    width: `${percentOfTotal}%`,
                                                                                }}
                                                                            ></div>
                                                                        </div>
                                                                        <span className="text-sm text-gray-700 dark:text-gray-300">
                                                                            {percentOfTotal.toFixed(
                                                                                1
                                                                            )}
                                                                            %
                                                                        </span>
                                                                    </div>
                                                                </td>
                                                            </tr>
                                                        );
                                                    })}
                                                </tbody>
                                            </table>
                                        </div>
                                    ) : (
                                        <EmptyState
                                            icon={BarChart}
                                            title="No coupon type data"
                                            description="There is no data available for the selected period."
                                        />
                                    )}
                                </CardContent>
                            </Card>
                        </TabsContent>

                        {/* By Coupon Tab */}
                        <TabsContent value="by-coupon" className="space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>
                                        Individual Coupon Performance
                                    </CardTitle>
                                    <CardDescription>
                                        Detailed breakdown of each coupon's
                                        usage and discount amount
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    {couponUsageData &&
                                    couponUsageData.length > 0 ? (
                                        <div className="overflow-x-auto">
                                            <table className="w-full border-collapse">
                                                <thead>
                                                    <tr className="border-b border-gray-200 dark:border-gray-700">
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "coupon.coupon_code"
                                                                    )
                                                                }
                                                            >
                                                                Coupon Code
                                                                {sortConfig.key ===
                                                                    "coupon.coupon_code" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            Type
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "usage_count"
                                                                    )
                                                                }
                                                            >
                                                                Uses
                                                                {sortConfig.key ===
                                                                    "usage_count" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "total_discount"
                                                                    )
                                                                }
                                                            >
                                                                Total Discount
                                                                {sortConfig.key ===
                                                                    "total_discount" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            Avg. Discount
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            Status
                                                        </th>
                                                    </tr>
                                                </thead>
                                                <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                                    {getSortedData(
                                                        couponUsageData
                                                    ).map(
                                                        (couponData, index) => {
                                                            const avgDiscount =
                                                                couponData.usage_count >
                                                                0
                                                                    ? parseFloat(
                                                                          couponData.total_discount
                                                                      ) /
                                                                      couponData.usage_count
                                                                    : 0;

                                                            return (
                                                                <tr
                                                                    key={index}
                                                                    className="hover:bg-gray-50 dark:hover:bg-gray-800/50"
                                                                >
                                                                    <td className="px-4 py-4 whitespace-nowrap">
                                                                        <div className="flex items-center">
                                                                            <TagIcon className="h-4 w-4 mr-1 text-gray-400" />
                                                                            <span className="text-sm font-mono text-gray-900 dark:text-white">
                                                                                {couponData
                                                                                    .coupon
                                                                                    ?.coupon_code ||
                                                                                    "Unknown"}
                                                                            </span>
                                                                        </div>
                                                                    </td>
                                                                    <td className="px-4 py-4 whitespace-nowrap">
                                                                        <Badge
                                                                            variant="outline"
                                                                            className="font-normal"
                                                                        >
                                                                            {getCouponTypeLabel(
                                                                                couponData
                                                                                    .coupon
                                                                                    ?.coupon_type ||
                                                                                    "unknown"
                                                                            )}
                                                                        </Badge>
                                                                    </td>
                                                                    <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                                                        {
                                                                            couponData.usage_count
                                                                        }
                                                                    </td>
                                                                    <td className="px-4 py-4 whitespace-nowrap text-sm text-green-600 dark:text-green-400 font-medium">
                                                                        {formatCurrency(
                                                                            couponData.total_discount
                                                                        )}
                                                                    </td>
                                                                    <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                                                        {formatCurrency(
                                                                            avgDiscount
                                                                        )}
                                                                    </td>
                                                                    <td className="px-4 py-4 whitespace-nowrap">
                                                                        {couponData
                                                                            .coupon
                                                                            ?.is_active ? (
                                                                            <Badge
                                                                                variant="success"
                                                                                className="bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300"
                                                                            >
                                                                                Active
                                                                            </Badge>
                                                                        ) : (
                                                                            <Badge
                                                                                variant="secondary"
                                                                                className="bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300"
                                                                            >
                                                                                Inactive
                                                                            </Badge>
                                                                        )}
                                                                    </td>
                                                                </tr>
                                                            );
                                                        }
                                                    )}
                                                </tbody>
                                            </table>
                                        </div>
                                    ) : (
                                        <EmptyState
                                            icon={TagIcon}
                                            title="No coupon usage data"
                                            description="There is no coupon usage data available for the selected period."
                                        />
                                    )}
                                </CardContent>
                            </Card>
                        </TabsContent>

                        {/* By User Tab */}
                        <TabsContent value="by-user" className="space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>
                                        User Coupon Redemptions
                                    </CardTitle>
                                    <CardDescription>
                                        Analysis of coupon usage by customer
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    {topUsers && topUsers.length > 0 ? (
                                        <div className="overflow-x-auto">
                                            <table className="w-full border-collapse">
                                                <thead>
                                                    <tr className="border-b border-gray-200 dark:border-gray-700">
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "user.name"
                                                                    )
                                                                }
                                                            >
                                                                User
                                                                {sortConfig.key ===
                                                                    "user.name" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "usage_count"
                                                                    )
                                                                }
                                                            >
                                                                Redemptions
                                                                {sortConfig.key ===
                                                                    "usage_count" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            <button
                                                                className="flex items-center focus:outline-none"
                                                                onClick={() =>
                                                                    requestSort(
                                                                        "total_discount"
                                                                    )
                                                                }
                                                            >
                                                                Total Discount
                                                                {sortConfig.key ===
                                                                    "total_discount" &&
                                                                    (sortConfig.direction ===
                                                                    "asc" ? (
                                                                        <ChevronUp className="h-4 w-4 ml-1" />
                                                                    ) : (
                                                                        <ChevronDown className="h-4 w-4 ml-1" />
                                                                    ))}
                                                            </button>
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            Avg. Discount
                                                        </th>
                                                        <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                            % of Total
                                                        </th>
                                                    </tr>
                                                </thead>
                                                <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                                    {getSortedData(
                                                        topUsers
                                                    ).map((userData, index) => {
                                                        const avgDiscount =
                                                            userData.usage_count >
                                                            0
                                                                ? parseFloat(
                                                                      userData.total_discount
                                                                  ) /
                                                                  userData.usage_count
                                                                : 0;
                                                        const percentOfTotal =
                                                            summary.totalDiscount >
                                                            0
                                                                ? (parseFloat(
                                                                      userData.total_discount
                                                                  ) /
                                                                      summary.totalDiscount) *
                                                                  100
                                                                : 0;

                                                        return (
                                                            <tr
                                                                key={index}
                                                                className="hover:bg-gray-50 dark:hover:bg-gray-800/50"
                                                            >
                                                                <td className="px-4 py-4 whitespace-nowrap">
                                                                    <div className="flex items-center">
                                                                        <div className="flex-shrink-0 h-8 w-8 bg-gray-200 dark:bg-gray-700 rounded-full flex items-center justify-center">
                                                                            <span className="text-xs font-medium">
                                                                                {userData.user?.name?.charAt(
                                                                                    0
                                                                                ) ||
                                                                                    "U"}
                                                                            </span>
                                                                        </div>
                                                                        <div className="ml-3">
                                                                            <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                                                {userData
                                                                                    .user
                                                                                    ?.name ||
                                                                                    "Unknown User"}
                                                                            </p>
                                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                                {userData
                                                                                    .user
                                                                                    ?.email ||
                                                                                    ""}
                                                                            </p>
                                                                        </div>
                                                                    </div>
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                                                    {
                                                                        userData.usage_count
                                                                    }
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-green-600 dark:text-green-400 font-medium">
                                                                    {formatCurrency(
                                                                        userData.total_discount
                                                                    )}
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                                                    {formatCurrency(
                                                                        avgDiscount
                                                                    )}
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap">
                                                                    <div className="flex items-center">
                                                                        <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2.5 mr-2 max-w-[100px]">
                                                                            <div
                                                                                className="bg-blue-600 dark:bg-blue-500 h-2.5 rounded-full"
                                                                                style={{
                                                                                    width: `${percentOfTotal}%`,
                                                                                }}
                                                                            ></div>
                                                                        </div>
                                                                        <span className="text-sm text-gray-700 dark:text-gray-300">
                                                                            {percentOfTotal.toFixed(
                                                                                1
                                                                            )}
                                                                            %
                                                                        </span>
                                                                    </div>
                                                                </td>
                                                            </tr>
                                                        );
                                                    })}
                                                </tbody>
                                            </table>
                                        </div>
                                    ) : (
                                        <EmptyState
                                            icon={Users}
                                            title="No user redemption data"
                                            description="There is no user redemption data available for the selected period."
                                        />
                                    )}
                                </CardContent>
                            </Card>
                        </TabsContent>
                    </Tabs>

                    {/* Export Options */}
                    <div className="mt-8 border-t border-gray-200 dark:border-gray-700 pt-4">
                        <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Export Options
                        </h3>
                        <div className="flex flex-wrap gap-2">
                            <Button variant="outline" size="sm">
                                <Download className="w-4 h-4 mr-2" />
                                Export as CSV
                            </Button>
                            <Button variant="outline" size="sm">
                                <Download className="w-4 h-4 mr-2" />
                                Export as PDF
                            </Button>
                            <Button variant="outline" size="sm">
                                <Download className="w-4 h-4 mr-2" />
                                Export Chart Images
                            </Button>
                        </div>
                    </div>

                    {/* Tips and insights */}
                    <Alert
                        variant="info"
                        title="Analytics Insights"
                        className="mt-6"
                    >
                        <div className="text-sm">
                            <p className="mb-2">
                                Tips for optimizing your coupon strategy based
                                on this data:
                            </p>
                            <ul className="list-disc pl-5 space-y-1">
                                <li>
                                    Focus on coupon types with the highest
                                    average discount to revenue ratio
                                </li>
                                <li>
                                    Consider targeting your most active coupon
                                    users with special offers
                                </li>
                                <li>
                                    Analyze seasonal trends to time your
                                    promotions more effectively
                                </li>
                                <li>
                                    Use the daily usage chart to identify
                                    periods where coupon usage could be
                                    increased
                                </li>
                            </ul>
                        </div>
                    </Alert>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Statistics;
