import React from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import { Save, ArrowLeft } from "lucide-react";

export default function Edit({ currency }) {
    const { data, setData, put, processing, errors } = useForm({
        name: currency.name || "",
        currency_code: currency.currency_code || "",
        value: currency.value || "1.000000",
        is_active: currency.is_active ?? true,
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        put(route("currencies.update", currency.id));
    };

    return (
        <AdminLayout>
            <Head title={`Edit ${currency.name}`} />

            <div className="py-6">
                <div className="max-w-3xl mx-auto">
                    {/* Header */}
                    <div className="mb-6">
                        <Link
                            href={route("currencies.index")}
                            className="inline-flex items-center gap-2 text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white mb-4"
                        >
                            <ArrowLeft size={20} />
                            Back to Currencies
                        </Link>
                        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                            Edit Currency
                        </h1>
                        <p className="text-gray-600 dark:text-gray-400 mt-1">
                            Update currency information
                        </p>
                    </div>

                    {/* Form */}
                    <form
                        onSubmit={handleSubmit}
                        className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6"
                    >
                        <div className="space-y-6">
                            {/* Currency Name */}
                            <div>
                                <label className="block text-sm font-medium text-gray-900 dark:text-white mb-2">
                                    Currency Name <span className="text-red-500">*</span>
                                </label>
                                <input
                                    type="text"
                                    value={data.name}
                                    onChange={(e) => setData("name", e.target.value)}
                                    placeholder="e.g., US Dollar, Euro"
                                    className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-indigo-500"
                                    required
                                />
                                {errors.name && (
                                    <p className="text-red-600 text-sm mt-1">{errors.name}</p>
                                )}
                            </div>

                            {/* Currency Code */}
                            <div>
                                <label className="block text-sm font-medium text-gray-900 dark:text-white mb-2">
                                    Currency Code <span className="text-red-500">*</span>
                                </label>
                                <input
                                    type="text"
                                    value={data.currency_code}
                                    onChange={(e) =>
                                        setData("currency_code", e.target.value.toUpperCase())
                                    }
                                    placeholder="e.g., USD, EUR, GBP"
                                    maxLength={10}
                                    className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-indigo-500 uppercase"
                                    required
                                    disabled={currency.currency_code === "USD"}
                                />
                                <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                                    {currency.currency_code === "USD"
                                        ? "🔒 Base currency code cannot be changed"
                                        : "ISO 4217 currency code (3 letters)"}
                                </p>
                                {errors.currency_code && (
                                    <p className="text-red-600 text-sm mt-1">
                                        {errors.currency_code}
                                    </p>
                                )}
                            </div>

                            {/* Exchange Rate */}
                            <div>
                                <label className="block text-sm font-medium text-gray-900 dark:text-white mb-2">
                                    Exchange Rate (vs USD) <span className="text-red-500">*</span>
                                </label>
                                <input
                                    type="number"
                                    value={data.value}
                                    onChange={(e) => setData("value", e.target.value)}
                                    placeholder="1.000000"
                                    step="0.000001"
                                    min="0"
                                    className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-indigo-500"
                                    required
                                />
                                <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                                    1 USD = {data.value || "1.000000"} {data.currency_code || "XXX"}
                                </p>
                                {errors.value && (
                                    <p className="text-red-600 text-sm mt-1">{errors.value}</p>
                                )}
                            </div>

                            {/* Active Status */}
                            <div>
                                <label className="flex items-center gap-3 cursor-pointer">
                                    <input
                                        type="checkbox"
                                        checked={data.is_active}
                                        onChange={(e) => setData("is_active", e.target.checked)}
                                        className="w-5 h-5 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500"
                                        disabled={currency.currency_code === "USD" && data.is_active}
                                    />
                                    <div>
                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                            Active Currency
                                        </span>
                                        <p className="text-xs text-gray-600 dark:text-gray-400">
                                            {currency.currency_code === "USD"
                                                ? "🔒 Base currency must remain active"
                                                : "Currency will be visible on the order form"}
                                        </p>
                                    </div>
                                </label>
                            </div>

                            {/* Info Box */}
                            <div className="p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
                                <p className="text-sm text-blue-800 dark:text-blue-200">
                                    💡 <strong>Tip:</strong> Keep exchange rates updated to reflect current market values for accurate pricing.
                                </p>
                            </div>
                        </div>

                        {/* Action Buttons */}
                        <div className="flex gap-3 mt-6 pt-6 border-t border-gray-200 dark:border-gray-700">
                            <Link
                                href={route("currencies.index")}
                                className="flex-1 px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-700 dark:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-700 text-center transition-colors"
                            >
                                Cancel
                            </Link>
                            <button
                                type="submit"
                                disabled={processing}
                                className="flex-1 flex items-center justify-center gap-2 px-4 py-2 bg-indigo-600 hover:bg-indigo-700 disabled:bg-gray-400 text-white font-medium rounded-lg transition-colors"
                            >
                                <Save size={18} />
                                {processing ? "Saving..." : "Save Changes"}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
}
