import AdminLayout from "@/Layouts/AdminLayout";
import {
    Users,
    FileText,
    Mail,
    ExternalLink,
    DollarSign,
    Repeat,
    TrendingDown,
    ShoppingCart,
    ArrowUp,
    ArrowDown,
    Percent,
    Search,
    Plus,
    Edit,
    Image,
} from "lucide-react";
import { Link, Head } from "@inertiajs/react";
import CleanPagination from "@/Components/CleanPagination";
import DashboardCharts from "@/Components/DashboardCharts";
import SectionContainer from "@/Components/SectionContainer";
import KPICard from "@/Components/KPICard";
import UrgentOrdersSummary from "@/Components/UrgentOrdersSummary";

export default function Dashboard({
    totalClients,
    totalOrders,
    totalWriters,
    chartData,
    clientsOrders,
    totalRevenue,
    grossRevenue,
    clients,
    refundStats,
    urgentOrders,
    urgentOrdersCount,
}) {
    // Function to format currency
    const formatCurrency = (amount) => {
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: "USD",
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
        }).format(amount);
    };

    return (
        <AdminLayout>
            <Head title="Admin Dashboard" />
            <div className="min-h-screen bg-slate-100 dark:bg-gray-900">
                {/* Header Section */}
                <div className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700 mb-8 shadow-sm">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                            Admin Dashboard
                        </h1>
                        <p className="mt-2 text-gray-600 dark:text-gray-400">
                            Monitor your business performance and manage operations
                        </p>
                    </div>
                </div>

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 pb-8">


                     {/* Main KPI Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                        <KPICard
                            label="GROSS REVENUE"
                            value={formatCurrency(grossRevenue)}
                            trend={{
                                value: 12.5,
                                label: 'vs last month'
                            }}
                            icon={DollarSign}
                            colorScheme="green"
                        />

                        <KPICard
                            label="TOTAL ORDERS"
                            value={totalOrders}
                            trend={{
                                value: 8.3,
                                label: 'vs last month'
                            }}
                            icon={ShoppingCart}
                            colorScheme="purple"
                        />

                        <KPICard
                            label="ACTIVE CLIENTS"
                            value={totalClients}
                            trend={{
                                value: 15.2,
                                label: 'vs last month'
                            }}
                            icon={Users}
                            colorScheme="blue"
                        />

                        <KPICard
                            label="REFUND RATE"
                            value={`${refundStats.refundRate}%`}
                            trend={{
                                value: -2.1,
                                label: 'vs last month'
                            }}
                            icon={Percent}
                            colorScheme="orange"
                        />
                    </div>

                    {/* Elegant Urgent Orders Summary */}
                    <div className="mb-8">
                        <UrgentOrdersSummary 
                            urgentOrders={urgentOrders}
                            urgentOrdersCount={urgentOrdersCount}
                            userType="admin"
                        />
                    </div>

                    {/* CMS Management Section */}
                    <SectionContainer
                        className="mb-8 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 shadow-sm"
                        title="Content Management System"
                        titleClassName="text-lg font-semibold text-gray-900 dark:text-white border-b border-gray-200 dark:border-gray-700"
                    >
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                            {/* Service Pages Management */}
                            <div className="group bg-gradient-to-br from-blue-500 to-blue-600 dark:from-blue-600 dark:to-blue-700 rounded-xl shadow-lg p-6 text-white hover:shadow-xl transition-all duration-300">
                                <div className="flex items-center justify-between mb-4">
                                    <FileText className="w-8 h-8 text-blue-200" />
                                </div>
                                <Link 
                                    href="/admin/cms/pages"
                                    className="text-lg font-semibold mb-2 hover:underline inline-block"
                                >
                                    Service Pages
                                </Link>
                                <p className="text-blue-100 text-sm mb-4 mt-2">
                                    Manage essay writing, research papers, and other service pages
                                </p>
                                <div className="flex items-center text-blue-200 text-sm">
                                    <span>Edit Content & SEO</span>
                                </div>
                            </div>

                            {/* Content Blocks */}
                            <div className="group bg-gradient-to-br from-green-500 to-green-600 dark:from-green-600 dark:to-green-700 rounded-xl shadow-lg p-6 text-white hover:shadow-xl transition-all duration-300">
                                <div className="flex items-center justify-between mb-4">
                                    <FileText className="w-8 h-8 text-green-200" />
                                </div>
                                <Link 
                                    href="/admin/cms/content-blocks"
                                    className="text-lg font-semibold mb-2 hover:underline inline-block"
                                >
                                    Content Blocks
                                </Link>
                                <p className="text-green-100 text-sm mb-4 mt-2">
                                    Manage hero sections, benefits, FAQs, and other content blocks
                                </p>
                                <div className="flex items-center text-green-200 text-sm">
                                    <span>Modular Content</span>
                                </div>
                            </div>

                            {/* SEO Management */}
                            <div className="group bg-gradient-to-br from-purple-500 to-purple-600 dark:from-purple-600 dark:to-purple-700 rounded-xl shadow-lg p-6 text-white hover:shadow-xl transition-all duration-300">
                                <div className="flex items-center justify-between mb-4">
                                    <Search className="w-8 h-8 text-purple-200" />
                                </div>
                                <Link 
                                    href="/admin/cms/pages"
                                    className="text-lg font-semibold mb-2 hover:underline inline-block"
                                >
                                    SEO Management
                                </Link>
                                <p className="text-purple-100 text-sm mb-4 mt-2">
                                    Optimize meta tags, structured data, and search visibility
                                </p>
                                <div className="flex items-center text-purple-200 text-sm">
                                    <span>Search Optimization</span>
                                </div>
                            </div>

                            {/* Media Library */}
                            <div className="group bg-gradient-to-br from-orange-500 to-orange-600 dark:from-orange-600 dark:to-orange-700 rounded-xl shadow-lg p-6 text-white hover:shadow-xl transition-all duration-300">
                                <div className="flex items-center justify-between mb-4">
                                    <Image className="w-8 h-8 text-orange-200" />
                                </div>
                                <Link 
                                    href="/admin/cms/media"
                                    className="text-lg font-semibold mb-2 hover:underline inline-block"
                                >
                                    Media Library
                                </Link>
                                <p className="text-orange-100 text-sm mb-4 mt-2">
                                    Upload, organize, and optimize images and media files
                                </p>
                                <div className="flex items-center text-orange-200 text-sm">
                                    <span>Asset Management</span>
                                </div>
                            </div>

                            {/* Quick Actions */}
                            <div className="md:col-span-2 lg:col-span-3 bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-700 dark:to-gray-800 rounded-xl shadow-lg p-6">
                                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                    Quick Actions
                                </h3>
                                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                    <Link
                                        href="/admin/cms/pages/create"
                                        className="flex items-center gap-3 p-3 bg-white dark:bg-gray-800 rounded-lg hover:shadow-md transition-all duration-200"
                                    >
                                        <Plus className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                            Create New Page
                                        </span>
                                    </Link>
                                    <Link
                                        href="/admin/cms/pages?type=service_page"
                                        className="flex items-center gap-3 p-3 bg-white dark:bg-gray-800 rounded-lg hover:shadow-md transition-all duration-200"
                                    >
                                        <FileText className="w-5 h-5 text-green-600 dark:text-green-400" />
                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                            View Service Pages
                                        </span>
                                    </Link>
                                    <Link
                                        href="/admin/cms/pages?status=draft"
                                        className="flex items-center gap-3 p-3 bg-white dark:bg-gray-800 rounded-lg hover:shadow-md transition-all duration-200"
                                    >
                                        <Edit className="w-5 h-5 text-yellow-600 dark:text-yellow-400" />
                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                            Review Drafts
                                        </span>
                                    </Link>
                                </div>
                            </div>
                        </div>
                    </SectionContainer>

                    {/* Revenue & Refund Stats */}
                    <SectionContainer
                        className="mb-8 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 shadow-sm"
                        title="Revenue & Refund Insights"
                        titleClassName="text-lg font-semibold text-gray-900 dark:text-white border-b border-gray-200 dark:border-gray-700"
                    >
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                            {/* Gross Revenue */}
                            <div className="bg-gradient-to-br from-green-500 to-emerald-600 dark:from-green-600 dark:to-emerald-700 rounded-xl shadow-lg p-6 text-white hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
                                <div className="flex justify-between items-center mb-4">
                                    <h3 className="text-lg font-semibold">
                                        Gross Revenue
                                    </h3>
                                    <DollarSign className="h-7 w-7 text-green-200" />
                                </div>
                                <p className="text-2xl font-bold mb-2">
                                    {formatCurrency(grossRevenue)}
                                </p>
                                <p className="text-sm opacity-80">
                                    Total revenue before refunds
                                </p>
                            </div>

                            {/* Total Refunded */}
                            <div className="bg-gradient-to-br from-orange-500 to-red-600 dark:from-orange-600 dark:to-red-700 rounded-xl shadow-lg p-6 text-white hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
                                <div className="flex justify-between items-center mb-4">
                                    <h3 className="text-lg font-semibold">
                                        Total Refunded
                                    </h3>
                                    <Repeat className="h-7 w-7 text-orange-200" />
                                </div>
                                <p className="text-2xl font-bold mb-2">
                                    {formatCurrency(refundStats.totalRefunded)}
                                </p>
                                <p className="text-sm opacity-80">
                                    {refundStats.refundedOrdersCount} orders
                                    affected
                                </p>
                            </div>

                            {/* Net Revenue */}
                            <div className="bg-gradient-to-br from-blue-500 to-indigo-600 dark:from-blue-600 dark:to-indigo-700 rounded-xl shadow-lg p-6 text-white hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
                                <div className="flex justify-between items-center mb-4">
                                    <h3 className="text-lg font-semibold">
                                        Net Revenue
                                    </h3>
                                    <TrendingDown className="h-7 w-7 text-blue-200" />
                                </div>
                                <p className="text-2xl font-bold mb-2">
                                    {formatCurrency(totalRevenue)}
                                </p>
                                <div className="flex items-center">
                                    <p className="text-sm opacity-80 flex items-center">
                                        {grossRevenue > 0
                                            ? (
                                                  ((grossRevenue -
                                                      totalRevenue) /
                                                      grossRevenue) *
                                                  100
                                              ).toFixed(1) + "% "
                                            : "0% "}
                                        <span>reduction due to refunds</span>
                                    </p>
                                </div>
                            </div>

                            {/* Refund Rate */}
                            <div className="bg-gradient-to-br from-purple-500 to-pink-600 dark:from-purple-600 dark:to-pink-700 rounded-xl shadow-lg p-6 text-white hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
                                <div className="flex justify-between items-center mb-4">
                                    <h3 className="text-lg font-semibold">
                                        Refund Rate
                                    </h3>
                                    <div className="h-8 w-8 rounded-full bg-purple-400 bg-opacity-30 flex items-center justify-center">
                                        {refundStats.refundRate > 5 ? (
                                            <ArrowUp className="h-5 w-5 text-red-200" />
                                        ) : (
                                            <ArrowDown className="h-5 w-5 text-green-200" />
                                        )}
                                    </div>
                                </div>
                                <p className="text-2xl font-bold mb-2">
                                    {refundStats.refundRate}%
                                </p>
                                <div className="flex flex-col space-y-1 text-xs">
                                    <div className="flex justify-between">
                                        <span>Full Refunds:</span>
                                        <span className="font-medium">
                                            {refundStats.fullyRefundedCount}
                                        </span>
                                    </div>
                                    <div className="flex justify-between">
                                        <span>Partial Refunds:</span>
                                        <span className="font-medium">
                                            {refundStats.partiallyRefundedCount}
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </SectionContainer>

                    <SectionContainer
                        className="hidden md:block mb-8 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 shadow-sm"
                        title="Client Orders"
                        titleClassName="text-lg font-semibold text-gray-900 dark:text-white border-b border-gray-200 dark:border-gray-700"
                    >
                        <div className="hidden md:flex gap-4">
                            <div className="flex flex-col lg:flex-row gap-6 w-full">
                                {/* Left side - Client table */}
                                <div className="w-full ">
                                    <div className="bg-white dark:bg-gray-800 rounded-xl overflow-hidden shadow-lg border border-gray-200 dark:border-gray-700">
                                        <table className="min-w-full">
                                            <thead className="bg-gradient-to-r from-blue-600 to-cyan-600 dark:from-blue-700 dark:to-cyan-700">
                                                <tr>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">
                                                        <div className="flex items-center gap-2">
                                                            <Users
                                                                size={16}
                                                                className="text-blue-100"
                                                            />
                                                            Client ID
                                                        </div>
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">
                                                        <div className="flex items-center gap-2">
                                                            <Users
                                                                size={16}
                                                                className="text-blue-100"
                                                            />
                                                            Client Name
                                                        </div>
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">
                                                        <div className="flex items-center gap-2">
                                                            <Mail
                                                                size={16}
                                                                className="text-blue-100"
                                                            />
                                                            Email
                                                        </div>
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-white uppercase tracking-wider">
                                                        <div className="flex items-center gap-2">
                                                            <FileText
                                                                size={16}
                                                                className="text-blue-100"
                                                            />
                                                            Orders Count
                                                        </div>
                                                    </th>
                                                </tr>
                                            </thead>
                                            <tbody className="divide-y divide-gray-200 dark:divide-gray-600">
                                                {clients.data.map((client) => (
                                                    <tr
                                                        key={client.id}
                                                        className="hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                                    >
                                                        <td className="px-6 py-4 whitespace-nowrap dark:text-gray-200">
                                                            {client.id}
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap dark:text-gray-200">
                                                            {client.name}
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap dark:text-gray-200">
                                                            {client.email}
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap dark:text-gray-200">
                                                            <Link
                                                                href={route(
                                                                    "admin.user-orders",
                                                                    client.id
                                                                )}
                                                                className="inline-block bg-gradient-to-r from-blue-600 to-cyan-600 hover:from-blue-700 hover:to-cyan-700 text-white px-3 py-1 rounded-full text-sm font-medium transition-all duration-300 transform hover:scale-105 shadow-md"
                                                            >
                                                                {
                                                                    client.orders_count
                                                                }{" "}
                                                                Orders
                                                            </Link>
                                                        </td>
                                                    </tr>
                                                ))}
                                            </tbody>
                                        </table>
                                    </div>
                                    <div className="mb-4 mt-4">
                                        <CleanPagination
                                            links={clients.links}
                                            total={clients.total}
                                            perPage={clients.per_page}
                                            currentPage={clients.current_page}
                                        />
                                    </div>
                                </div>
                            </div>
                        </div>
                    </SectionContainer>

                    {/* Responsive Layout for Small Screens */}
                    <SectionContainer
                        className="md:hidden mb-8 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 shadow-sm"
                        title="Client Orders"
                        titleClassName="text-lg font-semibold text-gray-900 dark:text-white border-b border-gray-200 dark:border-gray-700"
                    >
                        <div className="space-y-4">
                            {clients.data.map((client) => (
                                <div
                                    key={client.id}
                                    className="bg-white dark:bg-gray-800 p-4 rounded-xl shadow border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow"
                                >
                                    <div className="flex justify-between items-center mb-2">
                                        <div className="font-semibold text-gray-900 dark:text-gray-100">
                                            {client.name}
                                        </div>
                                        <Link
                                            href={route(
                                                "admin.user-orders",
                                                client.id
                                            )}
                                            className="inline-block bg-gradient-to-r from-blue-600 to-cyan-600 hover:from-blue-700 hover:to-cyan-700 text-white px-3 py-1 rounded-full text-sm font-medium transition-all duration-300 transform hover:scale-105 shadow-md"
                                        >
                                            {client.orders_count} Orders
                                        </Link>
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-400">
                                        {client.email}
                                    </div>
                                </div>
                            ))}
                            <div className="mt-4">
                                <CleanPagination
                                    links={clients.links}
                                    total={clients.total}
                                    perPage={clients.per_page}
                                    currentPage={clients.current_page}
                                />
                            </div>
                        </div>
                    </SectionContainer>

                    {/* Revenue Analytics Section */}
                    <SectionContainer
                        className="mb-8 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 shadow-sm"
                        title="Revenue Analytics"
                        titleClassName="text-lg font-semibold text-gray-900 dark:text-white border-b border-gray-200 dark:border-gray-700"
                    >
                        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                            {/* Monthly Revenue Chart */}
                            <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow border border-gray-200 dark:border-gray-700">
                                <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-white">
                                    Monthly Revenue & Orders
                                </h3>
                                <div className="h-80">
                                    <DashboardCharts.MonthlyRevenueChart
                                        data={chartData}
                                    />
                                </div>
                                <div className="mt-6 flex justify-around">
                                    <div className="text-center">
                                        <span className="block text-sm text-gray-500 dark:text-gray-400">
                                            Gross Revenue
                                        </span>
                                        <span className="text-lg font-semibold text-green-600 dark:text-green-400">
                                            {formatCurrency(grossRevenue)}
                                        </span>
                                    </div>
                                    <div className="text-center">
                                        <span className="block text-sm text-gray-500 dark:text-gray-400">
                                            Refunded
                                        </span>
                                        <span className="text-lg font-semibold text-red-600 dark:text-red-400">
                                            {formatCurrency(
                                                refundStats.totalRefunded
                                            )}
                                        </span>
                                    </div>
                                    <div className="text-center">
                                        <span className="block text-sm text-gray-500 dark:text-gray-400">
                                            Net Revenue
                                        </span>
                                        <span className="text-lg font-semibold text-blue-600 dark:text-blue-400">
                                            {formatCurrency(totalRevenue)}
                                        </span>
                                    </div>
                                </div>
                            </div>

                            {/* Refund Trends Chart */}
                            <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow border border-gray-200 dark:border-gray-700">
                                <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-white">
                                    Refund Trends
                                </h3>
                                <div className="h-80">
                                    <DashboardCharts.RefundTrendsChart
                                        data={chartData}
                                    />
                                </div>
                                <div className="mt-6 grid grid-cols-3 gap-4">
                                    <div className="text-center p-3 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
                                        <span className="block text-sm text-gray-500 dark:text-gray-400">
                                            Full Refunds
                                        </span>
                                        <span className="text-lg font-semibold text-gray-900 dark:text-gray-100">
                                            {refundStats.fullyRefundedCount}
                                        </span>
                                    </div>
                                    <div className="text-center p-3 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
                                        <span className="block text-sm text-gray-500 dark:text-gray-400">
                                            Partial Refunds
                                        </span>
                                        <span className="text-lg font-semibold text-gray-900 dark:text-gray-100">
                                            {refundStats.partiallyRefundedCount}
                                        </span>
                                    </div>
                                    <div className="text-center p-3 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
                                        <span className="block text-sm text-gray-500 dark:text-gray-400">
                                            Refund Rate
                                        </span>
                                        <span className="text-lg font-semibold text-gray-900 dark:text-gray-100">
                                            {refundStats.refundRate}%
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </SectionContainer>

                    {/* Revenue Comparison Chart */}
                    <SectionContainer
                        className="mb-8 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 shadow-sm"
                        title="Revenue Comparison"
                        titleClassName="text-lg font-semibold text-gray-900 dark:text-white border-b border-gray-200 dark:border-gray-700"
                    >
                        <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow border border-gray-200 dark:border-gray-700">
                            {/* <div className="h-96">
                                <DashboardCharts.RevenueComparisonChart
                                    data={chartData}
                                    grossRevenue={grossRevenue}
                                    netRevenue={totalRevenue}
                                    refundedAmount={refundStats.totalRefunded}
                                />
                            </div> */}
                            <div className="mt-4 grid grid-cols-1 md:grid-cols-3 gap-6">
                                <div className="bg-gradient-to-r from-green-50 to-emerald-50 dark:from-green-900/20 dark:to-emerald-900/20 p-4 rounded-xl border border-green-200 dark:border-green-800 shadow-sm">
                                    <h4 className="text-sm font-medium text-green-800 dark:text-green-300 mb-2">
                                        Gross Revenue
                                    </h4>
                                    <p className="text-2xl font-bold text-green-700 dark:text-green-400">
                                        {formatCurrency(grossRevenue)}
                                    </p>
                                    <p className="text-xs text-green-600 dark:text-green-500 mt-2">
                                        Total income before refunds
                                    </p>
                                </div>
                                <div className="bg-gradient-to-r from-red-50 to-pink-50 dark:from-red-900/20 dark:to-pink-900/20 p-4 rounded-xl border border-red-200 dark:border-red-800 shadow-sm">
                                    <h4 className="text-sm font-medium text-red-800 dark:text-red-300 mb-2">
                                        Refunded Amount
                                    </h4>
                                    <p className="text-2xl font-bold text-red-700 dark:text-red-400">
                                        {formatCurrency(
                                            refundStats.totalRefunded
                                        )}
                                    </p>
                                    <p className="text-xs text-red-600 dark:text-red-500 mt-2">
                                        {refundStats.refundedOrdersCount} orders
                                        refunded ({refundStats.refundRate}%)
                                    </p>
                                </div>
                                <div className="bg-gradient-to-r from-blue-50 to-cyan-50 dark:from-blue-900/20 dark:to-cyan-900/20 p-4 rounded-xl border border-blue-200 dark:border-blue-800 shadow-sm">
                                    <h4 className="text-sm font-medium text-blue-800 dark:text-blue-300 mb-2">
                                        Net Revenue
                                    </h4>
                                    <p className="text-2xl font-bold text-blue-700 dark:text-blue-400">
                                        {formatCurrency(totalRevenue)}
                                    </p>
                                    <p className="text-xs text-blue-600 dark:text-blue-500 mt-2">
                                        Actual income after refunds
                                    </p>
                                </div>
                            </div>
                        </div>
                    </SectionContainer>

                    {/* Top Clients by Orders */}
                    <SectionContainer
                        className="mb-8 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 shadow-sm"
                        title="Top Clients by Orders"
                        titleClassName="text-lg font-semibold text-gray-900 dark:text-white border-b border-gray-200 dark:border-gray-700"
                    >
                        <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow border border-gray-200 dark:border-gray-700">
                            <div className="h-80">
                                <DashboardCharts.ClientsOrdersChart
                                    data={clientsOrders}
                                />
                            </div>
                        </div>
                    </SectionContainer>

                   
                </div>
            </div>
        </AdminLayout>
    );
}
