import { Head, Link, useForm, router, usePage } from "@inertiajs/react";
import React, { useState } from "react";
import { Plus, Edit, Trash2, Eye, Power, PowerOff, Search, Filter, CheckCircle, XCircle } from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import { Tooltip } from "@/Components/ui/Tooltip";
import CleanPagination from "@/Components/CleanPagination";

export default function Index({ defaultCoupons }) {
    const { flash } = usePage().props;
    const { data, setData, get } = useForm({
        search: "",
        status: "",
    });

    const [selectedCoupons, setSelectedCoupons] = useState([]);

    const handleSearch = (e) => {
        e.preventDefault();
        get(route("admin.default-coupons.index"), {
            preserveScroll: true,
        });
    };

    const handleBulkToggleStatus = () => {
        if (selectedCoupons.length === 0) return;

        router.put(route("admin.coupons.bulk-toggle-status"), {
            coupon_ids: selectedCoupons,
        });
    };

    const handleBulkDelete = () => {
        if (selectedCoupons.length === 0) return;

        if (confirm("Are you sure you want to delete the selected coupons?")) {
            router.delete(route("admin.coupons.bulk-delete"), {
                data: { coupon_ids: selectedCoupons },
            });
        }
    };

    const toggleCouponSelection = (couponId) => {
        setSelectedCoupons((prev) =>
            prev.includes(couponId)
                ? prev.filter((id) => id !== couponId)
                : [...prev, couponId]
        );
    };

    const toggleAllCoupons = () => {
        if (selectedCoupons.length === defaultCoupons.data.length) {
            setSelectedCoupons([]);
        } else {
            setSelectedCoupons(defaultCoupons.data.map((coupon) => coupon.id));
        }
    };

    const formatCurrency = (amount, type) => {
        if (type === "percentage") {
            return `${amount}%`;
        }
        return `$${parseFloat(amount).toFixed(2)}`;
    };

    const getStatusBadge = (coupon) => {
        if (coupon.is_active) {
            return (
                <Badge variant="success" className="flex items-center gap-1">
                    <Power size={12} />
                    Active
                </Badge>
            );
        }
        return (
            <Badge variant="secondary" className="flex items-center gap-1">
                <PowerOff size={12} />
                Inactive
            </Badge>
        );
    };

    return (
        <AdminLayout>
            <Head title="Default Coupons" />

            <div className="py-6">
                <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
                    {/* Flash Messages */}
                    {flash?.success && (
                        <div className="mb-4 rounded-lg bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 p-4">
                            <div className="flex items-center">
                                <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400 mr-3" />
                                <div>
                                    <p className="text-sm font-medium text-green-800 dark:text-green-200">
                                        {flash.success}
                                    </p>
                                    {flash.message && (
                                        <p className="text-sm text-green-700 dark:text-green-300 mt-1">
                                            {flash.message}
                                        </p>
                                    )}
                                </div>
                            </div>
                        </div>
                    )}

                    {flash?.error && (
                        <div className="mb-4 rounded-lg bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 p-4">
                            <div className="flex items-center">
                                <XCircle className="w-5 h-5 text-red-600 dark:text-red-400 mr-3" />
                                <div>
                                    <p className="text-sm font-medium text-red-800 dark:text-red-200">
                                        {flash.error}
                                    </p>
                                    {flash.message && (
                                        <p className="text-sm text-red-700 dark:text-red-300 mt-1">
                                            {flash.message}
                                        </p>
                                    )}
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
                            <div>
                                <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                    Default Coupons
                                </h1>
                                <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                                    Manage default discount coupons that are available to all users
                                </p>
                            </div>
                            <div className="mt-4 sm:mt-0">
                                <Link
                                    href={route("admin.default-coupons.create")}
                                    className="inline-flex items-center px-4 py-2 bg-gradient-to-r from-blue-600 to-cyan-500 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:from-blue-700 hover:to-cyan-600 active:bg-blue-900 focus:outline-none focus:border-blue-900 focus:ring ring-blue-300 disabled:opacity-25 transition ease-in-out duration-150"
                                >
                                    <Plus className="w-4 h-4 mr-2" />
                                    Create Default Coupon
                                </Link>
                            </div>
                        </div>
                    </div>

                    {/* Filters and Search */}
                    <div className="mb-6">
                        <form onSubmit={handleSearch} className="flex flex-col sm:flex-row gap-4">
                            <div className="flex-1">
                                <div className="relative">
                                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-5 w-5" />
                                    <input
                                        type="text"
                                        placeholder="Search coupons..."
                                        className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-500 dark:placeholder-gray-400"
                                        value={data.search}
                                        onChange={(e) => setData("search", e.target.value)}
                                    />
                                </div>
                            </div>
                            <div className="flex gap-2">
                                <select
                                    className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                    value={data.status}
                                    onChange={(e) => setData("status", e.target.value)}
                                >
                                    <option value="">All Status</option>
                                    <option value="active">Active</option>
                                    <option value="inactive">Inactive</option>
                                </select>
                                <Button type="submit" variant="primary">
                                    <Filter className="w-4 h-4 mr-2" />
                                    Filter
                                </Button>
                            </div>
                        </form>
                    </div>

                    {/* Bulk Actions */}
                    {selectedCoupons.length > 0 && (
                        <div className="mb-4 p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
                            <div className="flex items-center justify-between">
                                <span className="text-sm text-blue-800 dark:text-blue-200">
                                    {selectedCoupons.length} coupon(s) selected
                                </span>
                                <div className="flex gap-2">
                                    <Button
                                        onClick={handleBulkToggleStatus}
                                        variant="secondary"
                                        size="sm"
                                    >
                                        <Power className="w-4 h-4 mr-2" />
                                        Toggle Status
                                    </Button>
                                    <Button
                                        onClick={handleBulkDelete}
                                        variant="danger"
                                        size="sm"
                                    >
                                        <Trash2 className="w-4 h-4 mr-2" />
                                        Delete
                                    </Button>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Coupons Table */}
                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                        <div className="overflow-x-auto">
                            <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                <thead className="bg-gray-50 dark:bg-gray-700">
                                    <tr>
                                        <th className="px-6 py-3 text-left">
                                            <input
                                                type="checkbox"
                                                checked={selectedCoupons.length === defaultCoupons.data.length}
                                                onChange={toggleAllCoupons}
                                                className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                            />
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Coupon Code
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Discount
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Status
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Uses
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Created
                                        </th>
                                        <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Actions
                                        </th>
                                    </tr>
                                </thead>
                                <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                    {defaultCoupons.data.map((coupon) => (
                                        <tr key={coupon.id} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <input
                                                    type="checkbox"
                                                    checked={selectedCoupons.includes(coupon.id)}
                                                    onChange={() => toggleCouponSelection(coupon.id)}
                                                    className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                />
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                    {coupon.coupon_code}
                                                </div>
                                                {coupon.description && (
                                                    <div className="text-sm text-gray-500 dark:text-gray-400">
                                                        {coupon.description}
                                                    </div>
                                                )}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <Badge variant="primary">
                                                    {formatCurrency(coupon.discount_amount, coupon.discount_type)}
                                                </Badge>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                {getStatusBadge(coupon)}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                {coupon.uses_count}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                {new Date(coupon.created_at).toLocaleDateString()}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                                <div className="flex items-center justify-end gap-2">
                                                    <Tooltip content="View Details">
                                                        <Link
                                                            href={route("admin.default-coupons.show", coupon.id)}
                                                            className="text-blue-600 hover:text-blue-900 dark:text-blue-400 dark:hover:text-blue-300"
                                                        >
                                                            <Eye className="w-4 h-4" />
                                                        </Link>
                                                    </Tooltip>
                                                    <Tooltip content="Edit Coupon">
                                                        <Link
                                                            href={route("admin.default-coupons.edit", coupon.id)}
                                                            className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300"
                                                        >
                                                            <Edit className="w-4 h-4" />
                                                        </Link>
                                                    </Tooltip>
                                                    <Tooltip content="Delete Coupon">
                                                        <button
                                                            onClick={() => {
                                                                if (confirm("Are you sure you want to delete this coupon?")) {
                                                                    router.delete(route("admin.default-coupons.destroy", coupon.id));
                                                                }
                                                            }}
                                                            className="text-red-600 hover:text-red-900 dark:text-red-400 dark:hover:text-red-300"
                                                        >
                                                            <Trash2 className="w-4 h-4" />
                                                        </button>
                                                    </Tooltip>
                                                </div>
                                            </td>
                                        </tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>

                        {defaultCoupons.data.length === 0 && (
                            <div className="text-center py-12">
                                <div className="text-gray-500 dark:text-gray-400">
                                    <Filter className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-white">
                                        No default coupons found
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        Get started by creating a new default coupon.
                                    </p>
                                    <div className="mt-6">
                                        <Link
                                            href={route("admin.default-coupons.create")}
                                            className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                        >
                                            <Plus className="w-4 h-4 mr-2" />
                                            Create Default Coupon
                                        </Link>
                                    </div>
                                </div>
                            </div>
                        )}
                    </div>

                    {/* Pagination */}
                    {defaultCoupons.data.length > 0 && (
                        <div className="mt-6">
                            <CleanPagination
                                links={defaultCoupons.links}
                                total={defaultCoupons.total}
                                perPage={defaultCoupons.per_page}
                                currentPage={defaultCoupons.current_page}
                            />
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
} 