import { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import InputLabel from "@/Components/InputLabel";
import TextInput from "@/Components/TextInput";
import InputError from "@/Components/InputError";
import { TextArea } from "@/Components/TextArea";
import PrimaryButton from "@/Components/PrimaryButton";
import { ArrowLeft, Save, Plus, X } from "lucide-react";

export default function Create({ categories }) {
    const [newCategory, setNewCategory] = useState("");
    const [showNewCategoryInput, setShowNewCategoryInput] = useState(false);

    const { data, setData, post, processing, errors } = useForm({
        title: "",
        description: "",
        category: "",
        is_active: true,
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.essay-topics.store"));
    };

    const addNewCategory = () => {
        if (newCategory.trim()) {
            setData("category", newCategory.trim());
            setNewCategory("");
            setShowNewCategoryInput(false);
        }
    };

    return (
        <AdminLayout>
            <Head title="Create Essay Topic" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex items-center justify-between mb-6">
                        <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                            Create Essay Topic
                        </h1>
                        <Link
                            href={route("admin.essay-topics.index")}
                            className="px-4 py-2 text-sm text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-800 transition-colors flex items-center"
                        >
                            <ArrowLeft className="h-4 w-4 mr-1" />
                            Back to Topics
                        </Link>
                    </div>

                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <form onSubmit={handleSubmit} className="p-6 space-y-6">
                            <div>
                                <InputLabel
                                    htmlFor="title"
                                    value="Title"
                                    className="text-gray-900 dark:text-gray-100"
                                />
                                <TextInput
                                    id="title"
                                    type="text"
                                    className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                    value={data.title}
                                    onChange={(e) =>
                                        setData("title", e.target.value)
                                    }
                                    required
                                />
                                <InputError
                                    message={errors.title}
                                    className="mt-2"
                                />
                            </div>

                            <div>
                                <InputLabel
                                    htmlFor="description"
                                    value="Description"
                                    className="text-gray-900 dark:text-gray-100"
                                />
                                <TextArea
                                    id="description"
                                    className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                    value={data.description}
                                    onChange={(e) =>
                                        setData("description", e.target.value)
                                    }
                                    rows={6}
                                    required
                                />
                                <InputError
                                    message={errors.description}
                                    className="mt-2"
                                />
                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                    Provide detailed instructions for the essay.
                                    Include any specific requirements, word
                                    count, formatting guidelines, etc.
                                </p>
                            </div>

                            <div>
                                <InputLabel
                                    htmlFor="category"
                                    value="Category (Optional)"
                                    className="text-gray-900 dark:text-gray-100"
                                />
                                {showNewCategoryInput ? (
                                    <div className="flex mt-1">
                                        <TextInput
                                            id="new-category"
                                            type="text"
                                            className="block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                            value={newCategory}
                                            onChange={(e) =>
                                                setNewCategory(e.target.value)
                                            }
                                            placeholder="Enter new category"
                                            autoFocus
                                        />
                                        <button
                                            type="button"
                                            onClick={addNewCategory}
                                            className="ml-2 px-3 py-2 bg-green-600 dark:bg-green-500 text-white rounded-md hover:bg-green-700 dark:hover:bg-green-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            <Save className="h-5 w-5" />
                                        </button>
                                        <button
                                            type="button"
                                            onClick={() =>
                                                setShowNewCategoryInput(false)
                                            }
                                            className="ml-2 px-3 py-2 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 rounded-md hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            <X className="h-5 w-5" />
                                        </button>
                                    </div>
                                ) : (
                                    <div className="flex mt-1">
                                        <select
                                            id="category"
                                            className="block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                            value={data.category}
                                            onChange={(e) =>
                                                setData(
                                                    "category",
                                                    e.target.value
                                                )
                                            }
                                        >
                                            <option value="">
                                                Select a category (optional)
                                            </option>
                                            {categories.map((category) => (
                                                <option
                                                    key={category}
                                                    value={category}
                                                >
                                                    {category}
                                                </option>
                                            ))}
                                        </select>
                                        <button
                                            type="button"
                                            onClick={() =>
                                                setShowNewCategoryInput(true)
                                            }
                                            className="ml-2 px-3 py-2 bg-blue-600 dark:bg-blue-500 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            title="Add new category"
                                        >
                                            <Plus className="h-5 w-5" />
                                        </button>
                                    </div>
                                )}
                                <InputError
                                    message={errors.category}
                                    className="mt-2"
                                />
                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                    Categorize the essay topic (e.g., Academic
                                    Writing, Creative Writing, Research)
                                </p>
                            </div>

                            <div className="flex items-center">
                                <input
                                    id="is_active"
                                    type="checkbox"
                                    className="rounded border-gray-300 dark:border-gray-700 text-indigo-600 shadow-sm focus:ring-indigo-500 dark:focus:ring-indigo-600"
                                    checked={data.is_active}
                                    onChange={(e) =>
                                        setData("is_active", e.target.checked)
                                    }
                                />
                                <label
                                    htmlFor="is_active"
                                    className="ml-2 text-sm text-gray-700 dark:text-gray-300"
                                >
                                    Active (available for writer tests)
                                </label>
                            </div>

                            <div className="flex items-center justify-end mt-6">
                                <Link
                                    href={route("admin.essay-topics.index")}
                                    className="px-4 py-2 text-sm text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-800 transition-colors"
                                >
                                    Cancel
                                </Link>
                                <PrimaryButton
                                    className="ml-4"
                                    disabled={processing}
                                >
                                    <Save className="h-4 w-4 mr-2" />
                                    Save Topic
                                </PrimaryButton>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
