import React, { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Award,
    ArrowLeft,
    Save,
    X,
    HelpCircle,
    AlertTriangle,
    Info,
} from "lucide-react";
import { Button } from "@/Components/ui/Button";
import { FormInput } from "@/Components/ui/FormInput";
import { Textarea } from "@/Components/ui/Textarea";
import Alert from "@/Components/ui/Alert";
import { Label } from "@/Components/ui/Label";
import { Switch } from "@/Components/ui/Switch";
import {
    Tooltip,
    TooltipContent,
    TooltipProvider,
    TooltipTrigger,
} from "@/Components/ui/Tooltip";
import {
    Card,
    CardContent,
    CardHeader,
    CardTitle,
    CardDescription,
} from "@/Components/ui/Card";

const Edit = ({ loyaltyTier, nextTier, previousTier, couponCount }) => {
    const { data, setData, patch, processing, errors } = useForm({
        name: loyaltyTier.name || "",
        min_orders: loyaltyTier.min_orders || 1,
        discount_percentage: loyaltyTier.discount_percentage || 5,
        description: loyaltyTier.description || "",
        is_active: loyaltyTier.is_active,
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        patch(route("admin.loyalty-tiers.update", loyaltyTier.id));
    };

    // Min and max values for order count based on adjacent tiers
    const minOrderValue = previousTier ? previousTier.min_orders + 1 : 1;
    const maxOrderValue = nextTier ? nextTier.min_orders - 1 : null;

    return (
        <AdminLayout>
            <Head title={`Edit ${loyaltyTier.name}`} />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Award className="h-6 w-6 mr-2 text-blue-500" />
                                Edit Loyalty Tier
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Update the details for "{loyaltyTier.name}"
                                loyalty tier
                            </p>
                        </div>
                        <div className="flex flex-wrap gap-3">
                            <Button
                                variant="outline"
                                className="inline-flex items-center"
                                asChild
                            >
                                <Link href={route("admin.loyalty-tiers.index")}>
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Tiers
                                </Link>
                            </Button>
                            <Button
                                className="inline-flex items-center"
                                asChild
                            >
                                <Link
                                    href={route(
                                        "admin.loyalty-tiers.show",
                                        loyaltyTier.id
                                    )}
                                >
                                    <Award className="w-4 h-4 mr-2" />
                                    View Details
                                </Link>
                            </Button>
                        </div>
                    </div>

                    {errors.general && (
                        <Alert
                            className="mt-6"
                            variant="destructive"
                            message={errors.general}
                        />
                    )}

                    {couponCount > 0 && (
                        <Alert
                            className="mt-6"
                            variant="warning"
                            title="Active Coupons"
                            message={`This tier has ${couponCount} coupon${
                                couponCount !== 1 ? "s" : ""
                            } associated with it. Changes to this tier will affect these coupons.`}
                        />
                    )}

                    <Card className="mt-6">
                        <CardHeader>
                            <CardTitle>Edit {loyaltyTier.name}</CardTitle>
                            <CardDescription>
                                Update the loyalty tier details below
                            </CardDescription>
                        </CardHeader>
                        <CardContent>
                            <form onSubmit={handleSubmit} className="space-y-6">
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div className="space-y-4">
                                        <FormInput
                                            id="name"
                                            name="name"
                                            label="Tier Name"
                                            value={data.name}
                                            onChange={(e) =>
                                                setData("name", e.target.value)
                                            }
                                            error={errors.name}
                                            placeholder="e.g., Silver, Gold, Platinum"
                                            required
                                        />

                                        <div>
                                            <div className="flex items-center gap-2">
                                                <Label
                                                    htmlFor="min_orders"
                                                    className="text-sm font-medium"
                                                >
                                                    Minimum Orders
                                                </Label>
                                                <TooltipProvider>
                                                    <Tooltip>
                                                        <TooltipTrigger asChild>
                                                            <span>
                                                                <HelpCircle className="h-4 w-4 text-gray-400" />
                                                            </span>
                                                        </TooltipTrigger>
                                                        <TooltipContent>
                                                            <p className="w-[250px]">
                                                                The minimum
                                                                number of orders
                                                                a customer must
                                                                place to qualify
                                                                for this tier.
                                                                {previousTier && (
                                                                    <span>
                                                                        {" "}
                                                                        Must be
                                                                        greater
                                                                        than{" "}
                                                                        {
                                                                            previousTier.min_orders
                                                                        }{" "}
                                                                        (the
                                                                        previous
                                                                        tier).
                                                                    </span>
                                                                )}
                                                                {nextTier && (
                                                                    <span>
                                                                        {" "}
                                                                        Must be
                                                                        less
                                                                        than{" "}
                                                                        {
                                                                            nextTier.min_orders
                                                                        }{" "}
                                                                        (the
                                                                        next
                                                                        tier).
                                                                    </span>
                                                                )}
                                                            </p>
                                                        </TooltipContent>
                                                    </Tooltip>
                                                </TooltipProvider>
                                            </div>
                                            <div className="mt-1">
                                                <input
                                                    type="number"
                                                    id="min_orders"
                                                    name="min_orders"
                                                    min={minOrderValue}
                                                    max={maxOrderValue}
                                                    step="1"
                                                    value={data.min_orders}
                                                    onChange={(e) =>
                                                        setData(
                                                            "min_orders",
                                                            parseInt(
                                                                e.target.value
                                                            ) || minOrderValue
                                                        )
                                                    }
                                                    className={`w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 ${
                                                        errors.min_orders
                                                            ? "border-red-500"
                                                            : ""
                                                    }`}
                                                    required
                                                />
                                            </div>
                                            {errors.min_orders && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.min_orders}
                                                </p>
                                            )}
                                            {previousTier && (
                                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                    Previous tier (
                                                    {previousTier.name}):{" "}
                                                    {previousTier.min_orders}{" "}
                                                    orders
                                                </p>
                                            )}
                                            {nextTier && (
                                                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                    Next tier ({nextTier.name}):{" "}
                                                    {nextTier.min_orders} orders
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <div className="flex items-center gap-2">
                                                <Label
                                                    htmlFor="discount_percentage"
                                                    className="text-sm font-medium"
                                                >
                                                    Discount Percentage
                                                </Label>
                                                <TooltipProvider>
                                                    <Tooltip>
                                                        <TooltipTrigger asChild>
                                                            <span>
                                                                <HelpCircle className="h-4 w-4 text-gray-400" />
                                                            </span>
                                                        </TooltipTrigger>
                                                        <TooltipContent>
                                                            <p className="w-[200px]">
                                                                The percentage
                                                                discount users
                                                                in this tier
                                                                will receive.
                                                            </p>
                                                        </TooltipContent>
                                                    </Tooltip>
                                                </TooltipProvider>
                                            </div>
                                            <div className="mt-1 relative">
                                                <input
                                                    type="number"
                                                    id="discount_percentage"
                                                    name="discount_percentage"
                                                    min="0"
                                                    max="100"
                                                    step="0.1"
                                                    value={
                                                        data.discount_percentage
                                                    }
                                                    onChange={(e) =>
                                                        setData(
                                                            "discount_percentage",
                                                            parseFloat(
                                                                e.target.value
                                                            ) || 0
                                                        )
                                                    }
                                                    className={`w-full pr-8 rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 ${
                                                        errors.discount_percentage
                                                            ? "border-red-500"
                                                            : ""
                                                    }`}
                                                    required
                                                />
                                                <div className="absolute inset-y-0 right-0 flex items-center px-3 pointer-events-none">
                                                    <span className="text-gray-500 dark:text-gray-400">
                                                        %
                                                    </span>
                                                </div>
                                            </div>
                                            {errors.discount_percentage && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.discount_percentage}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    <div className="space-y-4">
                                        <div>
                                            <Label
                                                htmlFor="description"
                                                className="text-sm font-medium"
                                            >
                                                Description
                                            </Label>
                                            <div className="mt-1">
                                                <Textarea
                                                    id="description"
                                                    name="description"
                                                    rows={5}
                                                    value={
                                                        data.description || ""
                                                    }
                                                    onChange={(e) =>
                                                        setData(
                                                            "description",
                                                            e.target.value
                                                        )
                                                    }
                                                    className={`w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 ${
                                                        errors.description
                                                            ? "border-red-500"
                                                            : ""
                                                    }`}
                                                    placeholder="Explain the benefits of this tier for customers..."
                                                />
                                            </div>
                                            {errors.description && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.description}
                                                </p>
                                            )}
                                        </div>

                                        <div className="flex items-center justify-between">
                                            <div className="flex items-center gap-2">
                                                <Label
                                                    htmlFor="is_active"
                                                    className="text-sm font-medium"
                                                >
                                                    Active Status
                                                </Label>
                                                <TooltipProvider>
                                                    <Tooltip>
                                                        <TooltipTrigger asChild>
                                                            <span>
                                                                <HelpCircle className="h-4 w-4 text-gray-400" />
                                                            </span>
                                                        </TooltipTrigger>
                                                        <TooltipContent>
                                                            <p className="w-[200px]">
                                                                When active,
                                                                this tier will
                                                                be used for
                                                                determining
                                                                customer loyalty
                                                                discounts.
                                                            </p>
                                                        </TooltipContent>
                                                    </Tooltip>
                                                </TooltipProvider>
                                            </div>
                                            <Switch
                                                id="is_active"
                                                checked={data.is_active}
                                                onCheckedChange={(value) =>
                                                    setData("is_active", value)
                                                }
                                            />
                                        </div>

                                        {couponCount > 0 && (
                                            <div className="bg-amber-50 dark:bg-amber-900/20 rounded-lg p-4 mt-4">
                                                <div className="flex">
                                                    <AlertTriangle className="h-5 w-5 text-amber-600 dark:text-amber-400 mr-2 flex-shrink-0" />
                                                    <div className="text-sm text-amber-700 dark:text-amber-300">
                                                        <p className="font-medium">
                                                            Associated Coupons
                                                        </p>
                                                        <p className="mt-1">
                                                            This tier is
                                                            currently used by{" "}
                                                            {couponCount} coupon
                                                            {couponCount !== 1
                                                                ? "s"
                                                                : ""}
                                                            . Changing the
                                                            discount percentage
                                                            will affect future
                                                            uses of these
                                                            coupons.
                                                        </p>
                                                    </div>
                                                </div>
                                            </div>
                                        )}

                                        <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4 mt-4">
                                            <div className="flex">
                                                <Info className="h-5 w-5 text-blue-600 dark:text-blue-400 mr-2 flex-shrink-0" />
                                                <div className="text-sm text-blue-700 dark:text-blue-300">
                                                    <p className="font-medium">
                                                        Tier Hierarchy
                                                    </p>
                                                    <p className="mt-1">
                                                        Loyalty tiers are
                                                        organized by order
                                                        count. Users
                                                        automatically qualify
                                                        for the highest tier
                                                        they're eligible for
                                                        based on their order
                                                        history.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div className="flex justify-end space-x-3">
                                    <Button
                                        type="button"
                                        variant="outline"
                                        onClick={() => window.history.back()}
                                        disabled={processing}
                                        className="inline-flex items-center"
                                    >
                                        <X className="w-4 h-4 mr-2" />
                                        Cancel
                                    </Button>
                                    <Button
                                        type="submit"
                                        disabled={processing}
                                        className="inline-flex items-center"
                                    >
                                        <Save className="w-4 h-4 mr-2" />
                                        {processing
                                            ? "Saving..."
                                            : "Save Changes"}
                                    </Button>
                                </div>
                            </form>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Edit;
