import React, { useState } from 'react';
import { Head, Link } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import QualityAssessmentForm from '@/Components/Admin/QualityAssessmentForm';
import RevisionRejectionForm from '@/Components/Admin/RevisionRejectionForm';
import {
    CurrencyDollarIcon,
    ChartBarIcon,
    UserGroupIcon,
    DocumentTextIcon,
    CogIcon,
    ArrowTrendingUpIcon,
    ArrowTrendingDownIcon,
    StarIcon,
    XCircleIcon
} from '@heroicons/react/24/outline';

export default function PaymentImpactsIndex({ auth, stats, recentAssessments }) {
    const [showQualityForm, setShowQualityForm] = useState(false);
    const [showRejectionForm, setShowRejectionForm] = useState(false);
    const [selectedRevision, setSelectedRevision] = useState(null);

    const handleQualityAssessment = (revision) => {
        setSelectedRevision(revision);
        setShowQualityForm(true);
    };

    const handleRejection = (revision) => {
        setSelectedRevision(revision);
        setShowRejectionForm(true);
    };

    const handleFormClose = () => {
        setShowQualityForm(false);
        setShowRejectionForm(false);
        setSelectedRevision(null);
    };

    const handleFormSuccess = () => {
        handleFormClose();
        // Refresh the page to show updated data
        window.location.reload();
    };

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    const getQualityRatingColor = (rating) => {
        if (rating >= 8) return 'text-green-600';
        if (rating >= 6) return 'text-yellow-600';
        return 'text-red-600';
    };

    const getPaymentImpactColor = (impact) => {
        if (impact > 0) return 'text-green-600';
        if (impact < 0) return 'text-red-600';
        return 'text-gray-600';
    };

    return (
        <AdminLayout auth={auth}>
            <Head title="Payment Impact Management" />
            
            <div className="max-w-7xl mx-auto py-6">
                {/* Header */}
                <div className="bg-white shadow-sm rounded-lg mb-6">
                    <div className="px-6 py-4 border-b border-gray-200">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                                <CurrencyDollarIcon className="h-6 w-6 text-green-600" />
                                <h1 className="text-xl font-semibold text-gray-900">
                                    Payment Impact Management
                                </h1>
                            </div>
                            <div className="flex items-center space-x-3">
                                <Link
                                    href={route('admin.payment-impacts.settings')}
                                    className="inline-flex items-center px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                >
                                    <CogIcon className="h-4 w-4 mr-2" />
                                    Settings
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Statistics Cards */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                    <div className="bg-white shadow-sm rounded-lg p-6">
                        <div className="flex items-center">
                            <div className="flex-shrink-0">
                                <DocumentTextIcon className="h-8 w-8 text-blue-600" />
                            </div>
                            <div className="ml-4">
                                <p className="text-sm font-medium text-gray-500">Total Assessments</p>
                                <p className="text-2xl font-bold text-gray-900">{stats.total_revisions}</p>
                            </div>
                        </div>
                    </div>

                    <div className="bg-white shadow-sm rounded-lg p-6">
                        <div className="flex items-center">
                            <div className="flex-shrink-0">
                                <ArrowTrendingUpIcon className="h-8 w-8 text-green-600" />
                            </div>
                            <div className="ml-4">
                                <p className="text-sm font-medium text-gray-500">Total Bonuses</p>
                                <p className="text-2xl font-bold text-green-600">
                                    {formatCurrency(stats.total_bonuses)}
                                </p>
                            </div>
                        </div>
                    </div>

                    <div className="bg-white shadow-sm rounded-lg p-6">
                        <div className="flex items-center">
                            <div className="flex-shrink-0">
                                <ArrowTrendingDownIcon className="h-8 w-8 text-red-600" />
                            </div>
                            <div className="ml-4">
                                <p className="text-sm font-medium text-gray-500">Total Penalties</p>
                                <p className="text-2xl font-bold text-red-600">
                                    {formatCurrency(stats.total_penalties)}
                                </p>
                            </div>
                        </div>
                    </div>

                    <div className="bg-white shadow-sm rounded-lg p-6">
                        <div className="flex items-center">
                            <div className="flex-shrink-0">
                                <ChartBarIcon className="h-8 w-8 text-purple-600" />
                            </div>
                            <div className="ml-4">
                                <p className="text-sm font-medium text-gray-500">Net Impact</p>
                                <p className={`text-2xl font-bold ${getPaymentImpactColor(stats.net_impact)}`}>
                                    {formatCurrency(stats.net_impact)}
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {/* Quality Distribution Chart */}
                    <div className="bg-white shadow-sm rounded-lg">
                        <div className="px-6 py-4 border-b border-gray-200">
                            <h3 className="text-lg font-medium text-gray-900">Quality Rating Distribution</h3>
                        </div>
                        <div className="p-6">
                            <div className="space-y-3">
                                {Object.entries(stats.quality_distribution).map(([rating, count]) => (
                                    <div key={rating} className="flex items-center justify-between">
                                        <div className="flex items-center space-x-3">
                                            <span className={`text-sm font-medium ${getQualityRatingColor(rating)}`}>
                                                {rating}/10
                                            </span>
                                            <div className="w-32 bg-gray-200 rounded-full h-2">
                                                <div
                                                    className="bg-blue-600 h-2 rounded-full"
                                                    style={{
                                                        width: `${(count / stats.total_revisions) * 100}%`
                                                    }}
                                                />
                                            </div>
                                        </div>
                                        <span className="text-sm text-gray-500">{count}</span>
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>

                    {/* Monthly Trends */}
                    <div className="bg-white shadow-sm rounded-lg">
                        <div className="px-6 py-4 border-b border-gray-200">
                            <h3 className="text-lg font-medium text-gray-900">Monthly Trends</h3>
                        </div>
                        <div className="p-6">
                            <div className="space-y-3">
                                {stats.monthly_trends.slice(0, 6).map((trend) => (
                                    <div key={trend.month} className="flex items-center justify-between py-2 border-b border-gray-100">
                                        <span className="text-sm font-medium text-gray-900">
                                            {new Date(trend.month + '-01').toLocaleDateString('en-US', {
                                                year: 'numeric',
                                                month: 'short'
                                            })}
                                        </span>
                                        <div className="flex items-center space-x-4 text-sm">
                                            <span className="text-gray-500">
                                                {trend.count} revisions
                                            </span>
                                            <span className="text-blue-600">
                                                {trend.avg_rating?.toFixed(1)} avg
                                            </span>
                                            <span className={`font-medium ${getPaymentImpactColor(trend.total_bonus - trend.total_penalty)}`}>
                                                {formatCurrency(trend.total_bonus - trend.total_penalty)}
                                            </span>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>
                </div>

                {/* Recent Assessments */}
                <div className="bg-white shadow-sm rounded-lg mt-6">
                    <div className="px-6 py-4 border-b border-gray-200">
                        <h3 className="text-lg font-medium text-gray-900">Recent Quality Assessments</h3>
                    </div>
                    <div className="overflow-x-auto">
                        <table className="min-w-full divide-y divide-gray-200">
                            <thead className="bg-gray-50">
                                <tr>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Revision
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Writer
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Order
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Quality Rating
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Payment Impact
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Assessed
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Actions
                                    </th>
                                </tr>
                            </thead>
                            <tbody className="bg-white divide-y divide-gray-200">
                                {recentAssessments.map((assessment) => (
                                    <tr key={assessment.id} className="hover:bg-gray-50">
                                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                            #{assessment.id}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                            {assessment.assigned_to?.name || 'Unknown'}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                            <Link
                                                href={route('admin.orders.show', assessment.order.id)}
                                                className="text-blue-600 hover:text-blue-900"
                                            >
                                                #{assessment.order.id}
                                            </Link>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getQualityRatingColor(assessment.quality_rating)}`}>
                                                {assessment.quality_rating}/10
                                            </span>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm">
                                            <span className={`font-medium ${getPaymentImpactColor(assessment.payment_impact)}`}>
                                                {formatCurrency(assessment.payment_impact)}
                                            </span>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                            {new Date(assessment.updated_at).toLocaleDateString('en-US', {
                                                year: 'numeric',
                                                month: 'short',
                                                day: 'numeric'
                                            })}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                            <div className="flex items-center space-x-2">
                                                <button
                                                    onClick={() => handleQualityAssessment(assessment)}
                                                    className="inline-flex items-center px-2 py-1 border border-transparent text-xs font-medium rounded text-blue-700 bg-blue-100 hover:bg-blue-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                                    title="Assess Quality"
                                                >
                                                    <StarIcon className="h-3 w-3 mr-1" />
                                                    Assess
                                                </button>
                                                <button
                                                    onClick={() => handleRejection(assessment)}
                                                    className="inline-flex items-center px-2 py-1 border border-transparent text-xs font-medium rounded text-red-700 bg-red-100 hover:bg-red-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                                                    title="Reject Revision"
                                                >
                                                    <XCircleIcon className="h-3 w-3 mr-1" />
                                                    Reject
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            {/* Quality Assessment Modal */}
            {showQualityForm && selectedRevision && (
                <QualityAssessmentForm
                    revision={selectedRevision}
                    onClose={handleFormClose}
                    onSuccess={handleFormSuccess}
                />
            )}

            {/* Revision Rejection Modal */}
            {showRejectionForm && selectedRevision && (
                <RevisionRejectionForm
                    revision={selectedRevision}
                    onClose={handleFormClose}
                    onSuccess={handleFormSuccess}
                />
            )}
        </AdminLayout>
    );
} 