import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import {
    CogIcon,
    ArrowLeftIcon,
    CurrencyDollarIcon,
    ExclamationTriangleIcon,
    CheckCircleIcon
} from '@heroicons/react/24/outline';

export default function PaymentImpactsSettings({ auth, settings }) {
    const [formData, setFormData] = useState({
        quality_bonus: settings.quality_bonus || 5.00,
        quality_penalty: settings.quality_penalty || 10.00,
        delay_penalty_per_hour: settings.delay_penalty_per_hour || 0.50,
    });

    const [isSubmitting, setIsSubmitting] = useState(false);

    const handleInputChange = (field, value) => {
        setFormData(prev => ({
            ...prev,
            [field]: parseFloat(value) || 0
        }));
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        setIsSubmitting(true);

        router.put(route('admin.payment-impacts.update-settings'), formData, {
            onSuccess: () => {
                setIsSubmitting(false);
            },
            onError: () => {
                setIsSubmitting(false);
            }
        });
    };

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    return (
        <AdminLayout auth={auth}>
            <Head title="Payment Impact Settings" />
            
            <div className="max-w-4xl mx-auto py-6">
                {/* Header */}
                <div className="bg-white shadow-sm rounded-lg mb-6">
                    <div className="px-6 py-4 border-b border-gray-200">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                                <CogIcon className="h-6 w-6 text-blue-600" />
                                <h1 className="text-xl font-semibold text-gray-900">
                                    Payment Impact Settings
                                </h1>
                            </div>
                            <div className="flex items-center space-x-3">
                                <a
                                    href={route('admin.payment-impacts.index')}
                                    className="inline-flex items-center px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                >
                                    <ArrowLeftIcon className="h-4 w-4 mr-2" />
                                    Back to Dashboard
                                </a>
                            </div>
                        </div>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Settings Form */}
                    <div className="lg:col-span-2">
                        <div className="bg-white shadow-sm rounded-lg">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h2 className="text-lg font-medium text-gray-900">Configure Payment Impact Rules</h2>
                                <p className="text-sm text-gray-500 mt-1">
                                    Set the financial incentives and penalties for revision quality and timeliness.
                                </p>
                            </div>
                            <form onSubmit={handleSubmit} className="p-6 space-y-6">
                                {/* Quality Bonus */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Quality Bonus for Excellent Work
                                    </label>
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <CurrencyDollarIcon className="h-5 w-5 text-gray-400" />
                                        </div>
                                        <input
                                            type="number"
                                            step="0.01"
                                            min="0"
                                            max="100"
                                            value={formData.quality_bonus}
                                            onChange={(e) => handleInputChange('quality_bonus', e.target.value)}
                                            className="block w-full pl-10 pr-12 border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500"
                                            placeholder="5.00"
                                        />
                                        <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                            <span className="text-gray-500 sm:text-sm">USD</span>
                                        </div>
                                    </div>
                                    <p className="text-sm text-gray-500 mt-1">
                                        Bonus amount awarded for revisions with quality ratings of 9-10.
                                    </p>
                                </div>

                                {/* Quality Penalty */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Quality Penalty for Poor Work
                                    </label>
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <CurrencyDollarIcon className="h-5 w-5 text-gray-400" />
                                        </div>
                                        <input
                                            type="number"
                                            step="0.01"
                                            min="0"
                                            max="100"
                                            value={formData.quality_penalty}
                                            onChange={(e) => handleInputChange('quality_penalty', e.target.value)}
                                            className="block w-full pl-10 pr-12 border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500"
                                            placeholder="10.00"
                                        />
                                        <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                            <span className="text-gray-500 sm:text-sm">USD</span>
                                        </div>
                                    </div>
                                    <p className="text-sm text-gray-500 mt-1">
                                        Penalty amount deducted for revisions with quality ratings of 1-4.
                                    </p>
                                </div>

                                {/* Delay Penalty */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Delay Penalty per Hour
                                    </label>
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <CurrencyDollarIcon className="h-5 w-5 text-gray-400" />
                                        </div>
                                        <input
                                            type="number"
                                            step="0.01"
                                            min="0"
                                            max="10"
                                            value={formData.delay_penalty_per_hour}
                                            onChange={(e) => handleInputChange('delay_penalty_per_hour', e.target.value)}
                                            className="block w-full pl-10 pr-12 border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500"
                                            placeholder="0.50"
                                        />
                                        <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                            <span className="text-gray-500 sm:text-sm">USD/hour</span>
                                        </div>
                                    </div>
                                    <p className="text-sm text-gray-500 mt-1">
                                        Penalty amount per hour for overdue revisions.
                                    </p>
                                </div>

                                {/* Submit Button */}
                                <div className="pt-4">
                                    <button
                                        type="submit"
                                        disabled={isSubmitting}
                                        className="w-full inline-flex items-center justify-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
                                    >
                                        {isSubmitting ? (
                                            <>
                                                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                                Updating Settings...
                                            </>
                                        ) : (
                                            <>
                                                <CheckCircleIcon className="h-4 w-4 mr-2" />
                                                Update Settings
                                            </>
                                        )}
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>

                    {/* Information Panel */}
                    <div className="space-y-6">
                        {/* Current Settings Summary */}
                        <div className="bg-white shadow-sm rounded-lg">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h3 className="text-lg font-medium text-gray-900">Current Settings</h3>
                            </div>
                            <div className="p-6 space-y-4">
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Quality Bonus:</span>
                                    <p className="text-lg font-semibold text-green-600">
                                        {formatCurrency(settings.quality_bonus)}
                                    </p>
                                </div>
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Quality Penalty:</span>
                                    <p className="text-lg font-semibold text-red-600">
                                        {formatCurrency(settings.quality_penalty)}
                                    </p>
                                </div>
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Delay Penalty:</span>
                                    <p className="text-lg font-semibold text-orange-600">
                                        {formatCurrency(settings.delay_penalty_per_hour)}/hour
                                    </p>
                                </div>
                            </div>
                        </div>

                        {/* How It Works */}
                        <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
                            <h3 className="text-lg font-medium text-blue-900 mb-3">How It Works</h3>
                            <div className="space-y-3 text-sm text-blue-800">
                                <div className="flex items-start space-x-2">
                                    <CheckCircleIcon className="h-4 w-4 text-blue-600 mt-0.5 flex-shrink-0" />
                                    <span>Quality ratings 9-10 receive bonus payments</span>
                                </div>
                                <div className="flex items-start space-x-2">
                                    <CheckCircleIcon className="h-4 w-4 text-blue-600 mt-0.5 flex-shrink-0" />
                                    <span>Quality ratings 7-8 have no payment impact</span>
                                </div>
                                <div className="flex items-start space-x-2">
                                    <CheckCircleIcon className="h-4 w-4 text-blue-600 mt-0.5 flex-shrink-0" />
                                    <span>Quality ratings 5-6 receive 60% penalty</span>
                                </div>
                                <div className="flex items-start space-x-2">
                                    <CheckCircleIcon className="h-4 w-4 text-blue-600 mt-0.5 flex-shrink-0" />
                                    <span>Quality ratings 1-4 receive full penalty</span>
                                </div>
                            </div>
                        </div>

                        {/* Impact Examples */}
                        <div className="bg-gray-50 border border-gray-200 rounded-lg p-6">
                            <h3 className="text-lg font-medium text-gray-900 mb-3">Impact Examples</h3>
                            <div className="space-y-3 text-sm text-gray-700">
                                <div>
                                    <span className="font-medium">Excellent (10/10):</span>
                                    <p className="text-green-600">+{formatCurrency(settings.quality_bonus)} bonus</p>
                                </div>
                                <div>
                                    <span className="font-medium">Good (8/10):</span>
                                    <p className="text-gray-600">No impact</p>
                                </div>
                                <div>
                                    <span className="font-medium">Poor (4/10):</span>
                                    <p className="text-red-600">-{formatCurrency(settings.quality_penalty)} penalty</p>
                                </div>
                                <div>
                                    <span className="font-medium">2 hours overdue:</span>
                                    <p className="text-orange-600">-{formatCurrency(settings.delay_penalty_per_hour * 2)} delay penalty</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
} 