import React, { useState } from "react";
import { Head } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
// Remove this import if it exists
// import { useTheme } from "@/Contexts/ThemeContext";
import {
    DollarSign,
    CreditCard,
    TrendingUp,
    TrendingDown,
    BarChart,
    PieChart,
    ArrowUpRight,
    ArrowDownRight,
    Calendar,
    Filter,
    RefreshCw,
    Users,
    Check,
    AlertCircle,
} from "lucide-react";
import {
    Chart as ChartJS,
    CategoryScale,
    LinearScale,
    PointElement,
    LineElement,
    BarElement,
    ArcElement,
    Title,
    Tooltip,
    Legend,
} from "chart.js";
import { Line, Bar, Pie } from "react-chartjs-2";
import { format } from "date-fns";

// Register ChartJS components
ChartJS.register(
    CategoryScale,
    LinearScale,
    PointElement,
    LineElement,
    BarElement,
    ArcElement,
    Title,
    Tooltip,
    Legend
);

const PaymentAnalytics = ({ analytics, filters }) => {
    // Remove this line that's causing the error
    // const { darkMode } = useTheme();
    const [dateRange, setDateRange] = useState({
        startDate: filters.startDate,
        endDate: filters.endDate,
    });
    const [period, setPeriod] = useState(filters.period);
    const [isFiltering, setIsFiltering] = useState(false);

    // Color schemes that work in both light and dark mode
    const colors = {
        primary: "rgba(59, 130, 246, 0.8)",
        secondary: "rgba(99, 102, 241, 0.8)",
        accent: "rgba(236, 72, 153, 0.7)",
        success: "rgba(34, 197, 94, 0.7)",
        warning: "rgba(245, 158, 11, 0.7)",
        danger: "rgba(239, 68, 68, 0.7)",
        text: "rgba(17, 24, 39, 0.8)",
        gridLines: "rgba(209, 213, 219, 0.5)",
    };

    // Apply filter changes
    const applyFilters = () => {
        const url = new URL(window.location.href);
        url.searchParams.set("start_date", dateRange.startDate);
        url.searchParams.set("end_date", dateRange.endDate);
        url.searchParams.set("period", period);
        window.location.href = url.toString();
    };

    // Format currency
    const formatCurrency = (value) => {
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: "USD",
        }).format(value);
    };

    // Chart configurations
    const revenueChartConfig = {
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                x: {
                    grid: {
                        color: colors.gridLines,
                    },
                    ticks: {
                        color: colors.text,
                    },
                },
                y: {
                    grid: {
                        color: colors.gridLines,
                    },
                    ticks: {
                        color: colors.text,
                        callback: function (value) {
                            return formatCurrency(value);
                        },
                    },
                },
            },
            plugins: {
                legend: {
                    labels: {
                        color: colors.text,
                    },
                },
                tooltip: {
                    callbacks: {
                        label: function (context) {
                            return formatCurrency(context.raw);
                        },
                    },
                },
            },
        },
        data: {
            labels: analytics.trends.labels,
            datasets: [
                {
                    label: "Revenue",
                    data: analytics.trends.revenue,
                    borderColor: colors.primary,
                    backgroundColor: colors.primary,
                    tension: 0.3,
                },
            ],
        },
    };

    const transactionCountChartConfig = {
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                x: {
                    grid: {
                        color: colors.gridLines,
                    },
                    ticks: {
                        color: colors.text,
                    },
                },
                y: {
                    grid: {
                        color: colors.gridLines,
                    },
                    ticks: {
                        color: colors.text,
                        precision: 0,
                    },
                },
            },
            plugins: {
                legend: {
                    labels: {
                        color: colors.text,
                    },
                },
            },
        },
        data: {
            labels: analytics.trends.labels,
            datasets: [
                {
                    label: "Number of Transactions",
                    data: analytics.trends.count,
                    backgroundColor: colors.secondary,
                    borderRadius: 4,
                },
            ],
        },
    };

    const paymentMethodChartConfig = {
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: "right",
                    labels: {
                        color: colors.text,
                    },
                },
                tooltip: {
                    callbacks: {
                        label: function (context) {
                            return `${context.label}: ${formatCurrency(
                                context.raw
                            )}`;
                        },
                    },
                },
            },
        },
        data: {
            labels: analytics.paymentMethods.map((method) => method.name),
            datasets: [
                {
                    data: analytics.paymentMethods.map(
                        (method) => method.revenue
                    ),
                    backgroundColor: [
                        colors.primary,
                        colors.secondary,
                        colors.accent,
                        colors.warning,
                        colors.danger,
                    ],
                    borderWidth: 1,
                    borderColor: "#ffffff", // Will be overridden by dark mode classes
                },
            ],
        },
    };

    const statusDistributionConfig = {
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: "right",
                    labels: {
                        color: colors.text,
                    },
                },
            },
        },
        data: {
            labels: analytics.statusDistribution.map((item) => item.status),
            datasets: [
                {
                    data: analytics.statusDistribution.map(
                        (item) => item.count
                    ),
                    backgroundColor: [
                        colors.success,
                        colors.danger,
                        colors.warning,
                        colors.secondary,
                    ],
                    borderWidth: 1,
                    borderColor: "#ffffff", // Will be overridden by dark mode classes
                },
            ],
        },
    };

    return (
        <AdminLayout>
            <Head title="Payment Analytics" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex justify-between items-center mb-6">
                        <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                            Payment Analytics
                        </h1>
                        <button
                            onClick={() => setIsFiltering(!isFiltering)}
                            className="flex items-center px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded transition-colors"
                        >
                            <Filter size={16} className="mr-2" />
                            Filter
                        </button>
                    </div>

                    {isFiltering && (
                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow mb-6">
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <div>
                                    <label
                                        htmlFor="startDate"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                    >
                                        Start Date
                                    </label>
                                    <input
                                        type="date"
                                        id="startDate"
                                        value={dateRange.startDate}
                                        onChange={(e) =>
                                            setDateRange({
                                                ...dateRange,
                                                startDate: e.target.value,
                                            })
                                        }
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                    />
                                </div>
                                <div>
                                    <label
                                        htmlFor="endDate"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                    >
                                        End Date
                                    </label>
                                    <input
                                        type="date"
                                        id="endDate"
                                        value={dateRange.endDate}
                                        onChange={(e) =>
                                            setDateRange({
                                                ...dateRange,
                                                endDate: e.target.value,
                                            })
                                        }
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                    />
                                </div>
                                <div>
                                    <label
                                        htmlFor="period"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                    >
                                        Group By
                                    </label>
                                    <select
                                        id="period"
                                        value={period}
                                        onChange={(e) =>
                                            setPeriod(e.target.value)
                                        }
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                    >
                                        <option value="daily">Daily</option>
                                        <option value="monthly">Monthly</option>
                                    </select>
                                </div>
                            </div>
                            <div className="mt-4 flex justify-end">
                                <button
                                    onClick={applyFilters}
                                    className="px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded transition-colors flex items-center"
                                >
                                    <RefreshCw size={16} className="mr-2" />
                                    Apply Filters
                                </button>
                            </div>
                        </div>
                    )}

                    {/* Summary Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
                        <SummaryCard
                            title="Total Revenue"
                            value={formatCurrency(
                                analytics.summary.totalRevenue
                            )}
                            icon={
                                <DollarSign className="h-8 w-8 text-green-500" />
                            }
                            change={analytics.summary.revenueGrowth}
                        />
                        <SummaryCard
                            title="Transactions"
                            value={analytics.summary.totalPayments}
                            icon={
                                <CreditCard className="h-8 w-8 text-blue-500" />
                            }
                        />
                        <SummaryCard
                            title="Success Rate"
                            value={`${analytics.summary.successRate}%`}
                            icon={
                                <Check className="h-8 w-8 text-emerald-500" />
                            }
                        />
                        <SummaryCard
                            title="Failed Payments"
                            value={analytics.summary.failedPayments}
                            icon={
                                <AlertCircle className="h-8 w-8 text-red-500" />
                            }
                        />
                    </div>

                    {/* Charts */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                <TrendingUp className="h-5 w-5 mr-2 text-blue-500" />
                                Revenue Trend
                            </h2>
                            <div className="h-80">
                                <Line
                                    options={revenueChartConfig.options}
                                    data={revenueChartConfig.data}
                                />
                            </div>
                        </div>
                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                <BarChart className="h-5 w-5 mr-2 text-purple-500" />
                                Transaction Count
                            </h2>
                            <div className="h-80">
                                <Bar
                                    options={
                                        transactionCountChartConfig.options
                                    }
                                    data={transactionCountChartConfig.data}
                                />
                            </div>
                        </div>
                    </div>

                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                <PieChart className="h-5 w-5 mr-2 text-blue-500" />
                                Payment Methods
                            </h2>
                            <div className="h-80">
                                <Pie
                                    options={paymentMethodChartConfig.options}
                                    data={paymentMethodChartConfig.data}
                                />
                            </div>
                        </div>
                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                <PieChart className="h-5 w-5 mr-2 text-indigo-500" />
                                Transaction Status
                            </h2>
                            <div className="h-80">
                                <Pie
                                    options={statusDistributionConfig.options}
                                    data={statusDistributionConfig.data}
                                />
                            </div>
                        </div>
                    </div>

                    {/* Top Clients */}
                    <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow mb-6">
                        <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                            <Users className="h-5 w-5 mr-2 text-indigo-500" />
                            Top Clients by Payment Volume
                        </h2>
                        <div className="overflow-x-auto">
                            <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                <thead className="bg-gray-50 dark:bg-gray-700">
                                    <tr>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Name
                                        </th>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Transactions
                                        </th>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Total Spent
                                        </th>
                                    </tr>
                                </thead>
                                <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                    {analytics.topClients.map((client) => (
                                        <tr key={client.id}>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-gray-100">
                                                {client.name ||
                                                    "Unknown Client"}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                {client.payment_count}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                {formatCurrency(
                                                    client.total_spent
                                                )}
                                            </td>
                                        </tr>
                                    ))}
                                    {analytics.topClients.length === 0 && (
                                        <tr>
                                            <td
                                                colSpan="3"
                                                className="px-6 py-4 text-center text-sm text-gray-500 dark:text-gray-400"
                                            >
                                                No client data available
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>

                    {/* Recent Transactions */}
                    <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                        <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                            <Calendar className="h-5 w-5 mr-2 text-green-500" />
                            Recent Transactions
                        </h2>
                        <div className="overflow-x-auto">
                            <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                <thead className="bg-gray-50 dark:bg-gray-700">
                                    <tr>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Date
                                        </th>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Order
                                        </th>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Client
                                        </th>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Amount
                                        </th>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Method
                                        </th>
                                        <th
                                            scope="col"
                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                        >
                                            Status
                                        </th>
                                    </tr>
                                </thead>
                                <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                    {analytics.recentTransactions.map(
                                        (transaction) => (
                                            <tr key={transaction.id}>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                    {transaction.date
                                                        ? format(
                                                              new Date(
                                                                  transaction.date
                                                              ),
                                                              "MMM dd, yyyy"
                                                          )
                                                        : "N/A"}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                    {transaction.order_number ? (
                                                        <a
                                                            href={`/admin/orders/${transaction.order_id}`}
                                                            className="text-blue-600 dark:text-blue-400 hover:underline"
                                                        >
                                                            #
                                                            {
                                                                transaction.order_number
                                                            }
                                                        </a>
                                                    ) : (
                                                        "N/A"
                                                    )}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                    {transaction.client_name}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-gray-100 font-medium">
                                                    {formatCurrency(
                                                        transaction.amount
                                                    )}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400 capitalize">
                                                    {transaction.method}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <span
                                                        className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                                            transaction.status ===
                                                            "completed"
                                                                ? "bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200"
                                                                : transaction.status ===
                                                                  "pending"
                                                                ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-200"
                                                                : "bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-200"
                                                        }`}
                                                    >
                                                        {transaction.status}
                                                    </span>
                                                </td>
                                            </tr>
                                        )
                                    )}
                                    {analytics.recentTransactions.length ===
                                        0 && (
                                        <tr>
                                            <td
                                                colSpan="6"
                                                className="px-6 py-4 text-center text-sm text-gray-500 dark:text-gray-400"
                                            >
                                                No transactions available
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

// Summary Card Component
const SummaryCard = ({ title, value, icon, change }) => {
    return (
        <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow">
            <div className="flex items-center justify-between">
                <div>
                    <p className="text-sm font-medium text-gray-600 dark:text-gray-400">
                        {title}
                    </p>
                    <p className="mt-1 text-2xl font-semibold text-gray-900 dark:text-white">
                        {value}
                    </p>
                    {change !== undefined && (
                        <div className="mt-2 flex items-center">
                            {change >= 0 ? (
                                <ArrowUpRight className="h-4 w-4 text-green-500 mr-1" />
                            ) : (
                                <ArrowDownRight className="h-4 w-4 text-red-500 mr-1" />
                            )}
                            <span
                                className={
                                    change >= 0
                                        ? "text-green-500"
                                        : "text-red-500"
                                }
                            >
                                {Math.abs(change)}%
                            </span>
                            <span className="text-xs text-gray-500 dark:text-gray-400 ml-1">
                                vs previous period
                            </span>
                        </div>
                    )}
                </div>
                <div className="p-3 rounded-full bg-blue-50 dark:bg-blue-900/30">
                    {icon}
                </div>
            </div>
        </div>
    );
};

export default PaymentAnalytics;
