import React, { useState, useEffect } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import {
    DocumentTextIcon,
    ExclamationTriangleIcon,
    ClockIcon,
    UserIcon,
    CheckCircleIcon
} from '@heroicons/react/24/outline';

export default function RevisionCreate({ auth, orders, writers }) {
    const { data, setData, post, processing, errors, reset } = useForm({
        order_id: '',
        revision_type: '',
        revision_reason: '',
        specific_instructions: '',
        client_notes: '',
        assigned_to: '',
        due_date: '',
        is_client_requested: false,
        is_final_revision: false,
    });

    const [selectedOrder, setSelectedOrder] = useState(null);
    const [filteredWriters, setFilteredWriters] = useState(writers);

    useEffect(() => {
        if (data.order_id) {
            const order = orders.find(o => o.id == data.order_id);
            setSelectedOrder(order);
            
            // Filter writers based on order requirements
            if (order) {
                const suitableWriters = writers.filter(writer => 
                    writer.is_active && 
                    (!order.academic_level || writer.academic_levels?.includes(order.academic_level)) &&
                    (!order.subject || writer.subjects?.includes(order.subject))
                );
                setFilteredWriters(suitableWriters);
            }
        }
    }, [data.order_id, orders, writers]);

    const handleSubmit = (e) => {
        e.preventDefault();
        
        if (data.is_client_requested) {
            post(route('admin.revisions.client-request'));
        } else {
            post(route('admin.revisions.store'));
        }
    };

    const handleOrderChange = (orderId) => {
        setData('order_id', orderId);
        setData('assigned_to', ''); // Reset writer assignment
    };

    const getRevisionTypeOptions = () => [
        { value: 'quality', label: 'Quality Issues', description: 'Grammar, formatting, or writing quality' },
        { value: 'requirements', label: 'Requirements Not Met', description: 'Missing elements or instructions' },
        { value: 'formatting', label: 'Formatting Issues', description: 'Citation style, layout, or structure' },
        { value: 'plagiarism', label: 'Plagiarism Concerns', description: 'Content originality issues' },
        { value: 'deadline', label: 'Deadline Issues', description: 'Timing or scheduling problems' },
        { value: 'communication', label: 'Communication Issues', description: 'Writer-client interaction problems' },
        { value: 'other', label: 'Other', description: 'Miscellaneous issues' },
    ];

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    return (
        <AdminLayout auth={auth}>
            <Head title="Create Revision" />
            
            <div className="max-w-4xl mx-auto py-6">
                <div className="bg-white shadow-sm rounded-lg">
                    {/* Header */}
                    <div className="px-6 py-4 border-b border-gray-200">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                                <DocumentTextIcon className="h-6 w-6 text-blue-600" />
                                <h1 className="text-xl font-semibold text-gray-900">Create New Revision</h1>
                            </div>
                            <button
                                onClick={() => router.visit(route('admin.revisions.index'))}
                                className="text-gray-500 hover:text-gray-700 text-sm"
                            >
                                ← Back to Revisions
                            </button>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit} className="p-6 space-y-6">
                        {/* Order Selection */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Select Order *
                            </label>
                            <select
                                value={data.order_id}
                                onChange={(e) => handleOrderChange(e.target.value)}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                required
                            >
                                <option value="">Choose an order...</option>
                                {orders.map((order) => (
                                    <option key={order.id} value={order.id}>
                                        #{order.id} - {order.title} ({order.order_status})
                                    </option>
                                ))}
                            </select>
                            {errors.order_id && (
                                <p className="mt-1 text-sm text-red-600">{errors.order_id}</p>
                            )}
                        </div>

                        {/* Order Details */}
                        {selectedOrder && (
                            <div className="bg-gray-50 rounded-lg p-4">
                                <h3 className="text-lg font-medium text-gray-900 mb-3">Order Details</h3>
                                <div className="grid grid-cols-2 gap-4 text-sm">
                                    <div>
                                        <span className="font-medium text-gray-600">Title:</span>
                                        <p className="text-gray-900">{selectedOrder.title}</p>
                                    </div>
                                    <div>
                                        <span className="font-medium text-gray-600">Status:</span>
                                        <p className="text-gray-900">{selectedOrder.order_status}</p>
                                    </div>
                                    <div>
                                        <span className="font-medium text-gray-600">Writer:</span>
                                        <p className="text-gray-900">
                                            {selectedOrder.writer ? selectedOrder.writer.name : 'Unassigned'}
                                        </p>
                                    </div>
                                    <div>
                                        <span className="font-medium text-gray-600">Client:</span>
                                        <p className="text-gray-900">{selectedOrder.user.name}</p>
                                    </div>
                                    <div>
                                        <span className="font-medium text-gray-600">Due Date:</span>
                                        <p className="text-gray-900">{formatDate(selectedOrder.due_date)}</p>
                                    </div>
                                    <div>
                                        <span className="font-medium text-gray-600">Revision Count:</span>
                                        <p className="text-gray-900">{selectedOrder.revisions?.length || 0}</p>
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Revision Type */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Revision Type *
                            </label>
                            <select
                                value={data.revision_type}
                                onChange={(e) => setData('revision_type', e.target.value)}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                required
                            >
                                <option value="">Select revision type...</option>
                                {getRevisionTypeOptions().map((option) => (
                                    <option key={option.value} value={option.value}>
                                        {option.label} - {option.description}
                                    </option>
                                ))}
                            </select>
                            {errors.revision_type && (
                                <p className="mt-1 text-sm text-red-600">{errors.revision_type}</p>
                            )}
                        </div>

                        {/* Revision Reason */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Revision Reason *
                            </label>
                            <textarea
                                value={data.revision_reason}
                                onChange={(e) => setData('revision_reason', e.target.value)}
                                rows={3}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="Describe why this revision is needed..."
                                required
                            />
                            {errors.revision_reason && (
                                <p className="mt-1 text-sm text-red-600">{errors.revision_reason}</p>
                            )}
                        </div>

                        {/* Specific Instructions */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Specific Instructions
                            </label>
                            <textarea
                                value={data.specific_instructions}
                                onChange={(e) => setData('specific_instructions', e.target.value)}
                                rows={4}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="Provide detailed instructions for the writer..."
                            />
                            {errors.specific_instructions && (
                                <p className="mt-1 text-sm text-red-600">{errors.specific_instructions}</p>
                            )}
                        </div>

                        {/* Client Notes */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Client Notes
                            </label>
                            <textarea
                                value={data.client_notes}
                                onChange={(e) => setData('client_notes', e.target.value)}
                                rows={3}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="Any additional notes from the client..."
                            />
                            {errors.client_notes && (
                                <p className="mt-1 text-sm text-red-600">{errors.client_notes}</p>
                            )}
                        </div>

                        {/* Assignment Options */}
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            {/* Writer Assignment */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-2">
                                    Assign to Writer
                                </label>
                                <select
                                    value={data.assigned_to}
                                    onChange={(e) => setData('assigned_to', e.target.value)}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                >
                                    <option value="">Select writer...</option>
                                    {filteredWriters.map((writer) => (
                                        <option key={writer.id} value={writer.id}>
                                            {writer.name} ({writer.email})
                                        </option>
                                    ))}
                                </select>
                                {errors.assigned_to && (
                                    <p className="mt-1 text-sm text-red-600">{errors.assigned_to}</p>
                                )}
                            </div>

                            {/* Due Date */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-2">
                                    Due Date
                                </label>
                                <input
                                    type="datetime-local"
                                    value={data.due_date}
                                    onChange={(e) => setData('due_date', e.target.value)}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                />
                                {errors.due_date && (
                                    <p className="mt-1 text-sm text-red-600">{errors.due_date}</p>
                                )}
                            </div>
                        </div>

                        {/* Options */}
                        <div className="space-y-4">
                            <div className="flex items-center space-x-3">
                                <input
                                    type="checkbox"
                                    id="is_client_requested"
                                    checked={data.is_client_requested}
                                    onChange={(e) => setData('is_client_requested', e.target.checked)}
                                    className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                                />
                                <label htmlFor="is_client_requested" className="text-sm text-gray-700">
                                    This is a client-requested revision
                                </label>
                            </div>

                            <div className="flex items-center space-x-3">
                                <input
                                    type="checkbox"
                                    id="is_final_revision"
                                    checked={data.is_final_revision}
                                    onChange={(e) => setData('is_final_revision', e.target.checked)}
                                    className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                                />
                                <label htmlFor="is_final_revision" className="text-sm text-gray-700">
                                    Mark as final revision (no more revisions allowed)
                                </label>
                            </div>
                        </div>

                        {/* Submit Buttons */}
                        <div className="flex justify-end space-x-3 pt-6 border-t border-gray-200">
                            <button
                                type="button"
                                onClick={() => router.visit(route('admin.revisions.index'))}
                                className="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                            >
                                Cancel
                            </button>
                            <button
                                type="submit"
                                disabled={processing}
                                className="px-4 py-2 bg-blue-600 border border-transparent rounded-md text-sm font-medium text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50"
                            >
                                {processing ? 'Creating...' : 'Create Revision'}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
} 