import React, { useState, useEffect } from 'react';
import { Head, Link } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { 
    ClockIcon, 
    CheckCircleIcon, 
    ExclamationTriangleIcon, 
    DocumentTextIcon,
    UserIcon,
    CalendarIcon,
    ChartBarIcon
} from '@heroicons/react/24/outline';

export default function RevisionDashboard({ auth, stats, recentRevisions, overdueRevisions, revisionTrends }) {
    const [selectedPeriod, setSelectedPeriod] = useState('30');

    const statusColors = {
        requested: 'bg-blue-100 text-blue-800',
        in_progress: 'bg-yellow-100 text-yellow-800',
        completed: 'bg-green-100 text-green-800',
        rejected: 'bg-red-100 text-red-800',
        escalated: 'bg-purple-100 text-purple-800',
    };

    const statusLabels = {
        requested: 'Requested',
        in_progress: 'In Progress',
        completed: 'Completed',
        rejected: 'Rejected',
        escalated: 'Escalated',
    };

    const revisionTypeColors = {
        quality: 'bg-red-100 text-red-800',
        requirements: 'bg-blue-100 text-blue-800',
        formatting: 'bg-green-100 text-green-800',
        plagiarism: 'bg-orange-100 text-orange-800',
        deadline: 'bg-purple-100 text-purple-800',
        communication: 'bg-indigo-100 text-indigo-800',
        other: 'bg-gray-100 text-gray-800',
    };

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const getTimeAgo = (dateString) => {
        const now = new Date();
        const date = new Date(dateString);
        const diffInHours = Math.floor((now - date) / (1000 * 60 * 60));
        
        if (diffInHours < 1) return 'Just now';
        if (diffInHours < 24) return `${diffInHours}h ago`;
        return `${Math.floor(diffInHours / 24)}d ago`;
    };

    return (
        <AdminLayout auth={auth}>
            <Head title="Revision Dashboard" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                            Revision Dashboard
                        </h1>
                        <p className="mt-2 text-gray-600 dark:text-gray-400">
                            Monitor and manage all order revisions across the platform
                        </p>
                    </div>

                    {/* Statistics Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                        <div className="bg-white dark:bg-gray-800 overflow-hidden shadow rounded-lg">
                            <div className="p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <DocumentTextIcon className="h-6 w-6 text-blue-600" />
                                    </div>
                                    <div className="ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-sm font-medium text-gray-500 dark:text-gray-400 truncate">
                                                Total Revisions
                                            </dt>
                                            <dd className="text-lg font-medium text-gray-900 dark:text-white">
                                                {stats.total}
                                            </dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white dark:bg-gray-800 overflow-hidden shadow rounded-lg">
                            <div className="p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <ClockIcon className="h-6 w-6 text-yellow-600" />
                                    </div>
                                    <div className="ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-sm font-medium text-gray-500 dark:text-gray-400 truncate">
                                                Pending
                                            </dt>
                                            <dd className="text-lg font-medium text-gray-900 dark:text-white">
                                                {stats.pending}
                                            </dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white dark:bg-gray-800 overflow-hidden shadow rounded-lg">
                            <div className="p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <ExclamationTriangleIcon className="h-6 w-6 text-red-600" />
                                    </div>
                                    <div className="ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-sm font-medium text-gray-500 dark:text-gray-400 truncate">
                                                Overdue
                                            </dt>
                                            <dd className="text-lg font-medium text-gray-900 dark:text-white">
                                                {stats.overdue}
                                            </dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white dark:bg-gray-800 overflow-hidden shadow rounded-lg">
                            <div className="p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <CheckCircleIcon className="h-6 w-6 text-green-600" />
                                    </div>
                                    <div className="ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-sm font-medium text-gray-500 dark:text-gray-400 truncate">
                                                Completed
                                            </dt>
                                            <dd className="text-lg font-medium text-gray-900 dark:text-white">
                                                {stats.completed}
                                            </dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Performance Metrics */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
                        {/* Revision Types Distribution */}
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg">
                            <div className="px-4 py-5 sm:p-6">
                                <h3 className="text-lg leading-6 font-medium text-gray-900 dark:text-white mb-4">
                                    Revision Types
                                </h3>
                                <div className="space-y-3">
                                    {Object.entries(stats.types).map(([type, count]) => (
                                        <div key={type} className="flex items-center justify-between">
                                            <span className="text-sm text-gray-600 dark:text-gray-400 capitalize">
                                                {type.replace('_', ' ')}
                                            </span>
                                            <div className="flex items-center space-x-2">
                                                <div className="w-24 bg-gray-200 rounded-full h-2">
                                                    <div 
                                                        className={`h-2 rounded-full ${revisionTypeColors[type] || 'bg-gray-400'}`}
                                                        style={{ width: `${(count / stats.total) * 100}%` }}
                                                    ></div>
                                                </div>
                                                <span className="text-sm font-medium text-gray-900 dark:text-white w-8 text-right">
                                                    {count}
                                                </span>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </div>

                        {/* Performance Metrics */}
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg">
                            <div className="px-4 py-5 sm:p-6">
                                <h3 className="text-lg leading-6 font-medium text-gray-900 dark:text-white mb-4">
                                    Performance Metrics
                                </h3>
                                <div className="space-y-4">
                                    <div className="flex items-center justify-between">
                                        <span className="text-sm text-gray-600 dark:text-gray-400">
                                            Average Completion Time
                                        </span>
                                        <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                            {stats.avg_completion_hours}h
                                        </span>
                                    </div>
                                    <div className="flex items-center justify-between">
                                        <span className="text-sm text-gray-600 dark:text-gray-400">
                                            Escalated Revisions
                                        </span>
                                        <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                            {stats.escalated}
                                        </span>
                                    </div>
                                    <div className="flex items-center justify-between">
                                        <span className="text-sm text-gray-600 dark:text-gray-400">
                                            Success Rate
                                        </span>
                                        <span className="text-lg font-semibold text-green-600">
                                            {stats.total > 0 ? Math.round((stats.completed / stats.total) * 100) : 0}%
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Recent Activity and Overdue Revisions */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
                        {/* Recent Revisions */}
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg">
                            <div className="px-4 py-5 sm:p-6">
                                <div className="flex items-center justify-between mb-4">
                                    <h3 className="text-lg leading-6 font-medium text-gray-900 dark:text-white">
                                        Recent Revisions
                                    </h3>
                                    <Link
                                        href={route('admin.revisions.index')}
                                        className="text-sm text-blue-600 hover:text-blue-500"
                                    >
                                        View All
                                    </Link>
                                </div>
                                <div className="space-y-3">
                                    {recentRevisions.map((revision) => (
                                        <div key={revision.id} className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                                            <div className="flex-1 min-w-0">
                                                <div className="flex items-center space-x-2">
                                                    <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                        Order #{revision.order.order_number}
                                                    </span>
                                                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${statusColors[revision.status]}`}>
                                                        {statusLabels[revision.status]}
                                                    </span>
                                                </div>
                                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                                    {revision.revision_type} • {getTimeAgo(revision.created_at)}
                                                </div>
                                            </div>
                                            <Link
                                                href={route('admin.revisions.show', revision.id)}
                                                className="text-blue-600 hover:text-blue-500 text-sm"
                                            >
                                                View
                                            </Link>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </div>

                        {/* Overdue Revisions */}
                        <div className="bg-white dark:bg-gray-800 shadow rounded-lg">
                            <div className="px-4 py-5 sm:p-6">
                                <div className="flex items-center justify-between mb-4">
                                    <h3 className="text-lg leading-6 font-medium text-gray-900 dark:text-white">
                                        Overdue Revisions
                                    </h3>
                                    <span className="text-sm text-red-600 font-medium">
                                        {overdueRevisions.length} overdue
                                    </span>
                                </div>
                                <div className="space-y-3">
                                    {overdueRevisions.length > 0 ? (
                                        overdueRevisions.map((revision) => (
                                            <div key={revision.id} className="flex items-center justify-between p-3 bg-red-50 dark:bg-red-900/20 rounded-lg border border-red-200 dark:border-red-800">
                                                <div className="flex-1 min-w-0">
                                                    <div className="flex items-center space-x-2">
                                                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                            Order #{revision.order.order_number}
                                                        </span>
                                                        <span className="text-xs text-red-600 dark:text-red-400">
                                                            {Math.floor(revision.getOverdueHours())}h overdue
                                                        </span>
                                                    </div>
                                                    <div className="text-sm text-gray-500 dark:text-gray-400">
                                                        Due: {formatDate(revision.due_date)}
                                                    </div>
                                                </div>
                                                <Link
                                                    href={route('admin.revisions.show', revision.id)}
                                                    className="text-red-600 hover:text-red-500 text-sm"
                                                >
                                                    Review
                                                </Link>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="text-center py-4 text-gray-500 dark:text-gray-400">
                                            No overdue revisions
                                        </div>
                                    )}
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Quick Actions */}
                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg">
                        <div className="px-4 py-5 sm:p-6">
                            <h3 className="text-lg leading-6 font-medium text-gray-900 dark:text-white mb-4">
                                Quick Actions
                            </h3>
                            <div className="flex flex-wrap gap-3">
                                <Link
                                    href={route('admin.revisions.create')}
                                    className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                >
                                    <DocumentTextIcon className="h-4 w-4 mr-2" />
                                    Create Revision
                                </Link>
                                <Link
                                    href={route('admin.revisions.index')}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                >
                                    <ChartBarIcon className="h-4 w-4 mr-2" />
                                    View All Revisions
                                </Link>
                                <Link
                                    href={route('admin.revisions.index', { status: 'overdue' })}
                                    className="inline-flex items-center px-4 py-2 border border-red-300 text-sm font-medium rounded-md text-red-700 bg-red-50 hover:bg-red-100 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                                >
                                    <ExclamationTriangleIcon className="h-4 w-4 mr-2" />
                                    Handle Overdue
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
} 