import React, { useState, useEffect } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import {
    DocumentTextIcon,
    ExclamationTriangleIcon,
    ClockIcon,
    UserIcon,
    CheckCircleIcon,
    PencilIcon
} from '@heroicons/react/24/outline';

export default function RevisionEdit({ auth, revision, writers }) {
    const { data, setData, put, processing, errors, reset } = useForm({
        revision_type: revision.revision_type || '',
        revision_reason: revision.revision_reason || '',
        specific_instructions: revision.specific_instructions || '',
        client_notes: revision.client_notes || '',
        admin_notes: revision.admin_notes || '',
        assigned_to: revision.assigned_to?.id || '',
        due_date: revision.due_date ? new Date(revision.due_date).toISOString().slice(0, 16) : '',
        is_final_revision: revision.is_final_revision || false,
        quality_rating: revision.quality_rating || '',
        quality_notes: revision.quality_notes || '',
    });

    const [filteredWriters, setFilteredWriters] = useState(writers);

    useEffect(() => {
        // Filter writers based on order requirements
        if (revision.order) {
            const suitableWriters = writers.filter(writer => 
                writer.is_active && 
                (!revision.order.academic_level || writer.academic_levels?.includes(revision.order.academic_level)) &&
                (!revision.order.subject || writer.subjects?.includes(revision.order.subject))
            );
            setFilteredWriters(suitableWriters);
        }
    }, [revision.order, writers]);

    const handleSubmit = (e) => {
        e.preventDefault();
        put(route('admin.revisions.update', revision.id));
    };

    const getRevisionTypeOptions = () => [
        { value: 'quality', label: 'Quality Issues', description: 'Grammar, formatting, or writing quality' },
        { value: 'requirements', label: 'Requirements Not Met', description: 'Missing elements or instructions' },
        { value: 'formatting', label: 'Formatting Issues', description: 'Citation style, layout, or structure' },
        { value: 'plagiarism', label: 'Plagiarism Concerns', description: 'Content originality issues' },
        { value: 'deadline', label: 'Deadline Issues', description: 'Timing or scheduling problems' },
        { value: 'communication', label: 'Communication Issues', description: 'Writer-client interaction problems' },
        { value: 'other', label: 'Other', description: 'Miscellaneous issues' },
    ];

    const formatDate = (dateString) => {
        if (!dateString) return 'Not set';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const getStatusBadge = (status) => {
        const statusConfig = {
            requested: { color: 'bg-yellow-100 text-yellow-800', label: 'Requested' },
            assigned: { color: 'bg-blue-100 text-blue-800', label: 'Assigned' },
            in_progress: { color: 'bg-purple-100 text-purple-800', label: 'In Progress' },
            completed: { color: 'bg-green-100 text-green-800', label: 'Completed' },
            rejected: { color: 'bg-red-100 text-red-800', label: 'Rejected' },
            escalated: { color: 'bg-orange-100 text-orange-800', label: 'Escalated' },
        };

        const config = statusConfig[status] || { color: 'bg-gray-100 text-gray-800', label: status };
        
        return (
            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${config.color}`}>
                {config.label}
            </span>
        );
    };

    return (
        <AdminLayout auth={auth}>
            <Head title={`Edit Revision #${revision.id}`} />
            
            <div className="max-w-4xl mx-auto py-6">
                <div className="bg-white shadow-sm rounded-lg">
                    {/* Header */}
                    <div className="px-6 py-4 border-b border-gray-200">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                                <PencilIcon className="h-6 w-6 text-blue-600" />
                                <h1 className="text-xl font-semibold text-gray-900">Edit Revision #{revision.id}</h1>
                                {getStatusBadge(revision.status)}
                            </div>
                            <button
                                onClick={() => router.visit(route('admin.revisions.show', revision.id))}
                                className="text-gray-500 hover:text-gray-700 text-sm"
                            >
                                ← Back to Revision
                            </button>
                        </div>
                    </div>

                    {/* Order Summary */}
                    <div className="px-6 py-4 bg-gray-50 border-b border-gray-200">
                        <h3 className="text-sm font-medium text-gray-900 mb-2">Order Summary</h3>
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                            <div>
                                <span className="text-gray-500">Order ID:</span>
                                <p className="text-gray-900">#{revision.order.id}</p>
                            </div>
                            <div>
                                <span className="text-gray-500">Title:</span>
                                <p className="text-gray-900">{revision.order.title}</p>
                            </div>
                            <div>
                                <span className="text-gray-500">Status:</span>
                                <p className="text-gray-900">{revision.order.order_status}</p>
                            </div>
                            <div>
                                <span className="text-gray-500">Revision #:</span>
                                <p className="text-gray-900">{revision.revision_number}</p>
                            </div>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit} className="p-6 space-y-6">
                        {/* Revision Type */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Revision Type *
                            </label>
                            <select
                                value={data.revision_type}
                                onChange={(e) => setData('revision_type', e.target.value)}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                required
                            >
                                <option value="">Select revision type...</option>
                                {getRevisionTypeOptions().map((option) => (
                                    <option key={option.value} value={option.value}>
                                        {option.label} - {option.description}
                                    </option>
                                ))}
                            </select>
                            {errors.revision_type && (
                                <p className="mt-1 text-sm text-red-600">{errors.revision_type}</p>
                            )}
                        </div>

                        {/* Revision Reason */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Revision Reason *
                            </label>
                            <textarea
                                value={data.revision_reason}
                                onChange={(e) => setData('revision_reason', e.target.value)}
                                rows={3}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="Describe why this revision is needed..."
                                required
                            />
                            {errors.revision_reason && (
                                <p className="mt-1 text-sm text-red-600">{errors.revision_reason}</p>
                            )}
                        </div>

                        {/* Specific Instructions */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Specific Instructions
                            </label>
                            <textarea
                                value={data.specific_instructions}
                                onChange={(e) => setData('specific_instructions', e.target.value)}
                                rows={4}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="Provide detailed instructions for the writer..."
                            />
                            {errors.specific_instructions && (
                                <p className="mt-1 text-sm text-red-600">{errors.specific_instructions}</p>
                            )}
                        </div>

                        {/* Client Notes */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Client Notes
                            </label>
                            <textarea
                                value={data.client_notes}
                                onChange={(e) => setData('client_notes', e.target.value)}
                                rows={3}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="Any additional notes from the client..."
                            />
                            {errors.client_notes && (
                                <p className="mt-1 text-sm text-red-600">{errors.client_notes}</p>
                            )}
                        </div>

                        {/* Admin Notes */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Admin Notes
                            </label>
                            <textarea
                                value={data.admin_notes}
                                onChange={(e) => setData('admin_notes', e.target.value)}
                                rows={3}
                                className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="Internal notes for administrators..."
                            />
                            {errors.admin_notes && (
                                <p className="mt-1 text-sm text-red-600">{errors.admin_notes}</p>
                            )}
                        </div>

                        {/* Assignment and Due Date */}
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            {/* Writer Assignment */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-2">
                                    Assign to Writer
                                </label>
                                <select
                                    value={data.assigned_to}
                                    onChange={(e) => setData('assigned_to', e.target.value)}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                >
                                    <option value="">Select writer...</option>
                                    {filteredWriters.map((writer) => (
                                        <option key={writer.id} value={writer.id}>
                                            {writer.name} ({writer.email})
                                        </option>
                                    ))}
                                </select>
                                {errors.assigned_to && (
                                    <p className="mt-1 text-sm text-red-600">{errors.assigned_to}</p>
                                )}
                            </div>

                            {/* Due Date */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-2">
                                    Due Date
                                </label>
                                <input
                                    type="datetime-local"
                                    value={data.due_date}
                                    onChange={(e) => setData('due_date', e.target.value)}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                />
                                {errors.due_date && (
                                    <p className="mt-1 text-sm text-red-600">{errors.due_date}</p>
                                )}
                            </div>
                        </div>

                        {/* Quality Assessment */}
                        {revision.status === 'completed' && (
                            <div className="border-t border-gray-200 pt-6">
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Quality Assessment</h3>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-2">
                                            Quality Rating (1-10)
                                        </label>
                                        <input
                                            type="number"
                                            min="1"
                                            max="10"
                                            value={data.quality_rating}
                                            onChange={(e) => setData('quality_rating', e.target.value)}
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                        />
                                        {errors.quality_rating && (
                                            <p className="mt-1 text-sm text-red-600">{errors.quality_rating}</p>
                                        )}
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-2">
                                            Quality Notes
                                        </label>
                                        <textarea
                                            value={data.quality_notes}
                                            onChange={(e) => setData('quality_notes', e.target.value)}
                                            rows={3}
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                            placeholder="Assessment notes and feedback..."
                                        />
                                        {errors.quality_notes && (
                                            <p className="mt-1 text-sm text-red-600">{errors.quality_notes}</p>
                                        )}
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Options */}
                        <div className="space-y-4">
                            <div className="flex items-center space-x-3">
                                <input
                                    type="checkbox"
                                    id="is_final_revision"
                                    checked={data.is_final_revision}
                                    onChange={(e) => setData('is_final_revision', e.target.checked)}
                                    className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                                />
                                <label htmlFor="is_final_revision" className="text-sm text-gray-700">
                                    Mark as final revision (no more revisions allowed)
                                </label>
                            </div>
                        </div>

                        {/* Submit Buttons */}
                        <div className="flex justify-end space-x-3 pt-6 border-t border-gray-200">
                            <button
                                type="button"
                                onClick={() => router.visit(route('admin.revisions.show', revision.id))}
                                className="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                            >
                                Cancel
                            </button>
                            <button
                                type="submit"
                                disabled={processing}
                                className="px-4 py-2 bg-blue-600 border border-transparent rounded-md text-sm font-medium text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50"
                            >
                                {processing ? 'Updating...' : 'Update Revision'}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
} 