import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import {
    DocumentTextIcon,
    ExclamationTriangleIcon,
    ClockIcon,
    UserIcon,
    CheckCircleIcon,
    PencilIcon,
    TrashIcon,
    ArrowUpIcon,
    ChatBubbleLeftRightIcon
} from '@heroicons/react/24/outline';

export default function RevisionShow({ auth, revision, writers }) {
    const [showAssignmentModal, setShowAssignmentModal] = useState(false);
    const [showEscalationModal, setShowEscalationModal] = useState(false);
    const [showRejectionModal, setShowRejectionModal] = useState(false);
    const [selectedWriter, setSelectedWriter] = useState('');
    const [escalationReason, setEscalationReason] = useState('');
    const [rejectionReason, setRejectionReason] = useState('');

    const formatDate = (dateString) => {
        if (!dateString) return 'Not set';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const getTimeAgo = (dateString) => {
        if (!dateString) return '';
        const now = new Date();
        const date = new Date(dateString);
        const diffInHours = Math.floor((now - date) / (1000 * 60 * 60));
        
        if (diffInHours < 1) return 'Just now';
        if (diffInHours < 24) return `${diffInHours} hours ago`;
        const diffInDays = Math.floor(diffInHours / 24);
        return `${diffInDays} days ago`;
    };

    const getStatusBadge = (status) => {
        const statusConfig = {
            requested: { color: 'bg-yellow-100 text-yellow-800', label: 'Requested' },
            assigned: { color: 'bg-blue-100 text-blue-800', label: 'Assigned' },
            in_progress: { color: 'bg-purple-100 text-purple-800', label: 'In Progress' },
            completed: { color: 'bg-green-100 text-green-800', label: 'Completed' },
            rejected: { color: 'bg-red-100 text-red-800', label: 'Rejected' },
            escalated: { color: 'bg-orange-100 text-orange-800', label: 'Escalated' },
        };

        const config = statusConfig[status] || { color: 'bg-gray-100 text-gray-800', label: status };
        
        return (
            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${config.color}`}>
                {config.label}
            </span>
        );
    };

    const getRevisionTypeLabel = (type) => {
        const typeLabels = {
            quality: 'Quality Issues',
            requirements: 'Requirements Not Met',
            formatting: 'Formatting Issues',
            plagiarism: 'Plagiarism Concerns',
            deadline: 'Deadline Issues',
            communication: 'Communication Issues',
            other: 'Other',
        };
        return typeLabels[type] || type;
    };

    const handleAssignWriter = () => {
        if (!selectedWriter) return;
        
        router.post(route('admin.revisions.assign-writer', revision.id), {
            writer_id: selectedWriter,
        }, {
            onSuccess: () => {
                setShowAssignmentModal(false);
                setSelectedWriter('');
            }
        });
    };

    const handleEscalate = () => {
        if (!escalationReason.trim()) return;
        
        router.post(route('admin.revisions.escalate', revision.id), {
            escalation_reason: escalationReason,
        }, {
            onSuccess: () => {
                setShowEscalationModal(false);
                setEscalationReason('');
            }
        });
    };

    const handleReject = () => {
        if (!rejectionReason.trim()) return;
        
        router.post(route('admin.revisions.reject', revision.id), {
            rejection_reason: rejectionReason,
        }, {
            onSuccess: () => {
                setShowRejectionModal(false);
                setRejectionReason('');
                router.reload();
            }
        });
    };

    const handleQualityAssessment = (e) => {
        e.preventDefault();
        const formData = new FormData(e.target);
        
        router.post(route('admin.revisions.assess-quality', revision.id), {
            quality_rating: formData.get('quality_rating'),
            quality_notes: formData.get('quality_notes'),
        }, {
            onSuccess: () => {
                // Refresh the page to show updated quality assessment
                router.reload();
            }
        });
    };

    const isOverdue = () => {
        if (!revision.due_date || revision.status === 'completed') return false;
        return new Date() > new Date(revision.due_date);
    };

    return (
        <AdminLayout auth={auth}>
            <Head title={`Revision #${revision.id}`} />
            
            <div className="max-w-6xl mx-auto py-6">
                {/* Header */}
                <div className="bg-white shadow-sm rounded-lg mb-6">
                    <div className="px-6 py-4 border-b border-gray-200">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                                <DocumentTextIcon className="h-6 w-6 text-blue-600" />
                                <h1 className="text-xl font-semibold text-gray-900">
                                    Revision #{revision.id}
                                </h1>
                                {getStatusBadge(revision.status)}
                            </div>
                            <div className="flex items-center space-x-3">
                                <button
                                    onClick={() => router.visit(route('admin.revisions.edit', revision.id))}
                                    className="inline-flex items-center px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                >
                                    <PencilIcon className="h-4 w-4 mr-2" />
                                    Edit
                                </button>
                                <button
                                    onClick={() => router.visit(route('admin.revisions.index'))}
                                    className="text-gray-500 hover:text-gray-700 text-sm"
                                >
                                    ← Back to Revisions
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Main Content */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Revision Details */}
                        <div className="bg-white shadow-sm rounded-lg">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h2 className="text-lg font-medium text-gray-900">Revision Details</h2>
                            </div>
                            <div className="p-6 space-y-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Type:</span>
                                        <p className="text-sm text-gray-900">{getRevisionTypeLabel(revision.revision_type)}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Number:</span>
                                        <p className="text-sm text-gray-900">{revision.revision_number}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Requested:</span>
                                        <p className="text-sm text-gray-900">
                                            {formatDate(revision.requested_at)}
                                            <span className="text-gray-500 ml-2">({getTimeAgo(revision.requested_at)})</span>
                                        </p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Due Date:</span>
                                        <p className="text-sm text-gray-900">
                                            {formatDate(revision.due_date)}
                                            {isOverdue() && (
                                                <span className="ml-2 text-red-600 font-medium">(Overdue)</span>
                                            )}
                                        </p>
                                    </div>
                                </div>

                                <div>
                                    <span className="text-sm font-medium text-gray-500">Reason:</span>
                                    <p className="text-sm text-gray-900 mt-1">{revision.revision_reason}</p>
                                </div>

                                {revision.specific_instructions && (
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Instructions:</span>
                                        <p className="text-sm text-gray-900 mt-1">{revision.specific_instructions}</p>
                                    </div>
                                )}

                                {revision.client_notes && (
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Client Notes:</span>
                                        <p className="text-sm text-gray-900 mt-1">{revision.client_notes}</p>
                                    </div>
                                )}

                                {revision.admin_notes && (
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Admin Notes:</span>
                                        <p className="text-sm text-gray-900 mt-1">{revision.admin_notes}</p>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* Order Information */}
                        <div className="bg-white shadow-sm rounded-lg">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h2 className="text-lg font-medium text-gray-900">Order Information</h2>
                            </div>
                            <div className="p-6">
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Order ID:</span>
                                        <p className="text-sm text-gray-900">#{revision.order.id}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Title:</span>
                                        <p className="text-sm text-gray-900">{revision.order.title}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Status:</span>
                                        <p className="text-sm text-gray-900">{revision.order.order_status}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Client:</span>
                                        <p className="text-sm text-gray-900">{revision.order.user.name}</p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        {/* Quality Assessment */}
                        <div className="bg-white shadow-sm rounded-lg">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h2 className="text-lg font-medium text-gray-900">Quality Assessment</h2>
                            </div>
                            <div className="p-6">
                                {revision.quality_rating ? (
                                    // Show existing quality assessment
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Rating:</span>
                                            <p className="text-sm text-gray-900">{revision.quality_rating}/10</p>
                                        </div>
                                        <div>
                                            <span className="text-sm font-medium text-gray-500">Payment Impact:</span>
                                            <p className="text-sm text-gray-900">
                                                {revision.payment_impact > 0 ? `+$${revision.payment_impact}` : 
                                                 revision.payment_impact < 0 ? `-$${Math.abs(revision.payment_impact)}` : 'None'}
                                            </p>
                                        </div>
                                    </div>
                                ) : revision.status === 'completed' ? (
                                    // Show quality assessment form for completed revisions
                                    <form onSubmit={handleQualityAssessment} className="space-y-4">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                                Quality Rating (1-10)
                                            </label>
                                            <select
                                                name="quality_rating"
                                                required
                                                className="w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                                            >
                                                <option value="">Select rating</option>
                                                <option value="10">10 - Excellent</option>
                                                <option value="9">9 - Very Good</option>
                                                <option value="8">8 - Good</option>
                                                <option value="7">7 - Satisfactory</option>
                                                <option value="6">6 - Below Average</option>
                                                <option value="5">5 - Poor</option>
                                                <option value="4">4 - Very Poor</option>
                                                <option value="3">3 - Unacceptable</option>
                                                <option value="2">2 - Very Unacceptable</option>
                                                <option value="1">1 - Extremely Poor</option>
                                            </select>
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                                Assessment Notes
                                            </label>
                                            <textarea
                                                name="quality_notes"
                                                rows={3}
                                                className="w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                                                placeholder="Provide detailed feedback on the revision quality..."
                                            />
                                        </div>
                                        <button
                                            type="submit"
                                            className="w-full inline-flex items-center justify-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                        >
                                            Submit Quality Assessment
                                        </button>
                                    </form>
                                ) : (
                                    // Show message for non-completed revisions
                                    <div className="text-center py-4">
                                        <p className="text-sm text-gray-500">
                                            Quality assessment will be available once the revision is completed.
                                        </p>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>

                    {/* Sidebar */}
                    <div className="space-y-6">
                        {/* Assignment Status */}
                        <div className="bg-white shadow-sm rounded-lg">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h3 className="text-lg font-medium text-gray-900">Assignment</h3>
                            </div>
                            <div className="p-6 space-y-4">
                                {revision.assigned_to ? (
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Assigned to:</span>
                                        <p className="text-sm text-gray-900 mt-1">{revision.assigned_to.name}</p>
                                        <p className="text-xs text-gray-500">{revision.assigned_to.email}</p>
                                    </div>
                                ) : (
                                    <div className="text-center py-4">
                                        <UserIcon className="h-8 w-8 text-gray-400 mx-auto mb-2" />
                                        <p className="text-sm text-gray-500">No writer assigned</p>
                                        <button
                                            onClick={() => setShowAssignmentModal(true)}
                                            className="mt-2 inline-flex items-center px-3 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700"
                                        >
                                            Assign Writer
                                        </button>
                                    </div>
                                )}

                                {revision.assigned_to && (
                                    <div className="pt-4 border-t border-gray-200">
                                        <button
                                            onClick={() => setShowAssignmentModal(true)}
                                            className="w-full inline-flex items-center justify-center px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                        >
                                            Reassign
                                        </button>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* Actions */}
                        <div className="bg-white shadow-sm rounded-lg">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h3 className="text-lg font-medium text-gray-900">Actions</h3>
                            </div>
                            <div className="p-6 space-y-3">
                                {revision.status === 'requested' && (
                                    <button
                                        onClick={() => setShowAssignmentModal(true)}
                                        className="w-full inline-flex items-center justify-center px-3 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700"
                                    >
                                        <UserIcon className="h-4 w-4 mr-2" />
                                        Assign Writer
                                    </button>
                                )}

                                {revision.status === 'in_progress' && (
                                    <button
                                        onClick={() => setShowEscalationModal(true)}
                                        className="w-full inline-flex items-center justify-center px-3 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-orange-600 hover:bg-orange-700"
                                    >
                                        <ArrowUpIcon className="h-4 w-4 mr-2" />
                                        Escalate
                                    </button>
                                )}

                                {revision.status === 'completed' && (
                                    <button
                                        onClick={() => setShowRejectionModal(true)}
                                        className="w-full inline-flex items-center justify-center px-3 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-red-600 hover:bg-red-700"
                                    >
                                        <ExclamationTriangleIcon className="h-4 w-4 mr-2" />
                                        Reject Revision
                                    </button>
                                )}

                                <button
                                    onClick={() => router.visit(route('admin.revisions.edit', revision.id))}
                                    className="w-full inline-flex items-center justify-center px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                >
                                    <PencilIcon className="h-4 w-4 mr-2" />
                                    Edit Revision
                                </button>
                            </div>
                        </div>

                        {/* Timeline */}
                        <div className="bg-white shadow-sm rounded-lg">
                            <div className="px-6 py-4 border-b border-gray-200">
                                <h3 className="text-lg font-medium text-gray-900">Timeline</h3>
                            </div>
                            <div className="p-6">
                                <div className="space-y-3">
                                    <div className="flex items-start space-x-3">
                                        <div className="flex-shrink-0">
                                            <div className="w-2 h-2 bg-blue-600 rounded-full mt-2"></div>
                                        </div>
                                        <div className="flex-1 min-w-0">
                                            <p className="text-sm text-gray-900">Revision requested</p>
                                            <p className="text-xs text-gray-500">{getTimeAgo(revision.requested_at)}</p>
                                        </div>
                                    </div>

                                    {revision.assigned_at && (
                                        <div className="flex items-start space-x-3">
                                            <div className="flex-shrink-0">
                                                <div className="w-2 h-2 bg-green-600 rounded-full mt-2"></div>
                                            </div>
                                            <div className="flex-1 min-w-0">
                                                <p className="text-sm text-gray-900">Assigned to writer</p>
                                                <p className="text-xs text-gray-500">{getTimeAgo(revision.assigned_at)}</p>
                                            </div>
                                        </div>
                                    )}

                                    {revision.started_at && (
                                        <div className="flex items-start space-x-3">
                                            <div className="flex-shrink-0">
                                                <div className="w-2 h-2 bg-purple-600 rounded-full mt-2"></div>
                                            </div>
                                            <div className="flex-1 min-w-0">
                                                <p className="text-sm text-gray-900">Work started</p>
                                                <p className="text-xs text-gray-500">{getTimeAgo(revision.started_at)}</p>
                                            </div>
                                        </div>
                                    )}

                                    {revision.completed_at && (
                                        <div className="flex items-start space-x-3">
                                            <div className="flex-shrink-0">
                                                <div className="w-2 h-2 bg-green-600 rounded-full mt-2"></div>
                                            </div>
                                            <div className="flex-1 min-w-0">
                                                <p className="text-sm text-gray-900">Completed</p>
                                                <p className="text-xs text-gray-500">{getTimeAgo(revision.completed_at)}</p>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Assignment Modal */}
                {showAssignmentModal && (
                    <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
                        <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
                            <div className="mt-3">
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Assign Writer</h3>
                                <select
                                    value={selectedWriter}
                                    onChange={(e) => setSelectedWriter(e.target.value)}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 mb-4"
                                >
                                    <option value="">Select a writer...</option>
                                    {writers.map((writer) => (
                                        <option key={writer.id} value={writer.id}>
                                            {writer.name} ({writer.email})
                                        </option>
                                    ))}
                                </select>
                                <div className="flex justify-end space-x-3">
                                    <button
                                        onClick={() => setShowAssignmentModal(false)}
                                        className="px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={handleAssignWriter}
                                        disabled={!selectedWriter}
                                        className="px-3 py-2 bg-blue-600 border border-transparent rounded-md text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50"
                                    >
                                        Assign
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )}

                {/* Escalation Modal */}
                {showEscalationModal && (
                    <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
                        <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
                            <div className="mt-3">
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Escalate Revision</h3>
                                <textarea
                                    value={escalationReason}
                                    onChange={(e) => setEscalationReason(e.target.value)}
                                    rows={4}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 mb-4"
                                    placeholder="Reason for escalation..."
                                />
                                <div className="flex justify-end space-x-3">
                                    <button
                                        onClick={() => setShowEscalationModal(false)}
                                        className="px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={handleEscalate}
                                        disabled={!escalationReason.trim()}
                                        className="px-3 py-2 bg-orange-600 border border-transparent rounded-md text-sm font-medium text-white hover:bg-orange-700 disabled:opacity-50"
                                    >
                                        Escalate
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )}

                {/* Rejection Modal */}
                {showRejectionModal && (
                    <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
                        <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
                            <div className="mt-3">
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Reject Revision</h3>
                                <textarea
                                    value={rejectionReason}
                                    onChange={(e) => setRejectionReason(e.target.value)}
                                    rows={4}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 mb-4"
                                    placeholder="Reason for rejection..."
                                />
                                <div className="flex justify-end space-x-3">
                                    <button
                                        onClick={() => setShowRejectionModal(false)}
                                        className="px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={handleReject}
                                        disabled={!rejectionReason.trim()}
                                        className="px-3 py-2 bg-red-600 border border-transparent rounded-md text-sm font-medium text-white hover:bg-red-700 disabled:opacity-50"
                                    >
                                        Reject
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )}
            </div>
        </AdminLayout>
    );
} 