import React, { useState, useEffect } from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import { Checkbox } from "@/Components/ui/Checkbox";
import Alert from "@/Components/ui/Alert";
import {
    ArrowLeft,
    CalendarRange,
    Calendar,
    Save,
    Info,
    AlertTriangle,
} from "lucide-react";
import { Textarea } from "@/Components/ui/Textarea";

const Edit = ({ seasonalType, couponCount }) => {
    const [showHelp, setShowHelp] = useState(false);
    const [showWarning, setShowWarning] = useState(couponCount > 0);

    const { data, setData, patch, processing, errors, reset } = useForm({
        name: seasonalType.name || "",
        description: seasonalType.description || "",
        default_start_date: seasonalType.formatted_start_date || "",
        default_end_date: seasonalType.formatted_end_date || "",
        is_active: seasonalType.is_active,
        color_code: seasonalType.color_code || "#3B82F6",
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        patch(route("admin.seasonal-types.update", seasonalType.id), {
            preserveScroll: true,
        });
    };

    return (
        <AdminLayout>
            <Head title={`Edit ${seasonalType.name}`} />

            <div className="py-12">
                <div className="max-w-3xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex items-center justify-between mb-6">
                        <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                            <CalendarRange className="w-6 h-6 mr-2 text-blue-500" />
                            Edit Seasonal Type
                        </h2>
                        <Button variant="outline" size="sm" asChild>
                            <Link href={route("admin.seasonal-types.index")}>
                                <ArrowLeft className="w-4 h-4 mr-2" />
                                Back to List
                            </Link>
                        </Button>
                    </div>

                    {/* Warning if the seasonal type is being used */}
                    {showWarning && (
                        <Alert
                            variant="warning"
                            className="mb-6"
                            title="Warning: This seasonal type is in use"
                            icon={AlertTriangle}
                            dismissible
                            onDismiss={() => setShowWarning(false)}
                        >
                            <p className="text-sm">
                                This seasonal type is currently associated with{" "}
                                {couponCount}{" "}
                                {couponCount === 1 ? "coupon" : "coupons"}.
                                Changes to the name or date range may affect
                                existing promotional campaigns.
                            </p>
                        </Alert>
                    )}

                    <Card>
                        <CardHeader>
                            <CardTitle>Edit {seasonalType.name}</CardTitle>
                            <CardDescription>
                                Update the details of this seasonal type
                            </CardDescription>
                        </CardHeader>
                        <CardContent>
                            <form onSubmit={handleSubmit} className="space-y-6">
                                <div className="space-y-4">
                                    {/* Name field */}
                                    <div>
                                        <Label
                                            htmlFor="name"
                                            className="required"
                                        >
                                            Name
                                        </Label>
                                        <Input
                                            id="name"
                                            type="text"
                                            className="mt-1 block w-full"
                                            value={data.name}
                                            onChange={(e) =>
                                                setData("name", e.target.value)
                                            }
                                            required
                                            placeholder="e.g. Summer Sale, Black Friday, Holiday Season"
                                        />
                                        {errors.name && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.name}
                                            </p>
                                        )}
                                    </div>

                                    {/* Description field */}
                                    <div>
                                        <Label htmlFor="description">
                                            Description
                                        </Label>
                                        <Textarea
                                            id="description"
                                            className="mt-1 block w-full"
                                            rows={3}
                                            value={data.description || ""}
                                            onChange={(e) =>
                                                setData(
                                                    "description",
                                                    e.target.value
                                                )
                                            }
                                            placeholder="Provide details about this seasonal period"
                                        />
                                        {errors.description && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.description}
                                            </p>
                                        )}
                                    </div>

                                    {/* Date Range */}
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        {/* Start Date */}
                                        <div>
                                            <Label
                                                htmlFor="default_start_date"
                                                className="required"
                                            >
                                                Start Date
                                            </Label>
                                            <div className="mt-1 relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <Calendar className="h-5 w-5 text-gray-400" />
                                                </div>
                                                <Input
                                                    id="default_start_date"
                                                    type="date"
                                                    className="pl-10"
                                                    value={
                                                        data.default_start_date
                                                    }
                                                    onChange={(e) =>
                                                        setData(
                                                            "default_start_date",
                                                            e.target.value
                                                        )
                                                    }
                                                    required
                                                />
                                            </div>
                                            {errors.default_start_date && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.default_start_date}
                                                </p>
                                            )}
                                        </div>

                                        {/* End Date */}
                                        <div>
                                            <Label
                                                htmlFor="default_end_date"
                                                className="required"
                                            >
                                                End Date
                                            </Label>
                                            <div className="mt-1 relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <Calendar className="h-5 w-5 text-gray-400" />
                                                </div>
                                                <Input
                                                    id="default_end_date"
                                                    type="date"
                                                    className="pl-10"
                                                    value={
                                                        data.default_end_date
                                                    }
                                                    onChange={(e) =>
                                                        setData(
                                                            "default_end_date",
                                                            e.target.value
                                                        )
                                                    }
                                                    required
                                                />
                                            </div>
                                            {errors.default_end_date && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.default_end_date}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    {/* Date Range Helper */}
                                    <div>
                                        <Button
                                            type="button"
                                            variant="outline"
                                            size="sm"
                                            onClick={() =>
                                                setShowHelp(!showHelp)
                                            }
                                            className="flex items-center text-sm"
                                        >
                                            <Info className="w-4 h-4 mr-1" />
                                            {showHelp ? "Hide" : "Show"} Date
                                            Range Information
                                        </Button>

                                        {showHelp && (
                                            <Alert
                                                variant="info"
                                                className="mt-2"
                                                title="About Seasonal Date Ranges"
                                            >
                                                <div className="text-sm space-y-1">
                                                    <p>
                                                        Date ranges represent
                                                        the default period when
                                                        this season occurs each
                                                        year.
                                                    </p>
                                                    <p>
                                                        For example, a "Summer
                                                        Sale" might be from June
                                                        1 to August 31, while
                                                        "Black Friday" might be
                                                        from November 25 to
                                                        November 30.
                                                    </p>
                                                    <p>
                                                        For seasons that cross
                                                        year boundaries (like
                                                        "Winter Holiday" from
                                                        December 15 to January
                                                        15), the system will
                                                        handle this correctly.
                                                    </p>
                                                </div>
                                            </Alert>
                                        )}
                                    </div>

                                    {/* Color picker */}
                                    <div>
                                        <Label htmlFor="color_code">
                                            Color Code
                                        </Label>
                                        <div className="mt-1 flex items-center space-x-2">
                                            <Input
                                                id="color_code"
                                                type="color"
                                                className="w-12 h-12 p-1 rounded"
                                                value={data.color_code}
                                                onChange={(e) =>
                                                    setData(
                                                        "color_code",
                                                        e.target.value
                                                    )
                                                }
                                            />
                                            <Input
                                                type="text"
                                                value={data.color_code}
                                                onChange={(e) =>
                                                    setData(
                                                        "color_code",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-28"
                                                placeholder="#HEX"
                                            />
                                            <span className="text-sm text-gray-500 dark:text-gray-400">
                                                Optional: Choose a color for
                                                this seasonal type
                                            </span>
                                        </div>
                                        {errors.color_code && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.color_code}
                                            </p>
                                        )}
                                    </div>

                                    {/* Active status */}
                                    <div className="flex items-center space-x-2">
                                        <Checkbox
                                            id="is_active"
                                            checked={data.is_active}
                                            onChange={(e) =>
                                                setData(
                                                    "is_active",
                                                    e.target.checked
                                                )
                                            }
                                        />
                                        <Label
                                            htmlFor="is_active"
                                            className="cursor-pointer"
                                        >
                                            Active
                                        </Label>
                                    </div>
                                </div>

                                <div className="flex justify-end space-x-3">
                                    <Button
                                        type="button"
                                        variant="outline"
                                        onClick={() => window.history.back()}
                                    >
                                        Cancel
                                    </Button>
                                    <Button
                                        type="submit"
                                        disabled={processing}
                                        className="flex items-center"
                                    >
                                        <Save className="w-4 h-4 mr-2" />
                                        {processing
                                            ? "Saving..."
                                            : "Save Changes"}
                                    </Button>
                                </div>
                            </form>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Edit;
