import React, { useState } from "react";
import { Head } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    BarChart,
    Bar,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    Legend,
    ResponsiveContainer,
    PieChart,
    Pie,
    Cell,
} from "recharts";
import {
    Calendar,
    ChevronLeft,
    BarChart as BarChartIcon,
    CalendarRange,
} from "lucide-react";
import { Button } from "@/Components/ui/Button";
import { Card } from "@/Components/ui/Card";

const COLORS = [
    "#0088FE",
    "#00C49F",
    "#FFBB28",
    "#FF8042",
    "#A52A2A",
    "#7B68EE",
    "#20B2AA",
    "#B22222",
];

const Statistics = ({ seasonalTypes, statistics, monthlyUsage }) => {
    // Format monthly data for charts
    const formatMonthName = (monthNum) => {
        const months = [
            "Jan",
            "Feb",
            "Mar",
            "Apr",
            "May",
            "Jun",
            "Jul",
            "Aug",
            "Sep",
            "Oct",
            "Nov",
            "Dec",
        ];
        return months[monthNum - 1];
    };

    // Prepare data for monthly chart
    const monthlyData = Array.from({ length: 12 }, (_, i) => {
        const month = i + 1;
        const monthName = formatMonthName(month);

        // Find all usage entries for this month
        const entries = monthlyUsage.filter(
            (entry) => parseInt(entry.month) === month
        );

        // Create a data point with all seasonal types for this month
        const dataPoint = { name: monthName };

        // Add each seasonal type's usage to this month's data point
        entries.forEach((entry) => {
            dataPoint[entry.name] = entry.usage_count;
        });

        return dataPoint;
    });

    // Prepare data for pie chart
    const pieData = seasonalTypes.map((type) => ({
        name: type.name,
        value: type.coupons_count || 0,
        color:
            type.color_code ||
            COLORS[Math.floor(Math.random() * COLORS.length)],
    }));

    // Handle seasonal types with no coupons
    const filteredPieData = pieData.filter((item) => item.value > 0);

    return (
        <AdminLayout>
            <Head title="Seasonal Type Statistics" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-3 md:space-y-0 mb-6">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white">
                                Seasonal Type Statistics
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                View usage statistics for seasonal promotional
                                periods
                            </p>
                        </div>
                        <div>
                            <Button
                                variant="outline"
                                onClick={() => window.history.back()}
                                className="flex items-center"
                            >
                                <ChevronLeft className="w-4 h-4 mr-2" />
                                Back to Seasonal Types
                            </Button>
                        </div>
                    </div>

                    {/* Overview Stats */}
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                        <Card className="p-6">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                        Total Seasonal Types
                                    </p>
                                    <h3 className="text-3xl font-bold text-gray-900 dark:text-white mt-1">
                                        {statistics.total_types}
                                    </h3>
                                </div>
                                <div className="bg-blue-100 dark:bg-blue-900/30 p-3 rounded-full">
                                    <CalendarRange className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                                </div>
                            </div>
                            <div className="mt-4 flex space-x-4 text-sm">
                                <div>
                                    <span className="text-green-600 dark:text-green-400 font-medium">
                                        {statistics.active_types}
                                    </span>{" "}
                                    active
                                </div>
                                <div>
                                    <span className="text-purple-600 dark:text-purple-400 font-medium">
                                        {statistics.currently_active}
                                    </span>{" "}
                                    in season
                                </div>
                            </div>
                        </Card>

                        <Card className="p-6">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                        Total Coupons
                                    </p>
                                    <h3 className="text-3xl font-bold text-gray-900 dark:text-white mt-1">
                                        {statistics.total_coupons}
                                    </h3>
                                </div>
                                <div className="bg-amber-100 dark:bg-amber-900/30 p-3 rounded-full">
                                    <Calendar className="w-6 h-6 text-amber-600 dark:text-amber-400" />
                                </div>
                            </div>
                            <div className="mt-4 flex space-x-4 text-sm">
                                <div>
                                    <span className="text-blue-600 dark:text-blue-400 font-medium">
                                        {statistics.total_redemptions}
                                    </span>{" "}
                                    redemptions
                                </div>
                            </div>
                        </Card>

                        <Card className="p-6">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                        Total Discount
                                    </p>
                                    <h3 className="text-3xl font-bold text-gray-900 dark:text-white mt-1">
                                        ${statistics.total_discount.toFixed(2)}
                                    </h3>
                                </div>
                                <div className="bg-green-100 dark:bg-green-900/30 p-3 rounded-full">
                                    <BarChartIcon className="w-6 h-6 text-green-600 dark:text-green-400" />
                                </div>
                            </div>
                            <div className="mt-4 text-sm text-gray-500 dark:text-gray-400">
                                Across all seasonal promotions
                            </div>
                        </Card>
                    </div>

                    {/* Seasonal Types Table */}
                    <Card className="mb-6 overflow-hidden">
                        <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                Seasonal Type Performance
                            </h3>
                        </div>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead className="bg-gray-50 dark:bg-gray-800/50">
                                    <tr>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Seasonal Type
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Status
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Coupons
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Uses
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Total Discount
                                        </th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                    {seasonalTypes.map((type) => (
                                        <tr
                                            key={type.id}
                                            className="hover:bg-gray-50 dark:hover:bg-gray-800/50"
                                        >
                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                                <div className="flex items-center">
                                                    <div
                                                        className="w-3 h-3 rounded-full mr-2"
                                                        style={{
                                                            backgroundColor:
                                                                type.color_code ||
                                                                "#CBD5E0",
                                                        }}
                                                    ></div>
                                                    {type.name}
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                {type.is_active ? (
                                                    type.is_currently_active ? (
                                                        <span className="px-2 py-1 text-xs rounded-full bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300">
                                                            Currently Active
                                                        </span>
                                                    ) : (
                                                        <span className="px-2 py-1 text-xs rounded-full bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-300">
                                                            Active
                                                        </span>
                                                    )
                                                ) : (
                                                    <span className="px-2 py-1 text-xs rounded-full bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300">
                                                        Inactive
                                                    </span>
                                                )}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                {type.coupons_count || 0}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                {type.coupons_sum_uses_count ||
                                                    0}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                $
                                                {(
                                                    type.total_discount || 0
                                                ).toFixed(2)}
                                            </td>
                                        </tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>
                    </Card>

                    {/* Charts */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                        {/* Monthly Usage Chart */}
                        <Card className="p-6">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                Monthly Usage by Season
                            </h3>
                            <div className="h-80">
                                <ResponsiveContainer width="100%" height="100%">
                                    <BarChart
                                        data={monthlyData}
                                        margin={{
                                            top: 20,
                                            right: 30,
                                            left: 20,
                                            bottom: 5,
                                        }}
                                    >
                                        <CartesianGrid strokeDasharray="3 3" />
                                        <XAxis dataKey="name" />
                                        <YAxis />
                                        <Tooltip />
                                        <Legend />
                                        {seasonalTypes.map((type, index) => (
                                            <Bar
                                                key={type.id}
                                                dataKey={type.name}
                                                stackId="a"
                                                fill={
                                                    type.color_code ||
                                                    COLORS[
                                                        index % COLORS.length
                                                    ]
                                                }
                                            />
                                        ))}
                                    </BarChart>
                                </ResponsiveContainer>
                            </div>
                        </Card>

                        {/* Coupon Distribution Pie Chart */}
                        <Card className="p-6">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                Coupon Distribution by Season
                            </h3>
                            <div className="h-80">
                                <ResponsiveContainer width="100%" height="100%">
                                    {filteredPieData.length > 0 ? (
                                        <PieChart>
                                            <Pie
                                                data={filteredPieData}
                                                cx="50%"
                                                cy="50%"
                                                labelLine={false}
                                                label={({ name, percent }) =>
                                                    `${name} (${(
                                                        percent * 100
                                                    ).toFixed(0)}%)`
                                                }
                                                outerRadius={80}
                                                fill="#8884d8"
                                                dataKey="value"
                                            >
                                                {filteredPieData.map(
                                                    (entry, index) => (
                                                        <Cell
                                                            key={`cell-${index}`}
                                                            fill={
                                                                entry.color ||
                                                                COLORS[
                                                                    index %
                                                                        COLORS.length
                                                                ]
                                                            }
                                                        />
                                                    )
                                                )}
                                            </Pie>
                                            <Tooltip
                                                formatter={(value) => [
                                                    `${value} coupons`,
                                                    "Count",
                                                ]}
                                            />
                                        </PieChart>
                                    ) : (
                                        <div className="flex items-center justify-center h-full text-gray-500 dark:text-gray-400">
                                            No coupon data available
                                        </div>
                                    )}
                                </ResponsiveContainer>
                            </div>
                        </Card>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Statistics;
