import { FormInput } from "@/Components/ui/FormInput";

import AdminLayout from "@/Layouts/AdminLayout";
import { Link, useForm, usePage } from "@inertiajs/react";
import { useEffect } from "react";
import { toast } from "react-hot-toast";

export default function Edit({ setting }) {
    const { flash } = usePage().props;
    
    const { data, setData, put, processing, errors } = useForm({
        writer_fee: setting.writer_fee || 0,
        writer_deadline: setting.writer_deadline || 0,
        price_per_slide: setting.price_per_slide || 0,
        enable_coupon: setting.enable_coupon || false,
        enable_stripe_payment: setting.enable_stripe_payment || false,
        questions_test_duration: setting.questions_test_duration || 60,
        essay_test_duration: setting.essay_test_duration || 60,
    });

    // Handle flash messages with toast notifications
    useEffect(() => {
        // Check for both 'success' and 'message' keys
        const successMessage = flash?.success || flash?.message;
        if (successMessage) {
            toast.success(successMessage, {
                duration: 4000,
                icon: '✅',
                style: {
                    background: '#10B981',
                    color: '#fff',
                    fontWeight: '500',
                },
            });
        }
        if (flash?.error) {
            toast.error(flash.error, {
                duration: 4000,
                icon: '❌',
                style: {
                    background: '#EF4444',
                    color: '#fff',
                    fontWeight: '500',
                },
            });
        }
    }, [flash]);

    const handleSubmit = (e) => {
        e.preventDefault();
        put(route("admin.settings.update", setting.id));
    };

    return (
        <AdminLayout>
            <div className="max-w-2xl mx-auto mt-10">
                <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg">
                    <div className="px-6 py-4 border-b dark:border-gray-700">
                        <h1 className="text-md lg:text-xl font-bold text-blue-500 dark:text-blue-400">
                            Update Settings
                        </h1>
                    </div>

                    <form onSubmit={handleSubmit} className="p-6 space-y-6">
                        <FormInput
                            id="writer_fee"
                            name="writer_fee"
                            label="Writer Fee (%) (0-100)"
                            type="number"
                            value={data.writer_fee}
                            onChange={(e) => setData("writer_fee", e.target.value)}
                            error={errors.writer_fee}
                            placeholder="Enter Writer Fee"
                            required
                        />

                        <FormInput
                            id="writer_deadline"
                            name="writer_deadline"
                            label="Writer Deadline (%) (0-100)"
                            type="number"
                            value={data.writer_deadline}
                            onChange={(e) => setData("writer_deadline", e.target.value)}
                            error={errors.writer_deadline}
                            placeholder="Enter Writer Deadline (%) (0-100)"
                            required
                        />

                        <FormInput
                            id="price_per_slide"
                            name="price_per_slide"
                            label="Price Per Slide"
                            type="number"
                            value={data.price_per_slide}
                            onChange={(e) => setData("price_per_slide", e.target.value)}
                            error={errors.price_per_slide}
                            placeholder="Enter Price Per Slide"
                            required
                        />

                        {/* Questions Test Duration field */}
                        <FormInput
                            id="questions_test_duration"
                            name="questions_test_duration"
                            label="Questions Test Duration (minutes)"
                            type="number"
                            value={data.questions_test_duration}
                            onChange={(e) => setData("questions_test_duration", e.target.value)}
                            error={errors.questions_test_duration}
                            placeholder="Enter Questions Test Duration in Minutes"
                            min="1"
                            max="180"
                            required
                        />

                        {/* Add Essay Test Duration field */}
                        <FormInput
                            id="essay_test_duration"
                            name="essay_test_duration"
                            label="Essay Test Duration (minutes)"
                            type="number"
                            value={data.essay_test_duration}
                            onChange={(e) => setData("essay_test_duration", e.target.value)}
                            error={errors.essay_test_duration}
                            placeholder="Enter Essay Test Duration in Minutes"
                            min="1"
                            max="180"
                            required
                        />

                        <div className="space-y-2">
                            <label
                                htmlFor="enable_coupon"
                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                            >
                                Enable Coupon
                            </label>
                            <input
                                type="checkbox"
                                id="enable_coupon"
                                name="enable_coupon"
                                checked={data.enable_coupon}
                                onChange={(e) =>
                                    setData("enable_coupon", e.target.checked)
                                }
                                className="h-5 w-5 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                            />
                            {errors.enable_coupon && (
                                <div className="text-red-500 text-sm mt-1">
                                    {errors.enable_coupon}
                                </div>
                            )}
                        </div>

                        <div className="space-y-2">
                            <label
                                htmlFor="enable_stripe_payment"
                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                            >
                                Enable Stripe Payment
                            </label>
                            <input
                                type="checkbox"
                                id="enable_stripe_payment"
                                name="enable_stripe_payment"
                                checked={data.enable_stripe_payment}
                                onChange={(e) =>
                                    setData(
                                        "enable_stripe_payment",
                                        e.target.checked
                                    )
                                }
                                className="h-5 w-5 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                            />
                            {errors.enable_stripe_payment && (
                                <div className="text-red-500 text-sm mt-1">
                                    {errors.enable_stripe_payment}
                                </div>
                            )}
                        </div>

                        <div className="flex justify-end gap-3">
                            <Link
                                href={route("admin.settings.show", setting.id)}
                                className="px-4 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600"
                            >
                                Back
                            </Link>
                            <button
                                type="submit"
                                disabled={processing}
                                className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50"
                            >
                                Update
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
}
