import React, { useState } from "react";
import { Head, useForm, usePage } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    MessageCircle,
    BarChart3,
    Facebook,
    Code,
    Save,
    Info,
    ExternalLink,
    AlertCircle,
    CheckCircle2,
} from "lucide-react";

export default function Index({ setting }) {
    const { flash } = usePage().props;
    const [activeSection, setActiveSection] = useState("live-chat");

    const { data, setData, put, processing, errors } = useForm({
        // Google Tag Manager settings
        enable_gtm: setting.enable_gtm || false,
        gtm_id: setting.gtm_id || "",
        gtm_delay: setting.gtm_delay || 2000,

        // Tawk.to settings
        enable_tawkto: setting.enable_tawkto || false,
        tawkto_property_id: setting.tawkto_property_id || "",
        tawkto_widget_id: setting.tawkto_widget_id || "",
        tawkto_delay: setting.tawkto_delay || 5000,

        // Comm100 settings
        enable_comm100: setting.enable_comm100 || false,
        comm100_site_id: setting.comm100_site_id || "",
        comm100_code_plan: setting.comm100_code_plan || "",
        comm100_div_id: setting.comm100_div_id || "",
        comm100_delay: setting.comm100_delay || 5000,
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        put(route("admin.third-party-scripts.update"));
    };

    const scrollToSection = (sectionId) => {
        setActiveSection(sectionId);
        document.getElementById(sectionId)?.scrollIntoView({
            behavior: "smooth",
            block: "start",
        });
    };

    return (
        <AdminLayout>
            <Head title="3rd Party Scripts Management" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto">
                    {/* Header */}
                    <div className="mb-6">
                        <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                            3rd Party Scripts Management
                        </h1>
                        <p className="text-gray-600 dark:text-gray-400">
                            Manage all third-party integrations and custom scripts for your website
                        </p>
                    </div>

                    {/* Flash Message */}
                    {flash?.message && (
                        <div className="mb-6 p-4 bg-green-100 dark:bg-green-900/30 border border-green-200 dark:border-green-800 text-green-800 dark:text-green-200 rounded-lg flex items-center gap-2">
                            <CheckCircle2 size={20} />
                            <span>{flash.message}</span>
                        </div>
                    )}

                    <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
                        {/* Sidebar Navigation */}
                        <div className="lg:col-span-1">
                            <div className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-4 sticky top-6">
                                <h3 className="text-sm font-semibold text-gray-900 dark:text-white mb-3 uppercase tracking-wider">
                                    Script Categories
                                </h3>
                                <nav className="space-y-1">
                                    <button
                                        type="button"
                                        onClick={() => scrollToSection("live-chat")}
                                        className={`w-full flex items-center gap-3 px-3 py-2 rounded-md text-sm transition-colors ${
                                            activeSection === "live-chat"
                                                ? "bg-indigo-100 dark:bg-indigo-900/30 text-indigo-700 dark:text-indigo-300 font-medium"
                                                : "text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
                                        }`}
                                    >
                                        <MessageCircle size={18} />
                                        <span>Live Chat</span>
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => scrollToSection("gtm")}
                                        className={`w-full flex items-center gap-3 px-3 py-2 rounded-md text-sm transition-colors ${
                                            activeSection === "gtm"
                                                ? "bg-indigo-100 dark:bg-indigo-900/30 text-indigo-700 dark:text-indigo-300 font-medium"
                                                : "text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
                                        }`}
                                    >
                                        <BarChart3 size={18} />
                                        <span>Google Tag Manager</span>
                                    </button>
                                </nav>
                            </div>
                        </div>

                        {/* Main Content */}
                        <div className="lg:col-span-3">
                            <form onSubmit={handleSubmit} className="space-y-6">
                                {/* Live Chat Section */}
                                <section id="live-chat" className="scroll-mt-6">
                                    <h2 className="text-xl font-bold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                                        <MessageCircle size={24} />
                                        Live Chat Widgets
                                    </h2>

                                    {/* Tawk.to Card */}
                                    <div className="bg-purple-50 dark:bg-purple-900/20 border border-purple-200 dark:border-purple-800 rounded-lg p-6 mb-4">
                                        <div className="flex items-start justify-between mb-4">
                                            <div>
                                                <h3 className="text-lg font-semibold text-purple-900 dark:text-purple-100 flex items-center gap-2">
                                                    Tawk.to Live Chat
                                                </h3>
                                                <p className="text-sm text-purple-700 dark:text-purple-300 mt-1">
                                                    Free live chat software for websites
                                                </p>
                                            </div>
                                            <a
                                                href="https://tawk.to"
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="text-purple-600 dark:text-purple-400 hover:text-purple-700 dark:hover:text-purple-300"
                                            >
                                                <ExternalLink size={18} />
                                            </a>
                                        </div>

                                        {/* Enable Toggle */}
                                        <div className="mb-4">
                                            <label className="flex items-center gap-3 cursor-pointer">
                                                <input
                                                    type="checkbox"
                                                    checked={data.enable_tawkto}
                                                    onChange={(e) => {
                                                        setData("enable_tawkto", e.target.checked);
                                                        if (e.target.checked) {
                                                            setData("enable_comm100", false);
                                                        }
                                                    }}
                                                    className="w-5 h-5 text-purple-600 border-gray-300 rounded focus:ring-purple-500"
                                                />
                                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                    Enable Tawk.to Chat Widget
                                                </span>
                                            </label>
                                        </div>

                                        {/* Configuration Fields */}
                                        {data.enable_tawkto && (
                                            <div className="space-y-4 mt-4 pt-4 border-t border-purple-200 dark:border-purple-700">
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        Property ID *
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={data.tawkto_property_id}
                                                        onChange={(e) =>
                                                            setData("tawkto_property_id", e.target.value)
                                                        }
                                                        placeholder="e.g., 68544bc0562852190e9a1841"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-purple-500"
                                                    />
                                                    {errors.tawkto_property_id && (
                                                        <p className="text-red-600 text-sm mt-1">
                                                            {errors.tawkto_property_id}
                                                        </p>
                                                    )}
                                                </div>

                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        Widget ID *
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={data.tawkto_widget_id}
                                                        onChange={(e) =>
                                                            setData("tawkto_widget_id", e.target.value)
                                                        }
                                                        placeholder="e.g., 1j3t97cdq"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-purple-500"
                                                    />
                                                    {errors.tawkto_widget_id && (
                                                        <p className="text-red-600 text-sm mt-1">
                                                            {errors.tawkto_widget_id}
                                                        </p>
                                                    )}
                                                </div>

                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        Load Delay (milliseconds)
                                                    </label>
                                                    <input
                                                        type="number"
                                                        value={data.tawkto_delay}
                                                        onChange={(e) =>
                                                            setData("tawkto_delay", e.target.value)
                                                        }
                                                        placeholder="5000"
                                                        min="0"
                                                        max="30000"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-purple-500"
                                                    />
                                                    <p className="text-xs text-purple-600 dark:text-purple-400 mt-1 flex items-center gap-1">
                                                        <Info size={12} />
                                                        Delayed loading improves page performance
                                                    </p>
                                                </div>
                                            </div>
                                        )}
                                    </div>

                                    {/* Comm100 Card */}
                                    <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-6">
                                        <div className="flex items-start justify-between mb-4">
                                            <div>
                                                <h3 className="text-lg font-semibold text-green-900 dark:text-green-100 flex items-center gap-2">
                                                    Comm100 Live Chat
                                                </h3>
                                                <p className="text-sm text-green-700 dark:text-green-300 mt-1">
                                                    Enterprise live chat solution
                                                </p>
                                            </div>
                                            <a
                                                href="https://comm100.com"
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="text-green-600 dark:text-green-400 hover:text-green-700 dark:hover:text-green-300"
                                            >
                                                <ExternalLink size={18} />
                                            </a>
                                        </div>

                                        {/* Mutual Exclusion Warning */}
                                        {data.enable_tawkto && (
                                            <div className="mb-4 p-3 bg-amber-100 dark:bg-amber-900/30 border border-amber-200 dark:border-amber-800 rounded-lg flex items-center gap-2">
                                                <AlertCircle size={16} className="text-amber-600 dark:text-amber-400" />
                                                <p className="text-sm text-amber-800 dark:text-amber-200">
                                                    Tawk.to is currently enabled. Only one chat widget can be active at a time.
                                                </p>
                                            </div>
                                        )}

                                        {/* Enable Toggle */}
                                        <div className="mb-4">
                                            <label className="flex items-center gap-3 cursor-pointer">
                                                <input
                                                    type="checkbox"
                                                    checked={data.enable_comm100}
                                                    onChange={(e) => {
                                                        setData("enable_comm100", e.target.checked);
                                                        if (e.target.checked) {
                                                            setData("enable_tawkto", false);
                                                        }
                                                    }}
                                                    className="w-5 h-5 text-green-600 border-gray-300 rounded focus:ring-green-500"
                                                />
                                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                    Enable Comm100 Chat Widget
                                                </span>
                                            </label>
                                        </div>

                                        {/* Configuration Fields */}
                                        {data.enable_comm100 && (
                                            <div className="space-y-4 mt-4 pt-4 border-t border-green-200 dark:border-green-700">
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        Site ID *
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={data.comm100_site_id}
                                                        onChange={(e) =>
                                                            setData("comm100_site_id", e.target.value)
                                                        }
                                                        placeholder="e.g., 10000123"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-green-500"
                                                    />
                                                    {errors.comm100_site_id && (
                                                        <p className="text-red-600 text-sm mt-1">
                                                            {errors.comm100_site_id}
                                                        </p>
                                                    )}
                                                </div>

                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        Code Plan *
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={data.comm100_code_plan}
                                                        onChange={(e) =>
                                                            setData("comm100_code_plan", e.target.value)
                                                        }
                                                        placeholder="e.g., 987"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-green-500"
                                                    />
                                                    {errors.comm100_code_plan && (
                                                        <p className="text-red-600 text-sm mt-1">
                                                            {errors.comm100_code_plan}
                                                        </p>
                                                    )}
                                                </div>

                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        Div ID *
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={data.comm100_div_id}
                                                        onChange={(e) =>
                                                            setData("comm100_div_id", e.target.value)
                                                        }
                                                        placeholder="e.g., comm100-container"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-green-500"
                                                    />
                                                    {errors.comm100_div_id && (
                                                        <p className="text-red-600 text-sm mt-1">
                                                            {errors.comm100_div_id}
                                                        </p>
                                                    )}
                                                </div>

                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        Load Delay (milliseconds)
                                                    </label>
                                                    <input
                                                        type="number"
                                                        value={data.comm100_delay}
                                                        onChange={(e) =>
                                                            setData("comm100_delay", e.target.value)
                                                        }
                                                        placeholder="5000"
                                                        min="0"
                                                        max="30000"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-green-500"
                                                    />
                                                    <p className="text-xs text-green-600 dark:text-green-400 mt-1 flex items-center gap-1">
                                                        <Info size={12} />
                                                        Delayed loading improves page performance
                                                    </p>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </section>

                                {/* Google Tag Manager Section */}
                                <section id="gtm" className="scroll-mt-6">
                                    <h2 className="text-xl font-bold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                                        <BarChart3 size={24} />
                                        Google Tag Manager
                                    </h2>

                                    {/* Google Tag Manager Card */}
                                    <div className="bg-cyan-50 dark:bg-cyan-900/20 border border-cyan-200 dark:border-cyan-800 rounded-lg p-6 mb-4">
                                        <div className="flex items-start justify-between mb-4">
                                            <div>
                                                <h3 className="text-lg font-semibold text-cyan-900 dark:text-cyan-100">
                                                    Google Tag Manager
                                                </h3>
                                                <p className="text-sm text-cyan-700 dark:text-cyan-300 mt-1">
                                                    Manage all marketing tags without editing code
                                                </p>
                                            </div>
                                            <a
                                                href="https://tagmanager.google.com"
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="text-cyan-600 dark:text-cyan-400 hover:text-cyan-700 dark:hover:text-cyan-300"
                                            >
                                                <ExternalLink size={18} />
                                            </a>
                                        </div>

                                        <div className="mb-4">
                                            <label className="flex items-center gap-3 cursor-pointer">
                                                <input
                                                    type="checkbox"
                                                    checked={data.enable_gtm}
                                                    onChange={(e) =>
                                                        setData("enable_gtm", e.target.checked)
                                                    }
                                                    className="w-5 h-5 text-cyan-600 border-gray-300 rounded focus:ring-cyan-500"
                                                />
                                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                    Enable Google Tag Manager
                                                </span>
                                            </label>
                                        </div>

                                        {data.enable_gtm && (
                                            <div className="space-y-4 mt-4 pt-4 border-t border-cyan-200 dark:border-cyan-700">
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        GTM Container ID *
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={data.gtm_id}
                                                        onChange={(e) =>
                                                            setData("gtm_id", e.target.value)
                                                        }
                                                        placeholder="e.g., GTM-XXXXXXX"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-cyan-500"
                                                    />
                                                    {errors.gtm_id && (
                                                        <p className="text-red-600 text-sm mt-1">
                                                            {errors.gtm_id}
                                                        </p>
                                                    )}
                                                </div>

                                                <div>
                                                    <label className="block text-sm font-medium text-gray-900 dark:text-white mb-1">
                                                        Load Delay (milliseconds)
                                                    </label>
                                                    <input
                                                        type="number"
                                                        value={data.gtm_delay}
                                                        onChange={(e) =>
                                                            setData("gtm_delay", e.target.value)
                                                        }
                                                        placeholder="2000"
                                                        min="0"
                                                        max="30000"
                                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-cyan-500"
                                                    />
                                                    <p className="text-xs text-cyan-600 dark:text-cyan-400 mt-1 flex items-center gap-1">
                                                        <Info size={12} />
                                                        Delayed loading improves page performance
                                                    </p>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </section>

                                {/* Save Button */}
                                <div className="sticky bottom-6 z-10">
                                    <div className="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg shadow-lg p-4">
                                        <div className="flex items-center justify-between">
                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                Make sure to save your changes before leaving this page
                                            </p>
                                            <button
                                                type="submit"
                                                disabled={processing}
                                                className="flex items-center gap-2 px-6 py-2.5 bg-indigo-600 hover:bg-indigo-700 disabled:bg-gray-400 text-white font-medium rounded-lg transition-colors"
                                            >
                                                <Save size={18} />
                                                {processing ? "Saving..." : "Save Changes"}
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
