import AdminLayout from "@/Layouts/AdminLayout";
import ExpiresIn from "@/Components/ExpiresIn";
import { Package2, Clock, MoveLeft, Eye } from "lucide-react";
import { useState, useEffect } from "react";
import { Link } from "@inertiajs/react";
import CleanPagination from "@/Components/CleanPagination";

export default function UserOrders({ user, orders }) {
    const [isMobile, setIsMobile] = useState(window.innerWidth < 768);

    // Handle responsive layout
    useEffect(() => {
        const handleResize = () => {
            setIsMobile(window.innerWidth < 768);
        };
        window.addEventListener("resize", handleResize);
        return () => window.removeEventListener("resize", handleResize);
    }, []);

    // Get status color based on order_status
    const getStatusColor = (status) => {
        switch (status?.toLowerCase()) {
            case "completed":
                return "bg-orange-100 text-green-800 dark:bg-green-900 dark:text-green-300";
            case "pending":
            case "in progress":
                return "bg-yellow-600 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300";
            case "cancelled":
                return "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300";
            default:
                return "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300";
        }
    };

    // Format date to more readable format
    const formatDate = (dateString) => {
        if (!dateString) return "N/A";
        return new Date(dateString).toLocaleString();
    };

    return (
        <AdminLayout>
            <div className=" dark:bg-gray-900 transition-colors duration-200">
                <div className="flex justify-start items-center mb-2">
                    <div className="flex items-center gap-3">
                        <h1 className="text-lg font-bold text-gray-800 dark:text-gray-200">
                            {user?.name} Orders
                        </h1>
                    </div>
                </div>

                <div className="flex justify-end items-center mb-4">
                    <Link
                        href="/admin/dashboard"
                        className="flex items-center px-3 py-1.5 sm:px-4 sm:py-2 text-indigo-600 dark:text-gray-100 bg-indigo-50 dark:bg-indigo-900 hover:bg-indigo-100 dark:hover:bg-indigo-800 rounded-full shadow-sm border border-indigo-200 dark:border-indigo-700 transition-colors duration-200 text-sm"
                    >
                        <MoveLeft className="mr-1 h-3 w-3 sm:h-4 sm:w-4" />
                        <span className="font-medium">Back</span>
                    </Link>
                </div>

                {!orders?.data?.length ? (
                    <div className="flex flex-col items-center justify-center p-8 bg-white dark:bg-gray-800 rounded-lg shadow dark:shadow-gray-700 transition-colors">
                        <Package2 className="w-16 h-16 text-gray-400 dark:text-gray-500 mb-4" />
                        <p className="text-gray-600 dark:text-gray-400 text-lg">
                            No orders found for this user
                        </p>
                    </div>
                ) : (
                    <>
                        {/* Desktop Table View
                        <div className={`${isMobile ? "hidden" : "block"}`}>
                            <div className="overflow-x-auto bg-white dark:bg-gray-800 rounded-lg shadow dark:shadow-gray-700 transition-colors"></div>
                        </div> */}

                        {orders.data.length > 0 ? (
                            <>
                                <div className="hidden lg:flex bg-indigo-600 dark:bg-gray-700 rounded-t-lg p-4 font-bold text-gray-100 dark:text-gray-300">
                                    <div className="w-1/6">Order Number</div>
                                    <div className="w-1/6">Posted On</div>
                                    <div className="w-1/6">Time Left</div>
                                    <div className="w-1/6">Order Status</div>
                                    <div className="w-1/6">Payment Status</div>
                                    <div className="w-1/6">Amount</div>
                                    <div className="w-1/6 text-center">
                                        Action
                                    </div>
                                </div>

                                <div className="space-y-4 lg:space-y-2">
                                    {orders.data.map((order) => (
                                        <div
                                            key={order.id}
                                            className="bg-gray-200 dark:bg-gray-900 shadow rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden"
                                        >
                                            <div className="hidden lg:flex items-center p-4 hover:bg-gray-100 dark:hover:bg-gray-800">
                                                <div className="w-1/6 font-medium text-indigo-600 dark:text-indigo-400 hover:text-indigo-500 dark:hover:text-indigo-300">
                                                    <Link
                                                        href={`/orders/${order.id}`}
                                                    >
                                                        {order.order_number}
                                                    </Link>
                                                </div>
                                                <div className="w-1/6 font-mono text-sm text-gray-600 dark:text-gray-400">
                                                    {new Date(
                                                        order.dateposted
                                                    ).toLocaleString("en-US", {
                                                        year: "numeric",
                                                        month: "short",
                                                        day: "numeric",
                                                        hour: "2-digit",
                                                        minute: "2-digit",
                                                    })}
                                                </div>

                                                <div className="w-1/6 ">
                                                    <ExpiresIn
                                                        deadline={
                                                            order.deadline
                                                        }
                                                    />
                                                </div>
                                                <div className="w-1/6">
                                                    <span
                                                        className={`px-3 py-2 rounded-full text-sm font-medium ${
                                                            order.order_status ===
                                                            "completed"
                                                                ? "bg-green-500 text-green-50"
                                                                : order.order_status ===
                                                                  "pending"
                                                                ? "bg-yellow-500 text-yellow-100"
                                                                : order.order_status ===
                                                                  "cancelled"
                                                                ? "bg-red-700 text-red-50"
                                                                : order.order_status ===
                                                                  "approved"
                                                                ? "bg-blue-500 text-blue-100"
                                                                : order.order_status ===
                                                                  "bidding"
                                                                ? "bg-purple-500 text-purple-100"
                                                                : "bg-gray-500 text-gray-100"
                                                        }`}
                                                    >
                                                        {order.order_status}
                                                    </span>
                                                </div>
                                                <div className="w-1/6">
                                                    <span
                                                        className={`px-3 py-2 rounded-full text-sm font-medium ${
                                                            order.payment_status
                                                                .trim()
                                                                .toLowerCase() ===
                                                            "paid"
                                                                ? "bg-green-500 text-green-50"
                                                                : order.payment_status
                                                                      .trim()
                                                                      .toLowerCase() ===
                                                                  "unpaid"
                                                                ? "bg-yellow-500 text-yellow-100"
                                                                : "bg-red-700 text-red-100"
                                                        }`}
                                                    >
                                                        {order.payment_status
                                                            .trim()
                                                            .toUpperCase()}
                                                    </span>
                                                </div>
                                                <div className="w-1/6">
                                                    $
                                                    {parseFloat(
                                                        order.net_amount
                                                    ).toFixed(2)}
                                                </div>
                                                <div className="w-1/6">
                                                    <Link
                                                        href={`/admin/orders/${order.id}`}
                                                    >
                                                        <button className="w-full bg-indigo-600 dark:bg-indigo-500 text-white py-2 rounded-md hover:bg-indigo-700 dark:hover:bg-indigo-400 transition-colors duration-200">
                                                            <Eye className="w-6 h-6 inline-block mr-2" />
                                                        </button>
                                                    </Link>
                                                </div>
                                            </div>

                                            {/* small screen view */}
                                            <div className="lg:hidden p-4">
                                                <div className="flex justify-between items-center mb-4">
                                                    <h2 className="text-lg font-semibold text-indigo-600 dark:text-indigo-400">
                                                        <Link
                                                            href={`/orders/${order.id}`}
                                                        >
                                                            Order #
                                                            {order.order_number}
                                                        </Link>
                                                    </h2>
                                                    <span
                                                        className={`px-3 py-1 rounded-full text-sm font-medium ${
                                                            order.order_status ===
                                                            "completed"
                                                                ? "bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200"
                                                                : order.order_status ===
                                                                  "assigned"
                                                                ? "bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200"
                                                                : "bg-yellow-400 dark:bg-yellow-800 text-gray-800 dark:text-gray-100"
                                                        }`}
                                                    >
                                                        {order.order_status}
                                                    </span>
                                                </div>
                                                <div className="space-y-2 text-sm">
                                                    <div className="flex justify-between mb-2">
                                                        <span className="text-gray-900 dark:text-gray-400">
                                                            Date Posted :
                                                        </span>
                                                        <span>
                                                            {new Date(
                                                                order.dateposted
                                                            ).toLocaleString(
                                                                "en-US",
                                                                {
                                                                    year: "numeric",
                                                                    month: "short",
                                                                    day: "numeric",
                                                                    hour: "2-digit",
                                                                    minute: "2-digit",
                                                                }
                                                            )}
                                                        </span>
                                                    </div>
                                                    <div className="flex justify-between mb-2">
                                                        <span className="text-gray-500 dark:text-gray-400">
                                                            Payment Status:
                                                        </span>
                                                        <span
                                                            className={`px-3 py-1 rounded-full text-sm font-medium ${
                                                                order.payment_status ===
                                                                "paid"
                                                                    ? "bg-green-600 dark:bg-green-900 text-green-800 dark:text-green-200"
                                                                    : "bg-red-600  dark:bg-yellow-900 text-gray-100 dark:text-yellow-200"
                                                            }`}
                                                        >
                                                            {
                                                                order.payment_status
                                                            }
                                                        </span>
                                                    </div>
                                                    <div className="flex justify-between">
                                                        <span className="text-gray-500 dark:text-gray-400">
                                                            Expires On
                                                        </span>
                                                        <span className="font-medium">
                                                            {new Date(
                                                                order.deadline
                                                            ).toLocaleString(
                                                                "en-US",
                                                                {
                                                                    year: "numeric",
                                                                    month: "short",
                                                                    day: "numeric",
                                                                    hour: "2-digit",
                                                                    minute: "2-digit",
                                                                }
                                                            )}
                                                        </span>
                                                    </div>
                                                    <div className="flex justify-between">
                                                        <span className="text-gray-500 dark:text-gray-400">
                                                            Total:
                                                        </span>
                                                        <span className="font-medium">
                                                            $
                                                            {parseFloat(
                                                                order.order_amount
                                                            ).toFixed(2)}
                                                        </span>
                                                    </div>
                                                </div>
                                                <div className="mt-4">
                                                    <Link
                                                        href={`/admin/orders/${order.id}`}
                                                    >
                                                        <button className="w-full bg-indigo-600 dark:bg-indigo-500 text-white py-2 rounded-md hover:bg-indigo-700 dark:hover:bg-indigo-400 transition-colors duration-200">
                                                            View Order
                                                        </button>
                                                    </Link>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </>
                        ) : (
                            <div className="text-center p-6 bg-gray-50 dark:bg-gray-800 rounded-lg shadow-md">
                                <p className="text-gray-500 dark:text-gray-300">
                                    You have no orders yet.
                                </p>
                                <Link
                                    href={route("order.create")}
                                    className="mt-4 inline-block bg-indigo-600 dark:bg-indigo-500 text-white px-4 py-2 rounded-md hover:bg-indigo-700 dark:hover:bg-indigo-400 transition-colors duration-200"
                                >
                                    Create an Order
                                </Link>
                            </div>
                        )}

                        {/* Pagination */}
                        <CleanPagination
                            links={orders.links}
                            total={orders.total}
                            perPage={orders.per_page}
                            currentPage={orders.current_page}
                        />
                    </>
                )}
            </div>
        </AdminLayout>
    );
}
