import { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { 
    AlertTriangle, 
    Plus, 
    Search, 
    Filter, 
    Download, 
    BarChart3,
    Eye,
    Edit,
    Trash2,
    Calendar,
    User,
    FileText,
    AlertCircle,
    CheckCircle,
    Clock,
    XCircle
} from 'lucide-react';
import CleanPagination from '@/Components/CleanPagination';

export default function WriterInfractionsIndex({ 
    infractions, 
    writers, 
    statuses, 
    severities, 
    types,
    filters 
}) {
    const [searchQuery, setSearchQuery] = useState(filters?.search || '');
    const [selectedFilters, setSelectedFilters] = useState({
        writer_id: filters?.writer_id || '',
        status: filters?.status || '',
        severity: filters?.severity || '',
        type: filters?.type || '',
        date_from: filters?.date_from || '',
        date_to: filters?.date_to || '',
        escalated: filters?.escalated || ''
    });

    const handleFilterChange = (key, value) => {
        setSelectedFilters(prev => ({ ...prev, [key]: value }));
    };

    const applyFilters = () => {
        router.get('/admin/writer-infractions', selectedFilters, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const clearFilters = () => {
        const clearedFilters = {
            writer_id: '',
            status: '',
            severity: '',
            type: '',
            date_from: '',
            date_to: '',
            escalated: ''
        };
        setSelectedFilters(clearedFilters);
        router.get('/admin/writer-infractions', clearedFilters, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'pending':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'under_review':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'resolved':
                return 'bg-green-100 text-green-800 border-green-200';
            case 'dismissed':
                return 'bg-gray-100 text-gray-800 border-gray-200';
            case 'escalated':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const getSeverityColor = (severity) => {
        switch (severity) {
            case 'warning':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'minor':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'major':
                return 'bg-orange-100 text-orange-800 border-orange-200';
            case 'critical':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const getInfractionTypeLabel = (type) => {
        const typeLabels = {
            'deadline_violation': 'Deadline Violation',
            'quality_issues': 'Quality Issues',
            'plagiarism': 'Plagiarism',
            'communication_failure': 'Communication Failure',
            'policy_violation': 'Policy Violation',
            'unprofessional_behavior': 'Unprofessional Behavior',
            'client_complaint': 'Client Complaint',
            'other': 'Other'
        };
        return typeLabels[type] || type;
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'pending':
                return <Clock size={16} className="text-yellow-600" />;
            case 'under_review':
                return <AlertCircle size={16} className="text-blue-600" />;
            case 'resolved':
                return <CheckCircle size={16} className="text-green-600" />;
            case 'dismissed':
                return <XCircle size={16} className="text-gray-600" />;
            case 'escalated':
                return <AlertTriangle size={16} className="text-red-600" />;
            default:
                return <Clock size={16} className="text-gray-600" />;
        }
    };

    return (
        <AdminLayout>
            <Head title="Writer Infractions" />
            
            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div>
                                <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                                    Writer Infractions
                                </h1>
                                <p className="mt-2 text-gray-600 dark:text-gray-400">
                                    Manage and track writer infractions, violations, and disciplinary actions
                                </p>
                            </div>
                            <div className="flex items-center gap-3">
                                <Link
                                    href={route('admin.writer-infractions.statistics')}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:bg-gray-800 dark:text-white dark:border-gray-600 dark:hover:bg-gray-700"
                                >
                                    <BarChart3 size={16} className="mr-2" />
                                    Statistics
                                </Link>
                                <Link
                                    href={route('admin.writer-infractions.export')}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:bg-gray-800 dark:text-white dark:border-gray-600 dark:hover:bg-gray-700"
                                >
                                    <Download size={16} className="mr-2" />
                                    Export
                                </Link>
                                <Link
                                    href={route('admin.writer-infractions.create')}
                                    className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                >
                                    <Plus size={16} className="mr-2" />
                                    Report Infraction
                                </Link>
                            </div>
                        </div>
                    </div>

                    {/* Filters */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-6 mb-6">
                        <div className="flex items-center gap-4 mb-4">
                            <Filter size={20} className="text-gray-500" />
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">Filters</h3>
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                            {/* Search */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Search
                                </label>
                                <div className="relative">
                                    <Search size={16} className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
                                    <input
                                        type="text"
                                        placeholder="Search infractions..."
                                        className="w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                        value={searchQuery}
                                        onChange={(e) => setSearchQuery(e.target.value)}
                                    />
                                </div>
                            </div>

                            {/* Writer Filter */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Writer
                                </label>
                                <select
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    value={selectedFilters.writer_id}
                                    onChange={(e) => handleFilterChange('writer_id', e.target.value)}
                                >
                                    <option value="">All Writers</option>
                                    {writers.map(writer => (
                                        <option key={writer.id} value={writer.id}>
                                            {writer.name}
                                        </option>
                                    ))}
                                </select>
                            </div>

                            {/* Status Filter */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Status
                                </label>
                                <select
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    value={selectedFilters.status}
                                    onChange={(e) => handleFilterChange('status', e.target.value)}
                                >
                                    <option value="">All Statuses</option>
                                    {statuses.map(status => (
                                        <option key={status} value={status}>
                                            {status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                                        </option>
                                    ))}
                                </select>
                            </div>

                            {/* Severity Filter */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Severity
                                </label>
                                <select
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    value={selectedFilters.severity}
                                    onChange={(e) => handleFilterChange('severity', e.target.value)}
                                >
                                    <option value="">All Severities</option>
                                    {severities.map(severity => (
                                        <option key={severity} value={severity}>
                                            {severity.charAt(0).toUpperCase() + severity.slice(1)}
                                        </option>
                                    ))}
                                </select>
                            </div>

                            {/* Type Filter */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Type
                                </label>
                                <select
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    value={selectedFilters.type}
                                    onChange={(e) => handleFilterChange('type', e.target.value)}
                                >
                                    <option value="">All Types</option>
                                    {types.map(type => (
                                        <option key={type} value={type}>
                                            {getInfractionTypeLabel(type)}
                                        </option>
                                    ))}
                                </select>
                            </div>

                            {/* Date From */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Date From
                                </label>
                                <input
                                    type="date"
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    value={selectedFilters.date_from}
                                    onChange={(e) => handleFilterChange('date_from', e.target.value)}
                                />
                            </div>

                            {/* Date To */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Date To
                                </label>
                                <input
                                    type="date"
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    value={selectedFilters.date_to}
                                    onChange={(e) => handleFilterChange('date_to', e.target.value)}
                                />
                            </div>

                            {/* Escalated Filter */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Escalated
                                </label>
                                <select
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    value={selectedFilters.escalated}
                                    onChange={(e) => handleFilterChange('escalated', e.target.value)}
                                >
                                    <option value="">All</option>
                                    <option value="true">Yes</option>
                                    <option value="false">No</option>
                                </select>
                            </div>
                        </div>

                        <div className="flex items-center gap-3 mt-4">
                            <button
                                onClick={applyFilters}
                                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                            >
                                Apply Filters
                            </button>
                            <button
                                onClick={clearFilters}
                                className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:bg-gray-800 dark:text-white dark:border-gray-600 dark:hover:bg-gray-700"
                            >
                                Clear Filters
                            </button>
                        </div>
                    </div>

                    {/* Infractions List */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md overflow-hidden">
                        <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                Infractions ({infractions.total})
                            </h3>
                        </div>

                        {infractions.data.length === 0 ? (
                            <div className="text-center py-12">
                                <AlertTriangle size={48} className="mx-auto text-gray-400 mb-4" />
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                                    No infractions found
                                </h3>
                                <p className="text-gray-600 dark:text-gray-400 mb-6">
                                    {Object.values(selectedFilters).some(v => v) 
                                        ? 'Try adjusting your filters or search criteria.'
                                        : 'No writer infractions have been reported yet.'
                                    }
                                </p>
                                {!Object.values(selectedFilters).some(v => v) && (
                                    <Link
                                        href={route('admin.writer-infractions.create')}
                                        className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                    >
                                        <Plus size={16} className="mr-2" />
                                        Report First Infraction
                                    </Link>
                                )}
                            </div>
                        ) : (
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                    <thead className="bg-gray-50 dark:bg-gray-700">
                                        <tr>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                ID
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                Writer
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                Type
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                Severity
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                Status
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                Reported
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                Actions
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                        {infractions.data.map(infraction => (
                                            <tr key={infraction.id} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                                                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                                    #{infraction.id}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div className="flex items-center">
                                                        <User size={16} className="text-gray-400 mr-2" />
                                                        <span className="text-sm text-gray-900 dark:text-white">
                                                            {infraction.writer?.name || 'Unknown'}
                                                        </span>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <span className="text-sm text-gray-900 dark:text-white">
                                                        {getInfractionTypeLabel(infraction.infraction_type)}
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getSeverityColor(infraction.severity_level)}`}>
                                                        {infraction.severity_level.charAt(0).toUpperCase() + infraction.severity_level.slice(1)}
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(infraction.status)}`}>
                                                        {getStatusIcon(infraction.status)}
                                                        <span className="ml-1">
                                                            {infraction.status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                                                        </span>
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div className="flex items-center">
                                                        <Calendar size={16} className="text-gray-400 mr-2" />
                                                        <span className="text-sm text-gray-900 dark:text-white">
                                                            {new Date(infraction.reported_at).toLocaleDateString()}
                                                        </span>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                    <div className="flex items-center gap-2">
                                                        <Link
                                                            href={route('admin.writer-infractions.show', infraction.id)}
                                                            className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300"
                                                        >
                                                            <Eye size={16} />
                                                        </Link>
                                                        <Link
                                                            href={route('admin.writer-infractions.edit', infraction.id)}
                                                            className="text-blue-600 hover:text-blue-900 dark:text-blue-400 dark:hover:text-blue-300"
                                                        >
                                                            <Edit size={16} />
                                                        </Link>
                                                        <button
                                                            onClick={() => {
                                                                if (confirm('Are you sure you want to delete this infraction?')) {
                                                                    router.delete(route('admin.writer-infractions.destroy', infraction.id));
                                                                }
                                                            }}
                                                            className="text-red-600 hover:text-red-900 dark:text-red-400 dark:hover:text-red-300"
                                                        >
                                                            <Trash2 size={16} />
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        )}

                        {/* Pagination */}
                        {infractions.links && infractions.links.length > 3 && (
                            <div className="px-6 py-4 border-t border-gray-200 dark:border-gray-700">
                                <CleanPagination links={infractions.links} />
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
} 