import { useState } from "react";
import { Head, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import Pagination from "@/Components/Pagination";
import {
    Search,
    Filter,
    RefreshCw,
    ChevronUp,
    ChevronDown,
    CheckCircle,
    XCircle,
    Clock,
    FileText,
    ArrowRight,
} from "lucide-react";
import { Transition } from "@headlessui/react";

export default function Index({ writers, filters, flash }) {
    const [showFilters, setShowFilters] = useState(false);
    const [searchParams, setSearchParams] = useState({
        search: filters.search || "",
        status: filters.status || "all",
        sort_field: filters.sort_field || "updated_at",
        sort_direction: filters.sort_direction || "desc",
    });

    const handleSearch = (e) => {
        e.preventDefault();
        applyFilters();
    };

    const resetFilters = () => {
        setSearchParams({
            search: "",
            status: "all",
            sort_field: "updated_at",
            sort_direction: "desc",
        });

        window.location.href = route("admin.writers.tests");
    };

    const applyFilters = () => {
        window.location.href = route("admin.writers.tests", searchParams);
    };

    const toggleSort = (field) => {
        const direction =
            field === searchParams.sort_field &&
            searchParams.sort_direction === "asc"
                ? "desc"
                : "asc";

        setSearchParams({
            ...searchParams,
            sort_field: field,
            sort_direction: direction,
        });

        window.location.href = route("admin.writers.tests", {
            ...searchParams,
            sort_field: field,
            sort_direction: direction,
        });
    };

    const getSortIcon = (field) => {
        if (field !== searchParams.sort_field) {
            return null;
        }

        return searchParams.sort_direction === "asc" ? (
            <ChevronUp className="h-4 w-4 ml-1" />
        ) : (
            <ChevronDown className="h-4 w-4 ml-1" />
        );
    };

    const getStatusBadge = (status, score) => {
        if (status === "passed") {
            return (
                <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200">
                    Passed
                </span>
            );
        }

        if (score >= 90) {
            return (
                <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200">
                    Passed MCQ
                </span>
            );
        }

        return (
            <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-200">
                Failed MCQ
            </span>
        );
    };

    const getTestStatusIcon = (status, score) => {
        if (status === "passed") {
            return (
                <CheckCircle className="h-5 w-5 text-green-500 dark:text-green-400" />
            );
        }

        if (status === "questions_failed" || status === "essay_failed") {
            return (
                <XCircle className="h-5 w-5 text-red-500 dark:text-red-400" />
            );
        }

        if (status === "essay_submitted") {
            return (
                <Clock className="h-5 w-5 text-yellow-500 dark:text-yellow-400" />
            );
        }

        if (status === "essay_pending") {
            return (
                <FileText className="h-5 w-5 text-blue-500 dark:text-blue-400" />
            );
        }

        return null;
    };

    const formatDate = (dateString) => {
        if (!dateString) return "—";
        return new Date(dateString).toLocaleString();
    };

    return (
        <AdminLayout>
            <Head title="Writer Test Review" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex justify-between items-center mb-6">
                        <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                            Writer Test Results
                        </h1>
                    </div>

                    {flash.success && (
                        <div className="mb-6 bg-green-100 dark:bg-green-900/30 border-l-4 border-green-500 text-green-700 dark:text-green-300 p-4 rounded">
                            <p>{flash.success}</p>
                        </div>
                    )}

                    {flash.error && (
                        <div className="mb-6 bg-red-100 dark:bg-red-900/30 border-l-4 border-red-500 text-red-700 dark:text-red-300 p-4 rounded">
                            <p>{flash.error}</p>
                        </div>
                    )}

                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6 gap-4">
                                <form
                                    onSubmit={handleSearch}
                                    className="flex items-center flex-1"
                                >
                                    <div className="relative flex-1">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <Search className="h-5 w-5 text-gray-400" />
                                        </div>
                                        <input
                                            type="text"
                                            className="block w-full pl-10 pr-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md leading-5 bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-100 placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm transition-colors"
                                            placeholder="Search writers..."
                                            value={searchParams.search}
                                            onChange={(e) =>
                                                setSearchParams({
                                                    ...searchParams,
                                                    search: e.target.value,
                                                })
                                            }
                                        />
                                    </div>
                                    <button
                                        type="submit"
                                        className="ml-3 px-3 py-2 bg-blue-600 dark:bg-blue-500 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                    >
                                        Search
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() =>
                                            setShowFilters(!showFilters)
                                        }
                                        className="ml-2 px-3 py-2 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 rounded-md hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-500 dark:focus:ring-offset-gray-800 transition-colors"
                                    >
                                        <Filter className="h-5 w-5" />
                                    </button>
                                    {(searchParams.status !== "all" ||
                                        searchParams.search ||
                                        searchParams.sort_field !==
                                            "updated_at" ||
                                        searchParams.sort_direction !==
                                            "desc") && (
                                        <button
                                            type="button"
                                            onClick={resetFilters}
                                            className="ml-2 px-3 py-2 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 rounded-md hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            title="Reset filters"
                                        >
                                            <RefreshCw className="h-5 w-5" />
                                        </button>
                                    )}
                                </form>
                            </div>

                            <Transition
                                show={showFilters}
                                enter="transition ease-out duration-100"
                                enterFrom="transform opacity-0 scale-95"
                                enterTo="transform opacity-100 scale-100"
                                leave="transition ease-in duration-75"
                                leaveFrom="transform opacity-100 scale-100"
                                leaveTo="transform opacity-0 scale-95"
                            >
                                <div className="mb-6 p-4 bg-gray-50 dark:bg-gray-700/50 rounded-md">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label
                                                htmlFor="status"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                            >
                                                Test Status
                                            </label>
                                            <select
                                                id="status"
                                                className="block w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm transition-colors"
                                                value={searchParams.status}
                                                onChange={(e) => {
                                                    setSearchParams({
                                                        ...searchParams,
                                                        status: e.target.value,
                                                    });
                                                }}
                                            >
                                                <option value="all">
                                                    All Statuses
                                                </option>
                                                <option value="passed">
                                                    Passed
                                                </option>
                                                <option value="failed">
                                                    Failed
                                                </option>
                                            </select>
                                        </div>
                                    </div>
                                    <div className="mt-4 flex justify-end">
                                        <button
                                            type="button"
                                            onClick={applyFilters}
                                            className="px-4 py-2 bg-blue-600 dark:bg-blue-500 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            Apply Filters
                                        </button>
                                    </div>
                                </div>
                            </Transition>

                            {writers.data.length > 0 ? (
                                <>
                                    <div className="overflow-x-auto">
                                        <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                            <thead className="bg-gray-50 dark:bg-gray-800">
                                                <tr>
                                                    <th
                                                        scope="col"
                                                        className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider cursor-pointer"
                                                        onClick={() =>
                                                            toggleSort("name")
                                                        }
                                                    >
                                                        <div className="flex items-center">
                                                            Writer
                                                            {getSortIcon(
                                                                "name"
                                                            )}
                                                        </div>
                                                    </th>
                                                    <th
                                                        scope="col"
                                                        className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider cursor-pointer"
                                                        onClick={() =>
                                                            toggleSort(
                                                                "test_status"
                                                            )
                                                        }
                                                    >
                                                        <div className="flex items-center">
                                                            Status
                                                            {getSortIcon(
                                                                "test_status"
                                                            )}
                                                        </div>
                                                    </th>
                                                    <th
                                                        scope="col"
                                                        className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider cursor-pointer"
                                                        onClick={() =>
                                                            toggleSort("score")
                                                        }
                                                    >
                                                        <div className="flex items-center">
                                                            Score
                                                            {getSortIcon(
                                                                "score"
                                                            )}
                                                        </div>
                                                    </th>
                                                    <th
                                                        scope="col"
                                                        className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider cursor-pointer"
                                                        onClick={() =>
                                                            toggleSort(
                                                                "completed_at"
                                                            )
                                                        }
                                                    >
                                                        <div className="flex items-center">
                                                            Completed
                                                            {getSortIcon(
                                                                "completed_at"
                                                            )}
                                                        </div>
                                                    </th>
                                                    <th
                                                        scope="col"
                                                        className="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider"
                                                    >
                                                        Actions
                                                    </th>
                                                </tr>
                                            </thead>
                                            <tbody className="bg-white dark:bg-gray-900 divide-y divide-gray-200 dark:divide-gray-800">
                                                {writers.data.map((writer) => (
                                                    <tr
                                                        key={writer.id}
                                                        className="hover:bg-gray-50 dark:hover:bg-gray-800/60"
                                                    >
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="flex items-center">
                                                                <div className="ml-0">
                                                                    <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                        {
                                                                            writer.name
                                                                        }
                                                                    </div>
                                                                    <div className="text-sm text-gray-500 dark:text-gray-400">
                                                                        {
                                                                            writer.email
                                                                        }
                                                                    </div>
                                                                </div>
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="flex items-center">
                                                                {getTestStatusIcon(
                                                                    writer.test_status,
                                                                    writer.score
                                                                )}
                                                                <span className="ml-2">
                                                                    {getStatusBadge(
                                                                        writer.test_status,
                                                                        writer.score
                                                                    )}
                                                                </span>
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                {writer.score ? (
                                                                    <span
                                                                        className={
                                                                            writer.score >=
                                                                            90
                                                                                ? "text-green-600 dark:text-green-400 font-medium"
                                                                                : "text-red-600 dark:text-red-400 font-medium"
                                                                        }
                                                                    >
                                                                        {Math.round(
                                                                            writer.score
                                                                        )}
                                                                        %
                                                                    </span>
                                                                ) : (
                                                                    "—"
                                                                )}
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                            {formatDate(
                                                                writer.completed_at
                                                            )}
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                                            <Link
                                                                href={route(
                                                                    "admin.writers.tests.show",
                                                                    writer.id
                                                                )}
                                                                className="text-indigo-600 dark:text-indigo-400 hover:text-indigo-900 dark:hover:text-indigo-300 inline-flex items-center"
                                                            >
                                                                View Details
                                                                <ArrowRight className="ml-1 h-4 w-4" />
                                                            </Link>
                                                        </td>
                                                    </tr>
                                                ))}
                                            </tbody>
                                        </table>
                                    </div>

                                    <div className="mt-6">
                                        <Pagination links={writers.links} />
                                    </div>
                                </>
                            ) : (
                                <div className="text-center py-12">
                                    <svg
                                        className="mx-auto h-12 w-12 text-gray-400 dark:text-gray-600"
                                        fill="none"
                                        viewBox="0 0 24 24"
                                        stroke="currentColor"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z"
                                        />
                                    </svg>
                                    <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-gray-100">
                                        No test results found
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        No writers have completed the
                                        qualification test yet.
                                    </p>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
